package ydbmigrate

import (
	"context"

	"github.com/ydb-platform/ydb-go-genproto/protos/Ydb"
	"github.com/ydb-platform/ydb-go-sdk/v3"
	ydbTypes "github.com/ydb-platform/ydb-go-sdk/v3/table/types"

	"a.yandex-team.ru/tasklet/experimental/internal/yandex/xydb"
)

var (
	MigrationsTableName       = "migrations"
	MigrationsLeasesTableName = "migration_leases"
)

type SchemaVersion int32

func (s SchemaVersion) ToInt32() int32 {
	return int32(s)
}

const NilSchemaVersion SchemaVersion = 0

// Columns
var (
	VersionColumnName       = "version"
	DescriptionColumnName   = "description"
	MigrationTimeColumnName = "migration_time"
	MutationColumnName      = "mutation"
	LeaseColumnName         = "lease"
	LeaseTTLColumnName      = "lease_ttl"
	AppliedColumnName       = "applied"
)

var MigrationsTable = xydb.TableSchema{
	Name: MigrationsTableName,
	Columns: []xydb.YdbColumn{
		{
			Name:       VersionColumnName,
			ValueType:  ydbTypes.Optional(ydbTypes.TypeInt32),
			PrimaryKey: true,
		},
		{
			Name:       DescriptionColumnName,
			ValueType:  ydbTypes.Optional(ydbTypes.TypeString),
			PrimaryKey: false,
		},
		{
			Name:       MigrationTimeColumnName,
			ValueType:  ydbTypes.Optional(ydbTypes.TypeTimestamp),
			PrimaryKey: false,
		},
	},
	SecondaryIndexes: nil,
	Queries:          nil,
	TTLSettings:      nil,
}

var MigrationsLeasesTable = xydb.TableSchema{
	Name: MigrationsLeasesTableName,
	Columns: []xydb.YdbColumn{
		{
			Name:       VersionColumnName,
			ValueType:  ydbTypes.Optional(ydbTypes.TypeInt32),
			PrimaryKey: true,
		},
		{
			Name:       MutationColumnName,
			ValueType:  ydbTypes.Optional(ydbTypes.TypeString),
			PrimaryKey: true,
		},
		{
			Name:      LeaseColumnName,
			ValueType: ydbTypes.Optional(ydbTypes.TypeString),
		},
		{
			Name:      LeaseTTLColumnName,
			ValueType: ydbTypes.Optional(ydbTypes.TypeTimestamp),
		},
		{
			Name:      MigrationTimeColumnName,
			ValueType: ydbTypes.Optional(ydbTypes.TypeTimestamp),
		},
		{
			Name:      AppliedColumnName,
			ValueType: ydbTypes.Optional(ydbTypes.TypeBool),
		},
	},
	SecondaryIndexes: nil,
	Queries:          nil,
	TTLSettings:      nil,
}

var MetaTables = map[string]xydb.TableSchema{
	MigrationsTable.Name:       MigrationsTable,
	MigrationsLeasesTable.Name: MigrationsLeasesTable,
}

var MetaQueryConsts = struct {
	MigrationsTable       string
	MigrationsLeasesTable string

	Version       string
	Description   string
	MigrationTime string
	Lease         string
	Mutation      string
	LeaseTTL      string
	Applied       string
}{
	MigrationsTable:       MigrationsTableName,
	MigrationsLeasesTable: MigrationsLeasesTableName,

	Version:       VersionColumnName,
	Description:   DescriptionColumnName,
	MigrationTime: MigrationTimeColumnName,
	Lease:         LeaseColumnName,
	Mutation:      MutationColumnName,
	LeaseTTL:      LeaseTTLColumnName,
	Applied:       AppliedColumnName,
}

var InitSchemaQuery = xydb.NewQuery(
	"init_schema", `
	INSERT INTO {{ .MigrationsTable }}
		({{ .Version }}, {{ .Description }}, {{ .MigrationTime }})
	VALUES
		(0, "initial schema", CurrentUtcTimestamp());
`,
)

func CreateMetaTables(ctx context.Context, cli *xydb.Client) error {
	if err := cli.CreateTablesBySchema(ctx, MetaTables); err != nil {
		return err
	}
	_, err := cli.Do(ctx, InitSchemaQuery.Query(), cli.WriteTxControl)
	if ydb.IsOperationError(err, Ydb.StatusIds_PRECONDITION_FAILED) {
		return nil
	} else {
		return err
	}
}

// PurgeMetaTables is for unit tests
func PurgeMetaTables(ctx context.Context, cli *xydb.Client) error {
	return cli.PurgeTablesBySchema(ctx, MetaTables)
}
