from typing import Set, Optional, Any
from pathlib import Path

from google.protobuf import descriptor_pb2
from google.protobuf.json_format import MessageToJson, Parse
from google.protobuf.descriptor_pool import DescriptorPool
from google.protobuf.descriptor import FieldDescriptor
from google.protobuf.message_factory import MessageFactory
from google.protobuf.message import Message


class ProtoSupport:
    INT_TO_TYPE = {
        8: "TYPE_BOOL",
        12: "TYPE_BYTES",
        1: "TYPE_DOUBLE",
        14: "TYPE_ENUM",
        7: "TYPE_FIXED32",
        6: "TYPE_FIXED64",
        2: "TYPE_FLOAT",
        10: "TYPE_GROUP",
        5: "TYPE_INT32",
        3: "TYPE_INT64",
        11: "TYPE_MESSAGE",
        15: "TYPE_SFIXED32",
        16: "TYPE_SFIXED64",
        17: "TYPE_SINT32",
        18: "TYPE_SINT64",
        9: "TYPE_STRING",
        13: "TYPE_UINT32",
        4: "TYPE_UINT64",
    }

    def __init__(self, fds_file: Path):
        self._pool: DescriptorPool = self._create_pool_from_path(fds_file)
        self._message_factory: MessageFactory = MessageFactory(self._pool)
        self._available_names: Set[str] = set()

    @staticmethod
    def _create_pool_from_path(fds_file: Path) -> DescriptorPool:
        pool = DescriptorPool()

        fds = descriptor_pb2.FileDescriptorSet()
        fds.MergeFromString(fds_file.read_bytes())

        file_by_name = {file_proto.name: file_proto for file_proto in fds.file}

        def _add_file(file_proto):
            for dependency in file_proto.dependency:
                if dependency in file_by_name:
                    _add_file(file_by_name.pop(dependency))
            pool.Add(file_proto)

        while file_by_name:
            _add_file(file_by_name.popitem()[1])

        return pool

    def get_message_object_by_name(self, name: str) -> Message:
        message_descriptor = self._pool.FindMessageTypeByName(name)

        prototype = self._message_factory.GetPrototype(message_descriptor)
        exts = self._pool.FindAllExtensions(message_descriptor)
        for ext in exts:
            prototype.RegisterExtension(ext)
        return prototype()

    def convert_protobuf_bytes_to_json(self, name: str, message_bytes: bytes) -> str:
        prototype = self.get_message_object_by_name(name)
        prototype.ParseFromString(message_bytes)
        return MessageToJson(prototype, descriptor_pool=self._pool)

    def convert_json_to_protobuf_bytes(self, name: str, json_str: str) -> bytes:
        prototoype = self.get_message_object_by_name(name)
        prototoype = Parse(json_str, prototoype)
        return prototoype.SerializeToString()

    def get_option_for_field(self, name: str, field_name: str, option) -> Optional[Any]:
        filed_descriptor: FieldDescriptor = self._pool.FindFieldByName(f"{name}.{field_name}")
        if option in filed_descriptor.GetOptions().Extensions:
            return filed_descriptor.GetOptions().Extensions[option]
        else:
            return None
