from pathlib import Path
from json import loads as json_loads

import os

from tasklet.experimental.registry_test.app.lib.protobuf_support import ProtoSupport

from tasklet.experimental.registry_test.primitive_containers.string_container_pb2 import StringContainer
from tasklet.experimental.registry_test.primitive_containers.simple_enum_pb2 import (
    SimpleEnumContainer,
    ENUM_OPTION_2,
)
from tasklet.experimental.registry_test.containers_with_options.container_with_options_pb2 import (
    custom_option_1,
    custom_option_2,
)


class TestProtoSupport:
    REGISTRY_FILE = Path(os.environ["ARCADIA_ROOT"]) / "tasklet/experimental/registry_test/registry/experimental-registry_test-registry.protodesc"
    PS = ProtoSupport(REGISTRY_FILE)

    STRING_CONTAINER_ID = "tasklet.experimental.registry_test.primitive_containers.StringContainer"
    ENUM_CONTAINER_ID = "tasklet.experimental.registry_test.primitive_containers.SimpleEnumContainer"
    BASE_MESSAGE_ID = "tasklet.experimental.registry_test.extended_messages.BaseMessage"
    CONTAINER_WITH_OPTIONS_ID = "tasklet.experimental.registry_test.containers_with_options.ContainerWithOptions"

    def test_find_by_name_in_registry(self):
        empty_message: StringContainer = self.PS.get_message_object_by_name(self.STRING_CONTAINER_ID)
        assert empty_message.string_value == ""

    def test_proto_to_json_conversion(self):
        str_container = StringContainer()
        str_container.string_value = "АБВГД"

        bytes = str_container.SerializeToString()

        json_str = self.PS.convert_protobuf_bytes_to_json(self.STRING_CONTAINER_ID, bytes)
        json_dict = json_loads(json_str)

        assert json_dict["stringValue"] == "АБВГД"

    def test_json_to_proto_conversion(self):
        json_str = "{\"stringValue\":\"АБВГД\"}"

        proto_bytes = self.PS.convert_json_to_protobuf_bytes(self.STRING_CONTAINER_ID, json_str)

        str_container = StringContainer()
        str_container.ParseFromString(proto_bytes)

        assert str_container.string_value == "АБВГД"

    def test_proto_to_json_enum_conversion(self):
        enum_container = SimpleEnumContainer()
        enum_container.enum_value = ENUM_OPTION_2

        bytes = enum_container.SerializeToString()

        json_str = self.PS.convert_protobuf_bytes_to_json(self.ENUM_CONTAINER_ID, bytes)
        json_dict = json_loads(json_str)

        assert json_dict["enumValue"] == "ENUM_OPTION_2"

    def test_json_to_proto_enum_conversion(self):
        json_str = "{\"enumValue\":\"ENUM_OPTION_2\"}"

        proto_bytes = self.PS.convert_json_to_protobuf_bytes(self.ENUM_CONTAINER_ID, json_str)

        enum_container = SimpleEnumContainer()
        enum_container.ParseFromString(proto_bytes)

        assert enum_container.enum_value == ENUM_OPTION_2

    def test_proto_to_json_extension_conversion(self):
        """
        Extensions are supported by python protobuf realization according to unit tests:
        https://github.com/protocolbuffers/protobuf/blob/01e84b129361913e5613464c857734fcfe095367/python/google/protobuf/internal/json_format_test.py#L171-L182

        However there is some issues with extensions when proto objects are created via MessageFactory. Further work
        require additional research and we decided to not support extension at this moment.
        """
        # message = BaseMessage()
        # message.base_value = "abc"
        # message.Extensions[derived_message_1].additional_value = "def"
        #
        # bytes = message.SerializeToString()
        #
        # json_str = self.PS.convert_protobuf_bytes_to_json(self.BASE_MESSAGE_ID, bytes)
        # json_dict = json_loads(json_str)
        #
        # assert json_dict["baseValue"] == "abc"
        # assert json_dict["derivedMessage1"]["additionalValue"] == "abc"
        pass

    def test_json_to_proto_extension_conversion(self):
        """
        see test_proto_to_json_extension_conversion doc
        """
        # json_str = "{\"baseValue\":\"abc\",\"derivedMessage1\":{\"additionalValue\":\"def\"}}"
        #
        # proto_bytes = self.PS.convert_json_to_protobuf_bytes(self.BASE_MESSAGE_ID, json_str)
        #
        # message = BaseMessage()
        # message.ParseFromString(proto_bytes)
        #
        # assert message.base_value == "abc"
        # assert message.Extensions[derived_message_1].additional_value == "def"
        pass

    def test_option_extraction(self):
        co_1 = self.PS.get_option_for_field(
            self.CONTAINER_WITH_OPTIONS_ID,
            "int_value",
            custom_option_1
        )
        co_2 = self.PS.get_option_for_field(
            self.CONTAINER_WITH_OPTIONS_ID,
            "int_value",
            custom_option_2
        )
        missing_option = self.PS.get_option_for_field(
            self.CONTAINER_WITH_OPTIONS_ID,
            "optionless_value",
            custom_option_1
        )

        assert co_1 == 1
        assert co_2 == "abc"
        assert missing_option is None
