#!/usr/bin/env python3
import base64

import click
import google.protobuf.pyext._message
from google.protobuf import descriptor_pb2
from google.protobuf import json_format
from google.protobuf import descriptor_pool
from google.protobuf import message_factory

from google.protobuf import timestamp_pb2

HOWTO = """
Собираем облачные fds в cloud_api.fds

```(shell)
cd ~/arc/cloud/bitbucket && \
find . -name '*.proto' -type f  | \
    grep -v third_party | \
    xargs protoc \
        --descriptor_set_out=cloud_api.bin  \
        -I private-api/third_party/googleapis/ \
        -I private-api/ \
        -I public-api/ \
        -I common-api/  \
         --include_imports --experimental_allow_proto3_optional
```

Зависимости файлов (прямые)
===

```(py)
$ ./dump_fds.py  --fds ~/cloud_api.fds  list-files  | head
* google/api/annotations.proto
  - google/api/http.proto
  - google/protobuf/descriptor.proto
* google/api/http.proto
* google/protobuf/any.proto
```

Просмотр сообщения:

```
$ ./dump_fds.py  --fds ~/cloud_api.fds  get-messages yandex.cloud.operation.Operation
message: Operation  file=yandex/cloud/operation/operation.proto
  field: id, id: 0, name: yandex.cloud.operation.Operation.id, type: TYPE_STRING
  field: description, id: 1, name: yandex.cloud.operation.Operation.description, type: TYPE_STRING
  field: created_at, id: 2, name: yandex.cloud.operation.Operation.created_at, type: TYPE_MESSAGE
  field: created_by, id: 3, name: yandex.cloud.operation.Operation.created_by, type: TYPE_STRING
  field: modified_at, id: 4, name: yandex.cloud.operation.Operation.modified_at, type: TYPE_MESSAGE
  field: done, id: 5, name: yandex.cloud.operation.Operation.done, type: TYPE_BOOL
  field: metadata, id: 6, name: yandex.cloud.operation.Operation.metadata, type: TYPE_MESSAGE
  field: error, id: 7, name: yandex.cloud.operation.Operation.error, type: TYPE_MESSAGE
  field: response, id: 8, name: yandex.cloud.operation.Operation.response, type: TYPE_MESSAGE
```


Конвертация сериализованного протобафа в json

```
$ ./dump_fds.py  --fds ~/cloud_api.fds  pb-to-json yandex.cloud.operation.Operation
```

```(json)
{
  "createdAt": "2021-12-15T13:38:14.421181Z",
  "createdBy": "alximik",
  "error": {
    "code": 1543,
    "message": "full oblom error"
  },
  "id": "op_id",
  "metadata": {},
  "modifiedAt": "2021-12-15T13:38:49.860391Z"
}
```

"""

_ = timestamp_pb2

_pool = descriptor_pool.DescriptorPool()
_fds = descriptor_pb2.FileDescriptorSet()

INT_TO_TYPE = {
    8: "TYPE_BOOL",
    12: "TYPE_BYTES",
    1: "TYPE_DOUBLE",
    14: "TYPE_ENUM",
    7: "TYPE_FIXED32",
    6: "TYPE_FIXED64",
    2: "TYPE_FLOAT",
    10: "TYPE_GROUP",
    5: "TYPE_INT32",
    3: "TYPE_INT64",
    11: "TYPE_MESSAGE",
    15: "TYPE_SFIXED32",
    16: "TYPE_SFIXED64",
    17: "TYPE_SINT32",
    18: "TYPE_SINT64",
    9: "TYPE_STRING",
    13: "TYPE_UINT32",
    4: "TYPE_UINT64",
}


def GetPool() -> descriptor_pool.DescriptorPool:
    return _pool


def GetFds() -> descriptor_pb2.FileDescriptorSet:
    return _fds


@click.group()
@click.option("--fds", type=click.File(mode="rb"))
def main(fds):
    p = GetPool()
    descriptor = GetFds()
    descriptor.MergeFromString(fds.read())
    file_by_name = {file_proto.name: file_proto for file_proto in descriptor.file}

    def _add_file(file_proto):
        for dependency in file_proto.dependency:
            if dependency in file_by_name:
                _add_file(file_by_name.pop(dependency))
        p.Add(file_proto)

    while file_by_name:
        _add_file(file_by_name.popitem()[1])


@main.command()
def list_files():
    descriptor = GetFds()
    files = {x.name: x.dependency for x in descriptor.file}
    for k, v in sorted(files.items()):
        click.echo(f"* {k}")
        for i in sorted(v):
            click.echo(f"  - {i}")


@main.command()
@click.argument("messages", nargs=-1, metavar="some.proto.package.MessageName ...")
def get_messages(messages):
    pool = GetPool()
    for name in messages:
        m: google.protobuf.pyext._message.Descriptor = pool.FindMessageTypeByName(name)
        print(f"message: {m.name}  file={m.file.name}")
        # google.protobuf.pyext._message.
        for ext in m.extensions:
            print(f"  ext: {type(ext)}")
        for field in m.fields:  # type: google.protobuf.pyext._message.FieldDescriptor
            print(f"  field: {field.name}, id: {field.index}, name: {field.full_name}, type: {INT_TO_TYPE[field.type]}")


@main.command()
@click.argument("message", type=click.STRING)
# @click.argument("inf", type=click.File("rb"))
def pb_to_json(message):  # , inf):
    assert message == "yandex.cloud.operation.Operation"
    # message_body = inf.read()

    # serialized message
    # Out[36]:
    # id: "op_id"
    # created_at {
    #   seconds: 1639575494
    #   nanos: 421181000
    # }
    # created_by: "alximik"
    # modified_at {
    #   seconds: 1639575529
    #   nanos: 860391000
    # }
    # error {
    #   code: 1543
    #   message: "full oblom error"
    # }

    message_body = base64.decodebytes(
        b'CgVvcF9pZBoMCMbf540GEMjs6sgBIgdhbHhpbWlrKgwI6d/njQYQ2IyimgM6AEIVCIcMEhBmdWxs\nIG9ibG9tIGVycm9y\n'
    )
    pool = GetPool()
    mf = message_factory.MessageFactory(pool)

    message_descriptor = pool.FindMessageTypeByName(message)
    prototype = mf.GetPrototype(message_descriptor)
    obj = prototype()
    obj.ParseFromString(message_body)
    click.echo(json_format.MessageToJson(obj, sort_keys=True, indent=2))


if __name__ == '__main__':
    main()
