#include "generator.h"

#include <tasklet/api/tasklet.pb.h>
#include <tasklet/gen/lib/names.h>
#include <tasklet/gen/lib/components.h>

#include <google/protobuf/compiler/cpp/cpp_helpers.h>
#include <google/protobuf/io/zero_copy_stream.h>

#include <util/string/join.h>
#include <util/string/split.h>

namespace NTaskletGen {

using namespace google::protobuf;
using namespace google::protobuf::compiler;

namespace {

    TProtoStringType ClassRepr(const Descriptor* descriptor) {
        TVector<TStringBuf> parts = StringSplitter(descriptor->file()->package()).Split('.');
        parts.push_back(descriptor->name());
        return JoinSeq("::", parts);
    }

} // anonymous namespace

TCppGenerator::TCppGenerator() = default;
TCppGenerator::~TCppGenerator() = default;

bool TCppGenerator::Generate(
    const FileDescriptor* file,
    const TProtoStringType& parameter,
    GeneratorContext* generator_context,
    TProtoStringType* error
) const {
    Y_UNUSED(parameter, error);

    TProtoStringType taskletHeader = TaskletHeader(file);
    THolder<io::ZeroCopyOutputStream> stream(generator_context->Open(taskletHeader));
    io::Printer printer(stream.Get(), '$');

    GenerateIncludes(printer, file);
    printer.Print("namespace NTasklet {\n\n");
    for (int i = 0; i < file->message_type_count(); ++i) {
        const Descriptor* descriptor = file->message_type(i);
        if (TTaskletComponents::IsTasklet(descriptor)) {
            GenerateClass(printer, descriptor);
        }
    }
    printer.Print("} // namespace NTasklet\n");

    return true;
}

void TCppGenerator::GenerateIncludes(io::Printer& printer, const FileDescriptor* file) const {
    Y_UNUSED(file);
    printer.Print("#pragma once\n");
    printer.Print("#include <tasklet/runtime/cpp/tasklet.h>\n");
    printer.Print("#include <$pb2$>\n\n", "pb2", ProtobufHeader(file));
}

void TCppGenerator::GenerateClass(io::Printer& printer, const Descriptor* descriptor) const {
    TProtoStringType className = BaseTaskletClassName(descriptor);
    TTaskletComponents tasklet(descriptor);

    printer.Print(
        "class $class$ : public NTasklet::ITasklet<$tasklet$,$input$,$output$> {\n",
        "class", className, "tasklet", ClassRepr(tasklet.D),
        "input", ClassRepr(tasklet.InputField->message_type()), "output", ClassRepr(tasklet.OutputField->message_type())
    );
    printer.Print("}; // $class$\n\n", "class", className);
}

} // namespace NTaskletGen
