import copy
import logging

from tasklet.api import tasklet_pb2


logger = logging.getLogger(__name__)

SERVICE_SUFFIX = 'Service'
_SCHEDULER_SERVICE_NAME = 'Scheduler'

_ALWAYS_CONSTRUCT_SERVICES = [
    'Executor',
    'Scheduler',
    'Memory',
    'Spy',
]

_SERVICES_DEFAULT_INFO = {
    'Executor': {
        'field_name': 'executor',
        'service_path': 'tasklet.api.executor',
        'impl_path': 'tasklet.domain.local:LocalExecutor',
    },
    _SCHEDULER_SERVICE_NAME: {
        'field_name': 'sched',
        'service_path': 'tasklet.api.sched',
        'impl_path': 'tasklet.domain.local:LocalScheduler',
    },
    'Memory': {
        'field_name': 'mem',
        'service_path': 'tasklet.api.mem',
        'impl_path': 'tasklet.domain.local:LocalMemory',
    },
    'Spy': {
        'field_name': 'spy',
        'service_path': 'tasklet.api.spy',
        'impl_path': 'tasklet.domain.local:LocalSpy',
    },
    'Yav': {
        'field_name': 'yav',
        'service_path': 'tasklet.services.yav.proto.yav',
        'impl_path': 'tasklet.services.yav:YavService',
    },
    'Ci': {
        'field_name': 'Ci',
        'service_path': 'tasklet.services.ci.proto.ci',
        'impl_path': 'tasklet.services.ci:CiService',
    }
}

DOMAIN_2_IMPL = {
    tasklet_pb2.Domain.NOT_SPECIFIED: None,
    tasklet_pb2.Domain.SANDBOX: {
        _SCHEDULER_SERVICE_NAME: {
            'impl_path': 'tasklet.domain.sandbox:SandboxTaskletScheduler',
        }
    },
    tasklet_pb2.Domain.LOCAL: {},
    tasklet_pb2.Domain.YT: {
        _SCHEDULER_SERVICE_NAME: {
            'impl_path': 'tasklet.domain.yt:YtScheduler',
        }
    },
    tasklet_pb2.Domain.TEST: {
        'Ci': {
            'impl_path': 'tasklet.services.ci:CiServiceLogging',
        },
        'Yav': {
            'impl_path': 'tasklet.services.yav:TestYavService',
        }
    },
    # Deprecated
    'SANDBOX_GLYCINE': {
        _SCHEDULER_SERVICE_NAME: {
            'impl_path': 'tasklet.domain.sandbox:SandboxScheduler',
        },
        'Memory': {
            'impl_path': 'tasklet.domain.sandbox:SandboxInsideMemory',
        }
    },

}


def _create_service_descr(service_name, service_info):
    entry = tasklet_pb2.ContextEntry()
    entry.name = service_info['field_name']
    entry.service_name_ctx = service_name + 'Service'

    s = tasklet_pb2.LocalService()
    s.register = '{}_pb2_grpc:add_{}Servicer_to_server'.format(service_info['service_path'], service_name)
    s.impl = service_info['impl_path']
    s.client = '{}_pb2_grpc:{}Stub'.format(service_info['service_path'], service_name)
    entry.any.Pack(s)

    return entry


def _get_services_info_by_domain(domain):
    services_info = copy.deepcopy(_SERVICES_DEFAULT_INFO)
    for service_name, service_info in DOMAIN_2_IMPL[domain].items():
        services_info[service_name].update(service_info)

    return services_info


def setup(domain, tasklet_descriptor):
    """

    :param domain: DomainImpl item
    """

    ctx_msg = tasklet_pb2.Context()

    if domain not in DOMAIN_2_IMPL:
        raise ValueError("Domain '{}' is not supported".format(domain))

    services_info = _get_services_info_by_domain(domain)
    services_to_cons = {}

    for service_name in _ALWAYS_CONSTRUCT_SERVICES:
        services_to_cons[_SERVICES_DEFAULT_INFO[service_name]['field_name']] = copy.deepcopy(services_info[service_name])
        services_to_cons[_SERVICES_DEFAULT_INFO[service_name]['field_name']]['name'] = service_name

    for service_field_name, service_description in tasklet_descriptor.fields_by_name.items():
        if hasattr(service_description.message_type, 'name') and service_description.message_type.name.endswith(SERVICE_SUFFIX):
            name = service_description.message_type.name[:-len(SERVICE_SUFFIX)]

            field_domain = service_description.GetOptions().Extensions[tasklet_pb2.domain]
            if field_domain == tasklet_pb2.Domain.NOT_SPECIFIED:
                field_domain = domain

            service_data = _get_services_info_by_domain(field_domain)[name]
            service_data['field_name'] = service_field_name
            service_data['name'] = name

            services_to_cons[service_field_name] = service_data

    for service, service_descr in services_to_cons.items():
        ctx_msg.entries.add().CopyFrom(_create_service_descr(service_descr['name'], service_descr))

    return ctx_msg
