#include "api.h"
#include "runtime.h"
#include <serp/ynode/napi_util/catch.h>
#include <serp/ynode/napi_util/arguments.h>
#include <serp/ynode/napi_util/from_js.h>
#include <serp/ynode/napi_util/to_js.h>
#include <contrib/libs/node-addon-api/napi.h>
#include <contrib/libs/nodejs_12/src/node.h>
#include <contrib/libs/nodejs_12/src/node_platform.h>
#include <contrib/libs/nodejs_12/src/env.h>
#include <contrib/libs/nodejs_12/src/inspector/worker_inspector.h>
#include <contrib/libs/nodejs_12/src/tracing/agent.h>
#include <contrib/libs/nodejs_12/src/tracing/trace_event.h>
#include <contrib/libs/nodejs_12/deps/v8/include/v8.h>
#include <contrib/libs/libuv/include/uv.h>
#include <library/cpp/threading/future/future.h>
#include <library/cpp/resource/resource.h>
#include <util/generic/yexception.h>
#include <util/generic/ptr.h>
#include <util/thread/factory.h>
#include <util/string/builder.h>
#include <string>
#include <vector>
#include <cstdlib>

using namespace NTasklet::NJS;

using NNodejs::CatchingExport;
using NNodejs::ToJs;
using NNodejs::ToJsTrait;
using NNodejs::ArgSpec;
using node::ArrayBufferAllocator;
using node::Environment;
using node::IsolateData;
using node::MultiIsolatePlatform;
using v8::Context;
using v8::HandleScope;
using v8::Isolate;
using v8::Local;
using v8::Locker;
using v8::MaybeLocal;
using v8::SealHandleScope;
using v8::V8;
using v8::Value;

namespace node {
extern int InitializeNodeWithArgs(std::vector<std::string>* argv, std::vector<std::string>* exec_argv, std::vector<std::string>* errors);
}

class TRunNodeException : public yexception {};

void RunNodeInstance(const TAtomicSharedPtr<MultiIsolatePlatform>& platform, std::vector<char*> cargs) {
    uv_loop_t loop;
    {
        int returnCode = uv_loop_init(&loop);
        Y_ENSURE_EX(returnCode == 0, TRunNodeException() << "UV loop initialization failed with code " << returnCode);
    }

    std::shared_ptr<ArrayBufferAllocator> allocator = ArrayBufferAllocator::Create();
    Isolate* isolate = node::NewIsolate(allocator.get(), &loop, platform.Get());
    Y_ENSURE_EX(isolate != nullptr, TRunNodeException() << "Failed to create V8 isolate");

    int exitCode = 0;
    {
        Locker locker(isolate);
        Isolate::Scope isolateScope(isolate);

        std::unique_ptr<IsolateData, decltype(&node::FreeIsolateData)> isolate_data(
                node::CreateIsolateData(isolate, &loop, platform.Get(), allocator.get()),
                node::FreeIsolateData);

        HandleScope handleScope(isolate);
        Local<Context> context = node::NewContext(isolate);
        Y_ENSURE_EX(!context.IsEmpty(), TRunNodeException() << "Failed to create V8 context");

        Context::Scope contextScope(context);
        std::unique_ptr<Environment, decltype(&node::FreeEnvironment)> env(
                node::CreateEnvironment(
                        isolate_data.get(),
                        context,
                        cargs.size(),
                        &cargs[0],
                        0,
                        nullptr),
                node::FreeEnvironment);
        env->InitializeDiagnostics();
        env->InitializeInspector({});
        node::LoadEnvironment(env.get());

        {
            SealHandleScope seal(isolate);
            bool continueLoop;
            do {
                uv_run(&loop, UV_RUN_DEFAULT);

                platform->DrainTasks(isolate);
                continueLoop = uv_loop_alive(&loop);
                if (continueLoop) {
                    continue;
                }
                node::EmitBeforeExit(env.get());
                continueLoop = uv_loop_alive(&loop);
            } while (continueLoop);
        }

        exitCode = node::EmitExit(env.get());
        node::Stop(env.get());
    }

    bool platform_finished = false;
    platform->AddIsolateFinishedCallback(
            isolate, [](void* data) {
                *static_cast<bool*>(data) = true;
            },
            &platform_finished);
    platform->UnregisterIsolate(isolate);
    isolate->Dispose();
    // Wait until the platform has cleaned up all relevant resources.
    while (!platform_finished) {
        uv_run(&loop, UV_RUN_ONCE);
    }
    int err = uv_loop_close(&loop);
    Y_ENSURE_EX(err == 0, TRunNodeException() << "Failed to close event-loop with error code " << err);

    Y_ENSURE_EX(exitCode == 0, TRunNodeException() << "Node.js terminated with non-zero exit code");
}

void StartNodejs() {
    std::vector<std::string> args {"node", "-e", NResource::Find("bootstrap.j$")};
    std::vector<char*> cargs;
    cargs.reserve(args.size());
    for(auto& arg : args) {
        cargs.push_back(&arg[0]);
    }

    {
        std::vector<std::string> execArgs;
        std::vector<std::string> errors;
        int retCode = node::InitializeNodeWithArgs(&args, &execArgs, &errors);
        if (retCode != 0) {
            TStringBuilder sb;
            for (const auto& err: errors) {
                sb << "\n" << err;
            }
            ythrow yexception() << "Failed to initialize Node.js instance: " << sb;
        }
    }

    node::tracing::Agent tracingAgent;
    node::tracing::TraceEventHelper::SetAgent(&tracingAgent);
    TAtomicSharedPtr<MultiIsolatePlatform> platform(node::CreatePlatform(4, tracingAgent.GetTracingController()));
    V8::InitializePlatform(platform.Get());
    V8::Initialize();

    TMaybe<TRunNodeException> nodeInstanceException;
    try {
        RunNodeInstance(platform, cargs);
    } catch(const TRunNodeException& e) {
        nodeInstanceException = MakeMaybe(e);
    }

    V8::Dispose();
    V8::ShutdownPlatform();

    ((node::NodePlatform*)(platform.Get()))->Shutdown();

    if (nodeInstanceException.Defined()) {
        ythrow nodeInstanceException.GetRef();
    }
}

void StopRuntime() {
    TRuntime::GetInstance()->Stop();
}

TRuntime::TRuntime() {
    JavaScriptInitialization_ = NThreading::NewPromise<void>();
    NodeMainThread_ = SystemThreadFactory()->Run(StartNodejs);
    atexit(&StopRuntime);
}

void TRuntime::Stop() {
    with_lock(StopLock_) {
        if (NodeMainThread_ != nullptr) {
            JavaScriptHandler_.Release();
            NodeMainThread_->Join();
            NodeMainThread_.Destroy();
            Y_ENSURE(NodeMainThread_ == nullptr);
        }
    }
}

TRuntime::~TRuntime() {
    Stop();
}

TFutureResult TRuntime::CallJavaScript(const TStringBuf &module, const TStringBuf &method, const TStringBuf &data) {
    if (!JavaScriptInitialization_.HasValue()) {
        JavaScriptInitialization_.GetFuture().Wait();
    }

    TResultPromise promise = NThreading::NewPromise<TString>();
    const auto future = promise.GetFuture();

    JavaScriptHandler_.BlockingCall([promise, module, method, data] (Napi::Env env, Napi::Function jsCallback) {
        auto _promise = promise;
        Napi::Object request = Napi::Object::New(env);
        request.Set("module", ToJs(env, NNodejs::TStringBufAsString{module}));
        request.Set("method", ToJs(env, NNodejs::TStringBufAsString{method}));
        request.Set("data", ToJs(env, NNodejs::TStringBufAsCopy<char>{data}));

        const auto& resolveFn = Napi::Function::New(env, [_promise] (const Napi::CallbackInfo& info) {
            auto __promise = _promise;
            TMaybe<TString> error;
            TMaybe<NNodejs::TBufferAsString> result;
            ParseArguments(info, JS_ARG(error), JS_ARG(result));

            if (!error.Empty()) {
                __promise.SetException("JavaScript error: " + error.GetRef());
                return;
            }
            if (result.Empty()) {
                __promise.SetException("JavaScript must call back with at least 2 arguments if first one is null or undefined");
                return;
            }
            __promise.TrySetValue(result.GetRef().string);
        }, "callback");

        jsCallback.Call({request, resolveFn});
    });

    return future;
}

TFutureResult NTasklet::NJS::CallJavaScript(const TStringBuf& module, const TStringBuf& method, const TStringBuf& data) {
    return TRuntime::GetInstance()->CallJavaScript(module, method, data);
}

void TRuntime::SetJavaScriptHandler(const Napi::ThreadSafeFunction& handler) {
    JavaScriptHandler_ = handler;
    JavaScriptInitialization_.TrySetValue();
}

Napi::String GetResourceFile(const Napi::CallbackInfo& info) {
    TString key;
    ParseArguments(info, JS_ARG(key));
    const auto& content = NResource::Find(key);
    return ToJs(info.Env(), content);
}

void SetJavaScriptCallback(const Napi::CallbackInfo& info) {
    const Napi::Env env = info.Env();
    Napi::Function jsHandler;
    ParseArguments(info, JS_ARG(jsHandler));

    const auto tsfn = Napi::ThreadSafeFunction::New(env, jsHandler, "TaskletCallback", 0, 1);
    TRuntime::GetInstance()->SetJavaScriptHandler(tsfn);
}

Napi::Object TaskletRuntimeModuleInit(Napi::Env env, Napi::Object exports) {
    CatchingExport(exports, "getResourceFile", GetResourceFile);
    CatchingExport(exports, "setTaskletCallback", SetJavaScriptCallback);
    return exports;
}

NODE_API_MODULE(tasklet_runtime, TaskletRuntimeModuleInit)
