#include "dispatch.h"

#include "py_wrapper.h"
#include "registry.h"
#include "wrapper.h"

#include <tasklet/api/tasklet.pb.h>

#include <util/generic/yexception.h>
#include <util/generic/vector.h>
#include <util/string/join.h>

namespace NTasklet {

    namespace {

    IWrapper* Dispatch(const TString &taskletName, bool throwOnUnknown) {
        IWrapper* w = Registry()->Find(taskletName);
        if (!w && throwOnUnknown) {
            ythrow yexception() << "Cannot find tasklet '" << taskletName << "'";
        }
        return w;
    }

    } // anonymous namespace

    void Initialize() {
        Registry()->Initialize();
    }

    TString ImplClassPath(const TString& taskletName) {
        IWrapper* w = Dispatch(taskletName, false);
        if (w == nullptr) {
            return "";
        }
        return w->ImplClassPath();
    }

    TString TaskletToImplName(const TString& taskletName) {
        return JoinSeq(";", Registry()->FindImpl(taskletName));
    }

    TString Execute(const TString& requestData) {
        tasklet::JobInstance request;
        if (!request.ParseFromString(requestData)) {
           ythrow yexception() << "Invalid request";
        }

        IWrapper* w = Dispatch(request.statement().name(), true);

        TMaybe<TString> responseData = w->Execute(requestData);
        if (responseData.Empty()) {
            ythrow yexception() << "Request execution failed: "
                << "tasklet=" << request.statement().name() << ", id=" << request.id();
        }
        return responseData.GetRef();
    }

    TString GetInitDescription(const TString& requestData) {
        tasklet::JobStatement request;
        if (!request.ParseFromString(requestData)) {
           ythrow yexception() << "Invalid request";
        }

        IWrapper* w = Dispatch(request.name(), true);

        TMaybe<TString> responseData = w->GetInitDescription(requestData);

        if (responseData.Empty()) {
            ythrow yexception() << "Init description evaluation failed: " << "tasklet=" << request.name();
        }

        return responseData.GetRef();
    }

    TString ListTaskletNames() {
        return JoinSeq(";", Registry()->ListTaskletNames());
    }

} // NTasklet namespace
