import logging
import pickle
import six

import tblib

from google.protobuf import empty_pb2

from library.python import func

from tasklet.api import tasklet_pb2

from tasklet.runtime import utils


class NonImportableException(Exception):
    pass


def exception_tuple(job_result):
    try:
        exc_class = utils.import_symbol(job_result.python_error.et) or NonImportableException
        exc_value = pickle.loads(job_result.python_error.ev)
    except Exception:
        logging.exception("Failed to recover base exception")
        exc_class = NonImportableException
        exc_value = None

    return (
        exc_class,
        exc_value,
        tblib.Traceback.from_string(job_result.python_error.tb, strict=False).as_traceback(),
    )


def reraise(job_result):
    six.reraise(*exception_tuple(job_result))


class TaskletHolder(object):

    name = None
    Context = empty_pb2.Empty
    Input = None
    Output = None
    Requirements = None

    _sbtask_import_path = None

    @func.classproperty
    def sbtask_cls(cls):
        """
        Sandbox class that executes tasklet
        :rtype: sandbox.sdk2.Task
        """
        return utils.import_symbol(cls._sbtask_import_path)

    def __init__(self, ctx_msg, inp):
        self._ctx_msg = ctx_msg
        self._input = inp

    @property
    def statement(self):
        job_desc = tasklet_pb2.JobStatement()
        job_desc.input.Pack(self._input)
        job_desc.name = self.name
        job_desc.ctx.CopyFrom(self._ctx_msg)

        return job_desc


class TaskletBase(object):

    __holder_cls__ = None

    def __init__(self, model):
        self.tasklet_id = ""
        if isinstance(model, tasklet_pb2.JobInstance):
            self.tasklet_id = model.id
            model = model.statement
        assert isinstance(model, tasklet_pb2.JobStatement), type(model)
        self.__model = model

    def wait(self, description, tasklet, scheduler=None):
        """
        Run tasklet and wait for it's completion.

        :param str description: Textual description of running tasklet
        :param TaskletHolder tasklet:
        :return: Tasklet's output
        :rtype TaskletHolder.Output
        """

        job = tasklet_pb2.JobInstance()
        job.statement.CopyFrom(tasklet.statement)
        job.parent_id = self.tasklet_id
        job.description = description

        if scheduler is None:
            scheduler = self.ctx.sched

        instance_id = scheduler.instance(job)
        self.ctx.mem.save(description, instance_id)
        scheduler.schedule(instance_id)

        scheduler.wait_for(instance_id)
        res = list(scheduler.status(instance_id))[0]

        if not res.result.success:
            if res.result.is_python_error:
                reraise(res.result)
            else:
                raise Exception("Tasklet failed with error: {}".format(res.result.error))

        output = tasklet.Output()
        res.result.output.Unpack(output)

        return output

    def run(self):
        raise NotImplementedError

    @classmethod
    def setup_default_input(cls, input):
        pass

    @classmethod
    def setup_requirements(cls, requirements, input):
        pass
