package ru.yandex.tasklet;

import java.io.IOException;
import java.nio.file.Path;
import java.util.Collection;
import java.util.List;
import java.util.function.Consumer;
import java.util.stream.Collectors;

import ru.yandex.ci.tasklet.SandboxResource;
import ru.yandex.sandbox.tasklet.sidecars.resource_manager.proto.ResourceManagerAPIGrpc.ResourceManagerAPIBlockingStub;
import ru.yandex.sandbox.tasklet.sidecars.resource_manager.proto.ResourceManagerApi;
import ru.yandex.sandbox.tasklet.sidecars.resource_manager.proto.ResourceManagerApi.CreateResourceRequest;
import ru.yandex.sandbox.tasklet.sidecars.resource_manager.proto.ResourceOuterClass;

public class SandboxResourcesContext {

    private final ResourceManagerAPIBlockingStub resourceManager;
    private final Path rootDir;

    public SandboxResourcesContext(ResourceManagerAPIBlockingStub resourceManager, Path rootDir) {
        this.resourceManager = resourceManager;
        this.rootDir = rootDir;
    }

    public ResourceOuterClass.Resource createRawResource(
            String filename,
            Consumer<CreateResourceRequest.Builder> metadataOp,
            FileAction writeOp
    ) {
        var newFile = rootDir.resolve(filename);

        var request = CreateResourceRequest.newBuilder();
        metadataOp.accept(request);
        request.setPath(newFile.toAbsolutePath().toString());

        try {
            writeOp.writeFile(newFile);
        } catch (IOException e) {
            throw new TaskletRuntimeException("Unable to write file " + newFile, e);
        }

        var response = resourceManager.createResource(request.build());
        return response.getResource();
    }

    public SandboxResource createResource(
            String filename,
            Consumer<CreateResourceRequest.Builder> metadataOp,
            FileAction writeOp
    ) {
        return toCiResource(createRawResource(filename, metadataOp, writeOp));
    }

    public Path downloadResource(long resourceId) {
        var request = ResourceManagerApi.DownloadResourceRequest.newBuilder().setId(resourceId).build();
        var response = resourceManager.downloadResource(request);
        return Path.of(response.getPath());
    }

    public List<ResourceOuterClass.Resource> getRawResources(
            Consumer<ResourceManagerApi.GetResourcesRequest.Builder> searchOp
    ) {
        var request = ResourceManagerApi.GetResourcesRequest.newBuilder();
        searchOp.accept(request);
        var response = resourceManager.getResources(request.build());
        return response.getResourcesList();
    }

    public List<SandboxResource> getResources(
            Consumer<ResourceManagerApi.GetResourcesRequest.Builder> searchOp
    ) {
        return toCiResources(getRawResources(searchOp));
    }

    public SandboxResource toCiResource(ResourceOuterClass.Resource sandboxResource) {
        return SandboxResource.newBuilder()
                .setId(sandboxResource.getId())
                .setType(sandboxResource.getType())
                .setTaskId(sandboxResource.getTask())
                .putAllAttributes(sandboxResource.getAttributesMap())
                .build();
    }

    public List<SandboxResource> toCiResources(Collection<ResourceOuterClass.Resource> sandboxResources) {
        return sandboxResources.stream()
                .map(this::toCiResource)
                .collect(Collectors.toList());
    }

}
