package ru.yandex.tasklet;

import java.nio.file.Path;
import java.util.function.Function;
import java.util.function.Supplier;

import javax.annotation.Nullable;

import com.google.common.base.Preconditions;
import com.google.common.base.Suppliers;
import io.grpc.ManagedChannel;
import io.grpc.inprocess.InProcessChannelBuilder;
import io.grpc.netty.NettyChannelBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.sandbox.tasklet.sidecars.resource_manager.proto.ResourceManagerAPIGrpc;
import ru.yandex.tasklet.api.v2.ContextOuterClass;
import ru.yandex.tasklet.api.v2.ExecutorServiceGrpc;
import ru.yandex.tasklet.api.v2.ExecutorServiceOuterClass.GetContextRequest;
import ru.yandex.tasklet.api.v2.ExecutorServiceOuterClass.GetSecretRefRequest;
import ru.yandex.tasklet.api.v2.ExecutorServiceOuterClass.SecretValue;
import ru.yandex.tasklet.api.v2.WellKnownStructures.SecretRef;

public class TaskletContext implements AutoCloseable {
    private static final Logger log = LoggerFactory.getLogger(TaskletContext.class);

    @Nullable
    private final Path rootDir;
    private final ManagedChannel executorChannel;
    private final ManagedChannel resourcesChannel;
    private final Supplier<ExecutorServiceGrpc.ExecutorServiceBlockingStub> executorStub;
    private final Supplier<ResourceManagerAPIGrpc.ResourceManagerAPIBlockingStub> resourceStub;
    private final Supplier<ContextOuterClass.Context> context;

    TaskletContext(@Nullable Path rootDir, String address, Function<String, ManagedChannel> channelSource) {
        this.rootDir = rootDir;

        log.debug("Opening gRPC client connection to Executor service: {}", address);
        this.executorChannel = channelSource.apply(address);
        this.executorStub = Suppliers.memoize(() -> ExecutorServiceGrpc.newBlockingStub(executorChannel));
        this.context = Suppliers.memoize(() ->
                getExecutor().getContext(GetContextRequest.getDefaultInstance()).getContext());

        var env = getTaskletContext().getEnvironment();

        var sandboxResources = env.getSandboxResourceManager();
        if (sandboxResources.getEnabled()) {
            var resourcesAddress = env.getSandboxResourceManager().getAddress();
            Preconditions.checkState(!resourcesAddress.isEmpty(),
                    "Internal error. Resources sidecar is enable but no address configured");
            log.debug("Opening gRPC client connection to Resources sidecar: {}", resourcesAddress);
            this.resourcesChannel = channelSource.apply(resourcesAddress);
            this.resourceStub = Suppliers.memoize(() -> ResourceManagerAPIGrpc.newBlockingStub(resourcesChannel));
        } else {
            log.debug("Resource sidecar is not available");
            this.resourcesChannel = null;
            this.resourceStub = null;
        }
    }

    public ContextOuterClass.Context getTaskletContext() {
        return context.get();
    }

    public SecretValue getSecretValue(SecretRef ref) {
        var request = GetSecretRefRequest.newBuilder().setRef(ref).build();
        return getExecutor().getSecretRef(request).getValue();
    }

    public SandboxResourcesContext getSandboxResourcesContext() {
        Preconditions.checkState(rootDir != null,
                "Unable to get resource service, root dir is not configured");
        return new SandboxResourcesContext(getSandboxResourceManager(), rootDir);
    }

    //

    public ExecutorServiceGrpc.ExecutorServiceBlockingStub getExecutor() {
        return executorStub.get();
    }

    public ResourceManagerAPIGrpc.ResourceManagerAPIBlockingStub getSandboxResourceManager() {
        Preconditions.checkState(resourceStub != null,
                "resourceStub is not available, enable it in t.yaml and test classes");
        return resourceStub.get();
    }

    @Override
    public void close() {
        if (resourcesChannel != null) {
            log.debug("Closing gRPC client connection to Resources sidecar: {}", resourcesChannel);
            resourcesChannel.shutdown();
        }

        log.debug("Closing gRPC client connection to Executor service: {}", executorChannel);
        executorChannel.shutdown();
    }
    //

    public static TaskletContext fromAddress(Path rootDir, String address) {
        return new TaskletContext(
                rootDir,
                address,
                networkAddress -> NettyChannelBuilder.forTarget(networkAddress).usePlaintext().build()
        );
    }

    public static TaskletContext fromInprocess(Path rootDir, String channelName) {
        return new TaskletContext(
                rootDir,
                channelName,
                channel -> InProcessChannelBuilder.forName(channel).directExecutor().build()
        );
    }

}
