package ru.yandex.tasklet;

import java.nio.file.Path;
import java.util.Objects;

import com.google.common.base.Preconditions;
import com.google.protobuf.Message;
import com.google.protobuf.TextFormat;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class TaskletRuntime<Input extends Message, Output extends Message> implements AutoCloseable {
    private static final Logger log = LoggerFactory.getLogger(TaskletRuntime.class);

    private final InputOutputSettings settings;
    private final Input defaultInput;
    private final Output defaultOutput;
    private final TaskletContext context;

    TaskletRuntime(
            InputOutputSettings settings,
            Input defaultInput,
            Output defaultOutput
    ) {
        Preconditions.checkArgument(settings != null, "settings cannot be null");

        this.settings = settings;
        this.defaultInput = defaultInput;
        this.defaultOutput = defaultOutput;

        this.context = TaskletContext.fromAddress(Path.of(""), settings.getServiceAddress());
        TaskletUtils.validate(context.getTaskletContext(), defaultInput, defaultOutput);
    }

    public void execute(TaskletAction<Input, Output> action) {
        Preconditions.checkArgument(action != null, "action cannot be null");

        var input = TaskletUtils.readProto(settings.getInputFileReference(), defaultInput);
        Preconditions.checkState(Objects.equals(input.getClass(), defaultInput.getClass()),
                "Internal error. Expect input of %s, got %s", defaultInput.getClass(), input.getClass());

        Result<Output> result = null;
        try {
            result = action.execute(input, context);
        } catch (RuntimeException re) {
            throw re;
        } catch (Exception e) {
            throw new TaskletRuntimeException("Unable to execute tasklet action", e);
        }

        if (result.hasResult()) {
            var filename = settings.getOutputFileReference();
            log.info("Store output into [{}]", filename);
            var output = result.getResult();
            Preconditions.checkState(Objects.equals(output.getClass(), defaultOutput.getClass()),
                    "Internal error. Expect output of %s, got %s", defaultOutput.getClass(), output.getClass());
            TaskletUtils.writeProto(filename, output);
        } else if (result.hasError()) {
            var filename = settings.getErrorFileReference();
            var userError = result.getError();
            if (filename != null) {
                log.info("Store user error into [{}]", filename);
                TaskletUtils.writeProto(filename, userError);
            } else {
                throw new RuntimeException(TextFormat.shortDebugString(userError));
            }
        }
    }

    @Override
    public void close() {
        context.close();
    }

    //

    //

    static <Input extends Message, Output extends Message> TaskletRuntime<Input, Output> createRuntime(
            Class<Input> inputMessageType,
            Class<Output> outputMessageType,
            String[] args
    ) {
        Preconditions.checkArgument(inputMessageType != null, "inputMessageType cannot be null");
        Preconditions.checkArgument(outputMessageType != null, "outputMessageType cannot be null");

        var settings = InputOutputSettings.parse(args);
        return new TaskletRuntime<>(
                settings,
                TaskletUtils.getDefaultMessageInstance(inputMessageType),
                TaskletUtils.getDefaultMessageInstance(outputMessageType)
        );
    }

    /**
     * Execute tasklet action
     *
     * @param inputMessageType  input message type
     * @param outputMessageType output message type
     * @param args              command line args
     * @param action            action to execute (input and context will be fed, output is expected)
     * @param <Input>           tasklet input type
     * @param <Output>          tasklet output type
     */
    public static <Input extends Message, Output extends Message> void execute(
            Class<Input> inputMessageType,
            Class<Output> outputMessageType,
            String[] args,
            TaskletAction<Input, Output> action
    ) {
        try (var runtime = createRuntime(inputMessageType, outputMessageType, args)) {
            runtime.execute(action);
        }
    }
}
