package ru.yandex.tasklet;

import java.io.FileInputStream;
import java.io.FileOutputStream;
import java.io.IOException;

import com.google.common.base.Preconditions;
import com.google.protobuf.Message;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.tasklet.api.v2.ContextOuterClass;

public class TaskletUtils {
    private static final Logger log = LoggerFactory.getLogger(TaskletUtils.class);

    private TaskletUtils() {
        //
    }

    static <Input extends Message, Output extends Message> void validate(
            ContextOuterClass.Context context,
            Input defaultInput,
            Output defaultOutput
    ) {
        log.debug("Current Tasklet context: {}", context);

        var protoDesc = context.getSchema().getSimpleProto();

        match(defaultInput, protoDesc.getInputMessage(), "Input");
        match(defaultOutput, protoDesc.getOutputMessage(), "Output");
    }

    @SuppressWarnings("unchecked")
    public static <T extends Message> T readProto(String fileReference, T defaultValue) {
        try (var stream = new FileInputStream(fileReference)) {
            return (T) defaultValue.toBuilder()
                    .mergeFrom(stream)
                    .build();
        } catch (IOException e) {
            throw new TaskletRuntimeException("Internal error. Unable to read from " + fileReference, e);
        }
    }

    public static <T extends Message> void writeProto(String fileReference, T output) {
        try (var stream = new FileOutputStream(fileReference)) {
            output.writeTo(stream);
        } catch (IOException e) {
            throw new TaskletRuntimeException("Internal error. Unable to write to " + fileReference, e);
        }
    }

    @SuppressWarnings("unchecked")
    public static <T extends Message> T getDefaultMessageInstance(Class<T> type) {
        checkClass(type);
        try {
            return (T) type.getMethod("getDefaultInstance").invoke(null);
        } catch (Exception ex) {
            throw new TaskletRuntimeException("Unable to get default instance from " + type, ex);
        }
    }

    private static <T extends Message> void match(T runtimeMessage, String contextDescriptor, String messageType) {
        var runtimeDescriptor = runtimeMessage.getDescriptorForType().getFullName();
        if (!runtimeDescriptor.equals(contextDescriptor)) {
            throw new TaskletRuntimeException(String.format(
                    "%s message mismatch. Context proto: %s, Runtime proto: %s",
                    messageType, contextDescriptor, runtimeDescriptor));
        }
    }

    private static <T extends Message> void checkClass(Class<T> type) {
        Preconditions.checkState(
                Message.class.isAssignableFrom(type),
                "Class %s expected to be generated protobuf message",
                type
        );
    }
}
