package ru.yandex.tasklet.test;

import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.atomic.AtomicReference;
import java.util.function.Consumer;
import java.util.function.Supplier;

import io.grpc.Status;
import io.grpc.stub.StreamObserver;

import ru.yandex.tasklet.api.v2.ContextOuterClass.Context;
import ru.yandex.tasklet.api.v2.ExecutorServiceGrpc;
import ru.yandex.tasklet.api.v2.ExecutorServiceOuterClass.GetContextRequest;
import ru.yandex.tasklet.api.v2.ExecutorServiceOuterClass.GetContextResponse;
import ru.yandex.tasklet.api.v2.ExecutorServiceOuterClass.GetSecretRefRequest;
import ru.yandex.tasklet.api.v2.ExecutorServiceOuterClass.GetSecretRefResponse;
import ru.yandex.tasklet.api.v2.ExecutorServiceOuterClass.SecretValue;
import ru.yandex.tasklet.api.v2.WellKnownStructures.SecretRef;

public class ExecutorServiceStub extends ExecutorServiceGrpc.ExecutorServiceImplBase {

    private final Supplier<Context> initialContext;
    private final Map<SecretRef, SecretValue> secrets = new ConcurrentHashMap<>();
    private final AtomicReference<Context> contextReference = new AtomicReference<>(Context.getDefaultInstance());

    public ExecutorServiceStub(Supplier<Context> initialContext) {
        this.initialContext = initialContext;
    }

    public void clear() {
        secrets.clear();
        contextReference.set(initialContext.get());
    }

    public Map<SecretRef, SecretValue> getModifiableSecretsMap() {
        return secrets;
    }

    public void updateContext(Consumer<Context.Builder> updateContext) {
        var ref = contextReference.get().toBuilder();
        updateContext.accept(ref);
        contextReference.set(ref.build());
    }

    @Override
    public void getContext(
            GetContextRequest request,
            StreamObserver<GetContextResponse> responseObserver
    ) {
        var context = contextReference.get();
        responseObserver.onNext(GetContextResponse.newBuilder()
                .setContext(context)
                .build());
        responseObserver.onCompleted();
    }

    @Override
    public void getSecretRef(
            GetSecretRefRequest request,
            StreamObserver<GetSecretRefResponse> responseObserver
    ) {
        var value = secrets.get(request.getRef());
        if (value == null) {
            responseObserver.onError(Status.NOT_FOUND
                    .withDescription("Secret is not found: " + request.getRef())
                    .asRuntimeException());
        } else {
            responseObserver.onNext(GetSecretRefResponse.newBuilder()
                    .setValue(value)
                    .build());
        }
        responseObserver.onCompleted();
    }
}
