package ru.yandex.tasklet.test;

import java.util.Map;
import java.util.Objects;
import java.util.concurrent.ConcurrentSkipListMap;
import java.util.concurrent.atomic.AtomicLong;
import java.util.stream.Collectors;

import io.grpc.Status;
import io.grpc.stub.StreamObserver;

import ru.yandex.sandbox.tasklet.sidecars.resource_manager.proto.ResourceManagerAPIGrpc;
import ru.yandex.sandbox.tasklet.sidecars.resource_manager.proto.ResourceManagerApi.CreateResourceRequest;
import ru.yandex.sandbox.tasklet.sidecars.resource_manager.proto.ResourceManagerApi.CreateResourceResponse;
import ru.yandex.sandbox.tasklet.sidecars.resource_manager.proto.ResourceManagerApi.DownloadResourceRequest;
import ru.yandex.sandbox.tasklet.sidecars.resource_manager.proto.ResourceManagerApi.DownloadResourceResponse;
import ru.yandex.sandbox.tasklet.sidecars.resource_manager.proto.ResourceManagerApi.GetResourcesRequest;
import ru.yandex.sandbox.tasklet.sidecars.resource_manager.proto.ResourceManagerApi.GetResourcesResponse;
import ru.yandex.sandbox.tasklet.sidecars.resource_manager.proto.ResourceOuterClass.Resource;

public class ResourceManagerAPIStub extends ResourceManagerAPIGrpc.ResourceManagerAPIImplBase {

    private final AtomicLong resourceId = new AtomicLong();
    private final Map<Long, ResourceWithPath> resources = new ConcurrentSkipListMap<>();

    public void clear() {
        resources.clear();
    }

    @Override
    public void downloadResource(
            DownloadResourceRequest request,
            StreamObserver<DownloadResourceResponse> responseObserver
    ) {
        var resource = resources.get(request.getId());
        if (resource == null) {
            throw Status.NOT_FOUND.withDescription("Unable to find resource " + request.getId()).asRuntimeException();
        }

        responseObserver.onNext(DownloadResourceResponse.newBuilder()
                .setPath(resource.path)
                .build());
        responseObserver.onCompleted();
    }

    @Override
    public void createResource(
            CreateResourceRequest request,
            StreamObserver<CreateResourceResponse> responseObserver
    ) {
        var id = resourceId.incrementAndGet();
        var resource = Resource.newBuilder()
                .setId(id)
                .setType(request.getType())
                .setArch(request.getArch())
                .setOwner(request.getOwner())
                .putAllAttributes(request.getAttributesMap())
                .setDescription(request.getDescription())
                .build();

        resources.put(id, new ResourceWithPath(resource, request.getPath()));
        responseObserver.onNext(CreateResourceResponse.newBuilder()
                .setResource(resource)
                .build());
        responseObserver.onCompleted();
    }

    @Override
    public void getResources(
            GetResourcesRequest request,
            StreamObserver<GetResourcesResponse> responseObserver
    ) {
        var filteredList = request.getIdsList().isEmpty()
                ? resources.values()
                :
                request.getIdsList().stream()
                        .map(resources::get)
                        .filter(Objects::nonNull)
                        .collect(Collectors.toList());

        var response = GetResourcesResponse.newBuilder();
        filteredList.stream()
                .map(ref -> ref.resource)
                .filter(resource -> match(resource, request))
                .skip(request.getOffset())
                .limit(request.getLimit() > 0 ? request.getLimit() : Integer.MAX_VALUE)
                .forEach(response::addResources);

        responseObserver.onNext(response.build());
        responseObserver.onCompleted();
    }

    private static boolean match(Resource resource, GetResourcesRequest request) {
        if (!request.getType().isEmpty()) {
            if (!request.getType().equals(resource.getType())) {
                return false;
            }
        }
        boolean anyMatched = false;
        boolean allMatched = true;
        for (var e : request.getAttributesQuery().getAttributesMap().entrySet()) {
            boolean matched = Objects.equals(resource.getAttributesMap().get(e.getKey()), e.getValue());
            anyMatched = anyMatched || matched;
            allMatched = allMatched && matched;
        }

        if (request.getAttributesQuery().getAnyAttr()) {
            return anyMatched;
        } else {
            return allMatched;
        }
    }


    private static class ResourceWithPath {
        private final Resource resource;
        private final String path;

        private ResourceWithPath(Resource resource, String path) {
            this.resource = resource;
            this.path = path;
        }
    }
}
