package ru.yandex.tasklet.test;

import java.io.IOException;
import java.nio.file.Path;
import java.util.UUID;

import javax.annotation.Nullable;

import com.google.common.base.Preconditions;
import com.google.protobuf.Message;
import io.grpc.Server;
import io.grpc.ServerBuilder;
import io.grpc.inprocess.InProcessServerBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import ru.yandex.tasklet.TaskletContext;
import ru.yandex.tasklet.TaskletUtils;
import ru.yandex.tasklet.api.v2.ContextOuterClass;

public class TaskletContextStub implements AutoCloseable {
    private static final Logger log = LoggerFactory.getLogger(TaskletContextStub.class);

    private final ExecutorServiceStub executorServiceStub;
    private final ResourceManagerAPIStub resourceManagerAPIStub;
    private final TaskletContext context;
    private final Server server;

    public TaskletContextStub(
            ExecutorServiceStub executorServiceStub,
            ResourceManagerAPIStub resourceManagerAPIStub,
            TaskletContext context,
            Server server
    ) {
        this.executorServiceStub = executorServiceStub;
        this.resourceManagerAPIStub = resourceManagerAPIStub;
        this.context = context;
        this.server = server;
    }

    public ExecutorServiceStub getExecutorServiceStub() {
        return executorServiceStub;
    }

    public ResourceManagerAPIStub getResourceManagerAPIStub() {
        return resourceManagerAPIStub;
    }

    public TaskletContext getContext() {
        return context;
    }

    Server getServer() {
        return server;
    }

    @Override
    public void close() {
        context.close();

        log.debug("Closing gRPC server: {}", server);
        server.shutdown();
    }

    //

    public static <Input extends Message, Output extends Message> TaskletContextStub stub(
            Class<Input> inputMessageType,
            Class<Output> outputMessageType
    ) {
        return new Builder<>(inputMessageType, outputMessageType, null).inprocess();
    }

    public static <Input extends Message, Output extends Message> TaskletContextStub stub(
            Class<Input> inputMessageType,
            Class<Output> outputMessageType,
            Path tempDir
    ) {
        return new Builder<>(inputMessageType, outputMessageType, tempDir).inprocess();
    }

    // For internal testing
    static <Input extends Message, Output extends Message> TaskletContextStub networkStub(
            Class<Input> inputMessageType,
            Class<Output> outputMessageType
    ) {
        return new Builder<>(inputMessageType, outputMessageType, null).network();
    }
    //

    private static class Builder<Input extends Message, Output extends Message> {
        private final ContextOuterClass.Context.Builder contextSource = ContextOuterClass.Context.newBuilder();
        private final ExecutorServiceStub executorServiceStub = new ExecutorServiceStub(contextSource::build);
        private final ResourceManagerAPIStub resourceManagerAPIStub = new ResourceManagerAPIStub();
        private final Message inputMessage;
        private final Message outputMessage;
        private final Path tempDir;

        private Builder(Class<Input> inputMessageType, Class<Output> outputMessageType, @Nullable Path tempDir) {
            this.inputMessage = TaskletUtils.getDefaultMessageInstance(inputMessageType);
            this.outputMessage = TaskletUtils.getDefaultMessageInstance(outputMessageType);
            this.tempDir = tempDir;
        }

        private void startServer(Server server) {
            log.debug("Starting gRPC server: {}", server);
            try {
                server.start();
            } catch (IOException e) {
                throw new RuntimeException("Unable to start gRPC server", e);
            }
        }

        private Server configure(ServerBuilder<?> builder) {
            builder.directExecutor();
            builder.addService(executorServiceStub);
            builder.addService(resourceManagerAPIStub);
            return builder.build();
        }


        private void updateContext(String address) {
            Preconditions.checkState(!contextSource.hasSchema(),
                    "Unable to initialize context more than once");
            contextSource.getSchemaBuilder().getSimpleProtoBuilder()
                    .setInputMessage(inputMessage.getDescriptorForType().getFullName())
                    .setOutputMessage(outputMessage.getDescriptorForType().getFullName());
            contextSource.getEnvironmentBuilder().getSandboxResourceManagerBuilder()
                    .setEnabled(true)
                    .setAddress(address);
            executorServiceStub.clear();
        }

        private TaskletContextStub buildStub(TaskletContext taskletContext, Server server) {
            return new TaskletContextStub(executorServiceStub, resourceManagerAPIStub, taskletContext, server);
        }

        TaskletContextStub inprocess() {
            var channelName = UUID.randomUUID().toString();
            var server = configure(InProcessServerBuilder.forName(channelName));

            this.startServer(server);
            this.updateContext(channelName);

            log.debug("Creating in-process gRPC server: {}", channelName);
            return buildStub(TaskletContext.fromInprocess(tempDir, channelName), server);
        }

        // for internal testing
        TaskletContextStub network() {
            var server = configure(ServerBuilder.forPort(0));

            this.startServer(server);
            var address = "localhost:" + server.getPort();
            this.updateContext(address);

            log.debug("Creating network-based gRPC server: {}", address);
            return buildStub(TaskletContext.fromAddress(tempDir, address), server);
        }
    }

}
