import logging

from google.protobuf import empty_pb2

from library.python import vault_client as lpvault

from tasklet import runtime
from tasklet.domain import oauth

from .proto import yav_pb2
from .proto import yav_pb2_grpc

from sandbox import sdk2


logger = logging.getLogger(__name__)


def get_client():
    import blackboxer
    try:
        bb_client = blackboxer.Blackbox("http://blackbox.yandex-team.ru/blackbox/")
        token = oauth.get_token()
        data = bb_client.oauth("127.0.0.1", token)
        if "vault:use" not in data["oauth"]["scope"].split():
            return None
    except blackboxer.BlackboxError as exc:
        logger.warning(
            "Can't validate tasklet token scope by blackbox. Try to use tasklet token for yav. Blackbox error: %s",
            exc,
        )

    return lpvault.instances.Production(
        rsa_auth=False,
        authorization="OAuth {}".format(oauth.get_token())
    )


class VaultAccessDenied(Exception):
    pass


class YavAdapter(object):

    def __init__(self, service):
        self._service = service

    def get_secret(self, yav_secret_spec, default_uuid="", default_key=""):
        if not yav_secret_spec.uuid:
            yav_secret_spec.uuid = default_uuid
        if not yav_secret_spec.key:
            yav_secret_spec.key = default_key
        logger.debug("Fetching key '%s' for uuid '%s'", yav_secret_spec.key, yav_secret_spec.uuid)
        return self._service.GetSecret(yav_secret_spec)


class TestYavService(yav_pb2_grpc.YavServicer):

    def __init__(self):
        pass

    def GetContext(self, request, context):
        return yav_pb2.YavContext()

    def Inject(self, request, context):
        runtime.inject(request, self)

        return empty_pb2.Empty()

    def GetSecret(self, request, context):
        response = yav_pb2.YavSecret()
        response.secret = "test value of '{}' at version {}".format(request.key, request.uuid)
        return response


class YavService(yav_pb2_grpc.YavServicer):

    def __init__(self):
        pass

    def GetContext(self, request, context):
        return yav_pb2.YavContext()

    def Inject(self, request, context):
        runtime.inject(request, self)

        return empty_pb2.Empty()

    def GetSecret(self, request, context):
        response = yav_pb2.YavSecret()
        client = get_client()
        if client is None:
            if sdk2.Task.current:
                response.secret = sdk2.yav.Secret(request.uuid, default_key=request.key or None).value()
            else:
                raise VaultAccessDenied("Tasklet token does not have scope vault:use")
        else:
            version = client.get_version(request.uuid)
            response.secret = version["value"][request.key]
        return response
