from typing import Optional

import logging
import os
import subprocess as sp

from tasklet.api.v2 import context_pb2
from tasklet.api.v2 import executor_service_pb2
from tasklet.api.v2 import executor_service_pb2_grpc
from tasklet.api.v2 import well_known_structures_pb2 as wks_pb2

from tasklet.experimental.sdk.py.dummy import interface as dummy_sdk

from tasklet.tasklets.lifecycle.internal import os as internal_os


class ArcError(Exception):
    pass


class Arc:

    DEFAULT_BRANCH = "trunk"

    def __init__(self, binary_path: str, token: str, logger: logging.Logger = None):
        self.__binary_path = binary_path
        self.__token = token

        if logger is None:
            logger = logging.getLogger("arc-tool")
        self.logger = logger

        self.__mount_path = None
        self.__store_path = None

    @property
    def mount_path(self) -> Optional[str]:
        return self.__mount_path

    def __enter__(self):
        return self

    def __exit__(self, *_):
        if self.__mount_path is not None:
            self.unmount()

    def __env(self):
        return {
            "ARC_TOKEN": self.__token,
        }

    def __run_command(self, args, **kwargs) -> sp.Popen:
        args = [self.__binary_path] + args
        return internal_os.run_command(self.logger, args, **kwargs)

    def mount(self, mount_path: str = None, store_path: str = None) -> "Arc":
        if self.__mount_path:
            raise ValueError(f"Arc is mounted already on {self.__mount_path}")

        if mount_path is None:
            mount_path = os.path.join(os.getcwd(), "arcadia")
        self.__mount_path = mount_path

        if store_path is None:
            store_path = os.path.join(os.path.dirname(mount_path), "arcadia_store")
        self.__store_path = store_path

        os.makedirs(mount_path, exist_ok=True)
        os.makedirs(store_path, exist_ok=True)

        p = self.__run_command(
            [
                "mount",
                "--mount", self.__mount_path,
                "--store", self.__store_path,
                "--no-ssh-tokens",
            ],
            env=self.__env(),
        )
        if p.returncode != 0:
            raise ArcError(f"Arc mount failed with return code {p.returncode}")
        return self

    def checkout(self, commit_or_branch: str):
        p = self.__run_command(
            ["checkout", "-q", commit_or_branch],
            env=self.__env(),
        )
        if p.returncode != 0:
            raise ArcError(f"Arc checkout failed with return code {p.returncode}")

    def unmount(self):
        if self.__mount_path is None:
            raise ValueError("Arc is not mounted")

        p = self.__run_command(
            [
                "unmount",
                self.__mount_path,
                "--forget",
            ],
            env=self.__env(),
        )
        if p.returncode != 0:
            raise ArcError(f"Arc unmount failed with return code {p.returncode}")
        self.__mount_path = None


def get_arc(tasklet_io: dummy_sdk.TaskletInterface, arc_token_ref: wks_pb2.SecretRef) -> Arc:
    client: executor_service_pb2_grpc.ExecutorServiceStub = tasklet_io.executor_client
    context: context_pb2.Context = client.GetContext(executor_service_pb2.GetContextRequest()).context
    token = tasklet_io.get_secret(arc_token_ref)
    return Arc(context.environment.arc_client.path, token.value)
