from typing import List

import logging
import json
import os
import subprocess as sp

from google.protobuf import json_format
from google.protobuf import message as pb_message

from tasklet.api.v2 import context_pb2
from tasklet.api.v2 import data_model_pb2
from tasklet.api.v2 import executor_service_pb2
from tasklet.api.v2 import executor_service_pb2_grpc
from tasklet.api.v2 import schema_registry_service_pb2
from tasklet.api.v2 import tasklet_service_pb2
from tasklet.api.v2 import well_known_structures_pb2 as wks_pb2

from sandbox.tasklet.sidecars.resource_manager.proto import resource_manager_api_pb2 as sbr_api_pb2

from tasklet.experimental.sdk.py.dummy import interface as dummy_sdk

from tasklet.tasklets.lifecycle.internal import os as internal_os
from tasklet.tasklets.lifecycle.internal import sbr


class TaskletTool:

    TOKEN_ENV_VAR_NAME = "TASKLET_TOKEN"

    def __init__(self, token: str, binary_path: str, cluster: str, logger=None):
        self.__token = token
        self.__binary_path = binary_path
        self.__tasklet_cluster = cluster or "prod"

        if logger is None:
            logger = logging.getLogger("tasklet-tool")
        self.logger = logger

    def __command_env(self):
        env = os.environ.copy()
        env[self.TOKEN_ENV_VAR_NAME] = self.__token
        return env

    def __run_command(self, args: List[str]) -> str:
        args = (
            [self.__binary_path] +
            args[:1] +
            ["-c", self.__tasklet_cluster] +
            args[1:] +
            ["--output", "json"]
        )
        p = internal_os.run_command(self.logger, args, env=self.__command_env(), stdout=sp.PIPE, text=True)
        stdout, _ = p.communicate()
        return stdout

    def __run_command_json(self, args: List[str]) -> dict:
        stdout = self.__run_command(args)
        try:
            return json.loads(stdout)
        except:
            logging.error("Failed to decode command output:\n%s\n", stdout)
            raise

    def __run_command_proto(self, args: List[str], message: pb_message.Message) -> pb_message.Message:
        obj = self.__run_command_json(args)
        try:
            return json_format.ParseDict(obj, message, ignore_unknown_fields=True)
        except:
            logging.error("Failed to parse obj to message %s:\n%s", type(message), obj)
            raise

    def register_schema(self, schema_path: str) -> schema_registry_service_pb2.CreateSchemaResponse:
        # noinspection PyTypeChecker
        return self.__run_command_proto(
            ["sr", "put", schema_path],
            schema_registry_service_pb2.CreateSchemaResponse(),
        )

    def register_build(self, sbr_id: int, schema_id: str) -> data_model_pb2.Build:
        # noinspection PyTypeChecker
        resp: tasklet_service_pb2.GetBuildResponse = self.__run_command_proto(
            ["build", "create", "--schema-id", schema_id, str(sbr_id)],
            tasklet_service_pb2.GetBuildResponse(),
        )
        return resp.build

    def move_label(self, label_name: str, build_id: str) -> data_model_pb2.Label:
        # noinspection PyTypeChecker
        resp: tasklet_service_pb2.GetLabelResponse = self.__run_command_proto(
            ["label", "move", label_name, "--to", build_id],
            tasklet_service_pb2.GetLabelResponse(),
        )
        label = resp.label
        self.logger.debug("Label '%s' moved to build %s", label.meta.name, label.spec.build_id)
        return label


def get_tasklet_tool(
    tasklet_io: dummy_sdk.TaskletInterface, tasklet_token_ref: wks_pb2.SecretRef, tasklet_cluster: str,
) -> TaskletTool:

    client: executor_service_pb2_grpc.ExecutorServiceStub = tasklet_io.executor_client
    context: context_pb2.Context = client.GetContext(executor_service_pb2.GetContextRequest()).context

    sbr_client = sbr.get_sbr_client(context)

    resources_resp: sbr_api_pb2.GetResourcesResponse = sbr_client.GetResources(
        sbr_api_pb2.GetResourcesRequest(
            type="TASKLET_TOOL",
            state="READY",
            owner="TASKLETS",
            attributes_query=sbr_api_pb2.GetResourcesRequest.AttributesQuery(
                attributes={"released": "stable"},
            ),
            limit=1,
        ),
    )
    if not resources_resp.resources:
        raise EnvironmentError("tasklet-tool cannot be found")

    download_resp: sbr_api_pb2.DownloadResourceResponse = sbr_client.DownloadResource(
        sbr_api_pb2.DownloadResourceRequest(id=resources_resp.resources[0].id),
    )

    token = tasklet_io.get_secret(tasklet_token_ref)
    return TaskletTool(token.value, download_resp.path, tasklet_cluster)
