package main

import (
	"context"
	"fmt"
	"sync"

	"a.yandex-team.ru/library/go/core/log"
	"a.yandex-team.ru/library/go/maxprocs"
	"a.yandex-team.ru/travel/library/go/grpcgateway"
	"a.yandex-team.ru/travel/library/go/logging"

	bizdev "a.yandex-team.ru/travel/api/proto/bizdev_admin"
)

const (
	grpcGatewayServiceName = "bizdev_admin_api"
)

var grpcGatewayConfig = grpcgateway.Config{
	Enabled:       true,
	Address:       "0.0.0.0:8081",
	EnableBinary:  false,
	EnableSwagger: true,
}

var loggingConfig = logging.Config{
	Level:                        "info",
	ErrorBoosterLevel:            "error",
	ErrorBoosterAdditionalFields: nil,
}

func main() {
	maxprocs.AdjustAuto()

	ctx, ctxCancel := context.WithCancel(context.Background())
	defer ctxCancel()

	logger, err := logging.NewDeploy(&loggingConfig)
	if err != nil {
		fmt.Println("failed to create logger, err:", err)
		return
	}

	defer func() {
		err = logger.L.Sync()
		if err != nil {
			fmt.Println("failed to close logger:", err)
			return
		}
	}()

	var YandexUIDHeader = grpcgateway.HeaderSpec{
		Name:        "X-Ya-YandexUid",
		Description: "YandexUid",
	}
	var YandexPassportIDHeader = grpcgateway.HeaderSpec{
		Name:        "X-Ya-PassportId",
		Description: "PassportId",
	}
	var UserTicketHeader = grpcgateway.HeaderSpec{
		Name:        "X-Ya-User-Ticket",
		Description: "User Ticket",
	}
	var ServiceTicketHeader = grpcgateway.HeaderSpec{
		Name:        "X-Ya-Service-Ticket",
		Description: "Service Ticket",
	}
	var LoginHeader = grpcgateway.HeaderSpec{
		Name:        "X-Ya-Login",
		Description: "Login",
	}

	headersToPass := []grpcgateway.HeaderSpec{
		YandexUIDHeader,
		YandexPassportIDHeader,
		UserTicketHeader,
		ServiceTicketHeader,
		LoginHeader,
	}

	grpcGateway := grpcgateway.NewGateway(
		&grpcGatewayConfig,
		grpcgateway.NewService(
			grpcGatewayServiceName,
			"/api/bizdev-admin",
			"localhost:9001",
			bizdev.RegisterBizdevAdminApiHandlerFromEndpoint,
			headersToPass),
		grpcgateway.NewService(
			grpcGatewayServiceName,
			"/",
			"localhost:9001",
			bizdev.RegisterBizdevAdminApiHandlerFromEndpoint,
			headersToPass),
	)

	// running server
	wg := sync.WaitGroup{}
	wg.Add(1)

	go func() {
		defer wg.Done()
		if !grpcGatewayConfig.Enabled {
			return
		}
		err = grpcGateway.Run(ctx)
		if err != nil {
			logger.Info("GRPC-gateway server closed", log.Error(err))
		}
	}()

	logger.Info("Started")
	wg.Wait()
}
