package ru.yandex.travel.api.config.common;

import java.time.Duration;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.stream.Collectors;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import io.micrometer.core.instrument.Meter;
import io.micrometer.core.instrument.MeterRegistry;
import io.micrometer.core.instrument.Tag;
import io.micrometer.core.instrument.Tags;
import io.micrometer.core.instrument.Timer;
import io.micrometer.core.instrument.config.MeterFilter;
import io.micrometer.core.instrument.distribution.DistributionStatisticConfig;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.actuate.autoconfigure.metrics.MeterRegistryCustomizer;
import org.springframework.boot.actuate.metrics.web.servlet.WebMvcTags;
import org.springframework.boot.actuate.metrics.web.servlet.WebMvcTagsProvider;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import ru.yandex.travel.commons.metrics.MetricsUtils;

@Slf4j
@Configuration
public class MeterConfiguration {
    public static final String ADDITIONAL_TAGS_ATTR_NAME = "ADDITIONAL_TAGS";

    private static final Duration[] SLAS = {
            Duration.ofMillis(10), Duration.ofMillis(50), Duration.ofMillis(100), Duration.ofMillis(150),
            Duration.ofMillis(200), Duration.ofMillis(500), Duration.ofMillis(1000), Duration.ofMillis(10000)
    };

    private static Tag tagFromStatus(HttpServletRequest request, HttpServletResponse response) {
        String status;
        if (response == null) {
            status = "UNKNOWN";
        } else if (response.getStatus() == 400) {
            status = "400";
        } else if (response.getStatus() == 401) {
            status = "401";
        } else if (response.getStatus() == 404) {
            status = "404";
        } else if (response.getStatus() == 409) {
            status = "409";
        } else {
            status = response.getStatus() / 100 + "xx";
        }
        return Tag.of("status", status);
    }

    @Bean
    public MeterRegistryCustomizer<MeterRegistry> getRegistryCustomizer() {
        return registry -> {
            registry.config().meterFilter(new MeterFilter() {
                @Override
                public DistributionStatisticConfig configure(Meter.Id id, DistributionStatisticConfig config) {
                    if (id.getName().startsWith("http.server.requests")) {
                        return DistributionStatisticConfig.builder()
                                .percentilesHistogram(true)
                                .serviceLevelObjectives(Arrays.stream(SLAS).mapToDouble(Duration::toNanos).toArray())
                                .percentiles(MetricsUtils.higherPercentiles())
                                .build();
                    } else {
                        return config;
                    }
                }
            });
            AtomicBoolean initAllTagsInProgressFlag = new AtomicBoolean(false);
            registry.config().onMeterAdded(meter -> {
                if (meter.getId().getName().startsWith("http.server.requests") && (meter instanceof Timer)) {
                    if (!initAllTagsInProgressFlag.compareAndSet(false, true)) {
                        // some (probably) recursive call is already in progress
                        // the latest micrometer lib version makes the logic below trigger itself infinitely
                        return;
                    }

                    List<Tag> tags = meter.getId().getTags().stream().filter(tag -> !tag.getKey().equals("status")).collect(Collectors.toList());
                    for (Tag tag : Arrays.asList(
                            Tag.of("status", "2xx"), Tag.of("status", "3xx"), Tag.of("status", "400"),
                            Tag.of("status", "401"), Tag.of("status", "404"), Tag.of("status", "409"),
                            Tag.of("status", "4xx"), Tag.of("status", "5xx"))) {
                        if (meter.getId().getTags().contains(tag)) {
                            // skipping the same meter passed to this method
                            continue;
                        }

                        List<Tag> copyTags = new ArrayList<>(tags);
                        copyTags.add(tag);
                        registry.timer(meter.getId().getName(), copyTags);
                    }
                    initAllTagsInProgressFlag.set(false);
                }
            });
        };
    }

    @Bean
    public WebMvcTagsProvider getTagsProvider() {
        return new WebMvcTagsProvider() {
            @Override
            public Iterable<Tag> getTags(HttpServletRequest request, HttpServletResponse response, Object handler, Throwable exception) {
                Tag uriTag = WebMvcTags.uri(request, response);
                if ("/error".equals(uriTag.getValue())) {
                    String originalUri = String.valueOf(request.getAttribute("javax.servlet.error.request_uri"));
                    if (originalUri.startsWith("/api")) {
                        uriTag = Tag.of("uri", originalUri);
                    }
                }
                Tag methodTag = WebMvcTags.method(request);
                Tag exceptionTag = WebMvcTags.exception(exception);
                Tag statusTag = tagFromStatus(request, response);

                var initialTags = Tags.of(methodTag, uriTag, exceptionTag, statusTag);

                var initialTagKeysSet = initialTags.stream().map(x -> x.getKey()).collect(Collectors.toUnmodifiableSet());
                var additionalTags = (List<Tag>)request.getAttribute(ADDITIONAL_TAGS_ATTR_NAME);
                if (additionalTags != null) {
                    var overridenTags = additionalTags.stream().map(Tag::getKey).filter(initialTagKeysSet::contains).collect(Collectors.toUnmodifiableList());
                    if (!overridenTags.isEmpty()) {
                        log.error("Tried to override tags {}, which is prohibited, dropping additional tags", overridenTags);
                    } else {
                        return initialTags.and(additionalTags);
                    }
                }
                return initialTags;
            }

            @Override
            public Iterable<Tag> getLongRequestTags(HttpServletRequest request, Object handler) {
                return Tags.of(WebMvcTags.method(request), WebMvcTags.uri(request, null));
            }
        };
    }
}
