package ru.yandex.travel.api.config.common;

import java.net.InetAddress;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.Set;

import io.grpc.ClientInterceptor;
import io.grpc.ManagedChannelBuilder;
import io.grpc.health.v1.HealthCheckRequest;
import io.grpc.health.v1.HealthCheckResponse;
import io.grpc.health.v1.HealthGrpc;
import io.opentracing.Tracer;
import lombok.RequiredArgsConstructor;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import ru.yandex.travel.api.services.notifier.NotifierClientFactory;
import ru.yandex.travel.commons.concurrent.FutureUtils;
import ru.yandex.travel.grpc.interceptors.DefaultTimeoutClientInterceptor;
import ru.yandex.travel.grpc.interceptors.LoggingClientInterceptor;
import ru.yandex.travel.grpc.interceptors.TracerClientInterceptor;
import ru.yandex.travel.grpc.interceptors.TvmHeaderClientHelper;
import ru.yandex.travel.notifier.subscriptions.v1.SubscriptionsServiceGrpc;
import ru.yandex.travel.orders.client.ChannelState;
import ru.yandex.travel.orders.client.ChannelSupplier;
import ru.yandex.travel.orders.client.GrpcChannelSupplierFactory;
import ru.yandex.travel.orders.client.HAGrpcChannelFactory;
import ru.yandex.travel.orders.client.LabeledChannel;
import ru.yandex.travel.tvm.TvmWrapper;

@Configuration
@RequiredArgsConstructor
@EnableConfigurationProperties(NotifierConfigurationProperties.class)
@Slf4j
public class NotifierConfiguration {
    private final NotifierConfigurationProperties properties;
    private final Tracer tracer;

    @Bean
    public NotifierClientFactory notifierClientFactory(@Qualifier("NotifierGrpcChannelFactory") HAGrpcChannelFactory haGrpcChannelFactory) {
        return () -> SubscriptionsServiceGrpc.newFutureStub(haGrpcChannelFactory.getRoundRobinChannel());
    }

    @Bean("NotifierSupplier")
    public ChannelSupplier getChannelSupplier() {
        GrpcChannelSupplierFactory grpcChannelSupplierFactory = new GrpcChannelSupplierFactory(properties);

        return grpcChannelSupplierFactory.getChannelSupplier();
    }

    @Bean("NotifierGrpcChannelFactory")
    public HAGrpcChannelFactory haGrpcChannelFactory(@Qualifier("NotifierSupplier") ChannelSupplier channelSupplier,
                                                     @Autowired(required = false) TvmWrapper tvm) {
        if (tvm != null) {
            tvm.validateAlias(properties.getTvm().getDestinationAlias());
        }
        HAGrpcChannelFactory.Builder builder = HAGrpcChannelFactory.Builder.newBuilder();
        return builder.withPingProducer(
                channel -> FutureUtils.buildCompletableFuture(HealthGrpc
                        .newFutureStub(channel)
                        .check(HealthCheckRequest.newBuilder().build())
                ).thenApply((rsp) -> rsp.getStatus() == HealthCheckResponse.ServingStatus.SERVING ? ChannelState.READY :
                        ChannelState.NOT_READY))
                .withFailureDetectorProperties(properties.getFailureDetection())
                .withChannelSupplier(channelSupplier)
                .withChannelBuilder(label -> createChannel(label, getTvmHelper(tvm)))
                .build();
    }


    @SneakyThrows
    private LabeledChannel createChannel(String target, TvmHeaderClientHelper tvmHeaderClientHelper) {
        String clientFqdn = Objects.requireNonNull(InetAddress.getLocalHost().getCanonicalHostName());
        LoggingClientInterceptor loggingClientInterceptor = new LoggingClientInterceptor(
                clientFqdn, target, Set.of(HealthGrpc.getCheckMethod().getFullMethodName()), tracer
        );
        DefaultTimeoutClientInterceptor defaultTimeoutClientInterceptor = new DefaultTimeoutClientInterceptor(
                properties.getTimeout()
        );
        TracerClientInterceptor tracerClientInterceptor = new TracerClientInterceptor(
                tracer
        );
        List<ClientInterceptor> interceptors = new ArrayList<>(4);
        interceptors.addAll(Arrays.asList(loggingClientInterceptor, defaultTimeoutClientInterceptor,
                tracerClientInterceptor));
        if (tvmHeaderClientHelper != null) {
            interceptors.add(tvmHeaderClientHelper.getInterceptor(properties.getTvm().getDestinationAlias()));
        }
        return new LabeledChannel(target,
                ManagedChannelBuilder
                        .forTarget(target)
                        .intercept(interceptors)
                        .usePlaintext()
                        .maxInboundMessageSize(properties.getMaxMessageSize())
                        .build());
    }

    private TvmHeaderClientHelper getTvmHelper(TvmWrapper tvm) {
        if (properties.getTvm() != null && properties.getTvm().isEnabled() && tvm != null) {
            return new TvmHeaderClientHelper(tvm);
        } else {
            return null;
        }
    }
}
