package ru.yandex.travel.api.config.common;

import java.net.InetAddress;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.Set;

import io.grpc.Channel;
import io.grpc.ClientInterceptor;
import io.grpc.ManagedChannelBuilder;
import io.opentracing.Tracer;
import lombok.RequiredArgsConstructor;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import ru.yandex.travel.api.services.orders.OrchestratorAdminClientFactory;
import ru.yandex.travel.api.services.orders.OrchestratorClientFactory;
import ru.yandex.travel.api.services.orders.OrchestratorIdmClientFactory;
import ru.yandex.travel.api.services.orders.OrchestratorTakeoutClientFactory;
import ru.yandex.travel.commons.concurrent.FutureUtils;
import ru.yandex.travel.grpc.interceptors.DefaultTimeoutClientInterceptor;
import ru.yandex.travel.grpc.interceptors.LoggingClientInterceptor;
import ru.yandex.travel.grpc.interceptors.TracerClientInterceptor;
import ru.yandex.travel.grpc.interceptors.TvmHeaderClientHelper;
import ru.yandex.travel.grpc.interceptors.UserCredentialsClientInterceptor;
import ru.yandex.travel.orders.admin.proto.OrdersAdminInterfaceV1Grpc;
import ru.yandex.travel.orders.client.ChannelState;
import ru.yandex.travel.orders.client.ChannelSupplier;
import ru.yandex.travel.orders.client.GrpcChannelSupplierFactory;
import ru.yandex.travel.orders.client.HAGrpcChannelFactory;
import ru.yandex.travel.orders.client.LabeledChannel;
import ru.yandex.travel.orders.cpa.CpaOrderSnapshotsServiceGrpc;
import ru.yandex.travel.orders.idm.proto.OrdersIdmInterfaceV1Grpc;
import ru.yandex.travel.orders.proto.EServerState;
import ru.yandex.travel.orders.proto.HADiscoveryInterfaceV1Grpc;
import ru.yandex.travel.orders.proto.OrderInterfaceV1Grpc;
import ru.yandex.travel.orders.proto.OrderNoAuthInterfaceV1Grpc;
import ru.yandex.travel.orders.proto.PromoCodesUserInterfaceV1Grpc;
import ru.yandex.travel.orders.proto.TPingRpcReq;
import ru.yandex.travel.orders.takeout.proto.TakeoutOrdersInterfaceV1Grpc;
import ru.yandex.travel.tvm.TvmWrapper;

@Configuration
@RequiredArgsConstructor
@EnableConfigurationProperties(OrchestratorConfigurationProperties.class)
@Slf4j
public class OrchestratorConfiguration {
    private final OrchestratorConfigurationProperties properties;
    private final Tracer tracer;

    @Bean
    public OrchestratorClientFactory orchestratorClientFactory(@Qualifier("OrchestratorGrpcChannelFactory") HAGrpcChannelFactory haGrpcChannelFactory) {
        return new OrchestratorClientFactory() {
            @Override
            public OrderInterfaceV1Grpc.OrderInterfaceV1FutureStub createRoundRobinStub() {
                return OrderInterfaceV1Grpc.newFutureStub(haGrpcChannelFactory.getRoundRobinChannel());
            }

            @Override
            public OrderInterfaceV1Grpc.OrderInterfaceV1FutureStub createFutureStubForHappyPage() {
                return OrderInterfaceV1Grpc.newFutureStub(haGrpcChannelFactory.getRoundRobinChannel());
            }

            // TODO (mbobrov): remove in favor of api wide round-robin usage
            @Override
            public OrderInterfaceV1Grpc.OrderInterfaceV1FutureStub createFutureStubForTrains() {
                return OrderInterfaceV1Grpc.newFutureStub(haGrpcChannelFactory.getRoundRobinChannel());
            }

            @Override
            public OrderInterfaceV1Grpc.OrderInterfaceV1FutureStub createFutureStubForTrainsReadOnly() {
                return OrderInterfaceV1Grpc.newFutureStub(haGrpcChannelFactory.getRoundRobinSlavePreferredChannel());
            }

            @Override
            public OrderInterfaceV1Grpc.OrderInterfaceV1FutureStub createFutureStubForHotels() {
                return OrderInterfaceV1Grpc.newFutureStub(haGrpcChannelFactory.getRoundRobinChannel());
            }

            @Override
            public PromoCodesUserInterfaceV1Grpc.PromoCodesUserInterfaceV1FutureStub createRoundRobinStubPromoCodeStub() {
                return PromoCodesUserInterfaceV1Grpc.newFutureStub(haGrpcChannelFactory.getRoundRobinChannel());
            }

            @Override
            public OrderInterfaceV1Grpc.OrderInterfaceV1FutureStub createFutureStubForAvia() {
                return OrderInterfaceV1Grpc.newFutureStub(haGrpcChannelFactory.getRoundRobinChannel());
            }

            @Override
            public OrderInterfaceV1Grpc.OrderInterfaceV1FutureStub createFutureStubForAviaReadOnly() {
                return OrderInterfaceV1Grpc.newFutureStub(haGrpcChannelFactory.getRoundRobinSlavePreferredChannel());
            }

            @Override
            public OrderNoAuthInterfaceV1Grpc.OrderNoAuthInterfaceV1FutureStub createOrderNoAuthFutureStub() {
                return OrderNoAuthInterfaceV1Grpc.newFutureStub(haGrpcChannelFactory.getRoundRobinChannel());
            }

            @Override
            public CpaOrderSnapshotsServiceGrpc.CpaOrderSnapshotsServiceFutureStub createCpaOrderSnapshotFutureStub() {
                return CpaOrderSnapshotsServiceGrpc.newFutureStub(haGrpcChannelFactory.getRoundRobinChannel());
            }
        };
    }

    @Bean
    public OrchestratorAdminClientFactory orchestratorAdminClientFactory(@Qualifier("OrchestratorGrpcChannelFactory") HAGrpcChannelFactory haGrpcChannelFactory) {
        return requireMasterChannel -> {
            Channel channel = requireMasterChannel ? haGrpcChannelFactory.getMasterChannel() :
                    haGrpcChannelFactory.getFastestChannel();
            return OrdersAdminInterfaceV1Grpc.newFutureStub(channel);
        };
    }

    @Bean
    public OrchestratorTakeoutClientFactory orchestratorTakeoutClientFactory(@Qualifier(
            "OrchestratorGrpcChannelFactory") HAGrpcChannelFactory haGrpcChannelFactory) {
        return () -> {
            Channel channel = haGrpcChannelFactory.getFastestChannel();
            return TakeoutOrdersInterfaceV1Grpc.newFutureStub(channel);
        };
    }

    @Bean
    public OrchestratorIdmClientFactory orchestratorIdmClientFactory(@Qualifier("OrchestratorGrpcChannelFactory") HAGrpcChannelFactory haGrpcChannelFactory) {
        return requireMasterChannel -> {
            Channel channel = requireMasterChannel ? haGrpcChannelFactory.getMasterChannel() :
                    haGrpcChannelFactory.getFastestChannel();
            return OrdersIdmInterfaceV1Grpc.newFutureStub(channel);
        };
    }

    @Bean("OrchestratorSupplier")
    public ChannelSupplier getChannelSupplier() {
        GrpcChannelSupplierFactory grpcChannelSupplierFactory = new GrpcChannelSupplierFactory(properties);

        return grpcChannelSupplierFactory.getChannelSupplier();
    }

    @Bean("OrchestratorGrpcChannelFactory")
    public HAGrpcChannelFactory haGrpcChannelFactory(@Qualifier("OrchestratorSupplier") ChannelSupplier channelSupplier,
                                                     @Autowired(required = false) TvmWrapper tvm) {

        HAGrpcChannelFactory.Builder builder = HAGrpcChannelFactory.Builder.newBuilder();
        return builder.withPingProducer(
                channel -> FutureUtils.buildCompletableFuture(HADiscoveryInterfaceV1Grpc
                        .newFutureStub(channel)
                        .ping(TPingRpcReq.newBuilder().build())
                ).thenApply((rsp) -> rsp.getState() == EServerState.SS_MASTER ? ChannelState.READY_MASTER :
                        ChannelState.READY))
                .withFailureDetectorProperties(properties.getFailureDetection())
                .withChannelSupplier(channelSupplier)
                .withChannelBuilder(label -> this.createChannel(label, getTvmHelper(tvm)))
                .build();
    }

    @SneakyThrows
    private LabeledChannel createChannel(String target, TvmHeaderClientHelper tvmHeaderClientHelper) {
        String clientFqdn = Objects.requireNonNull(InetAddress.getLocalHost().getCanonicalHostName());
        LoggingClientInterceptor loggingClientInterceptor = new LoggingClientInterceptor(
                clientFqdn, target, Set.of(HADiscoveryInterfaceV1Grpc.getPingMethod().getFullMethodName()), tracer
        );
        DefaultTimeoutClientInterceptor defaultTimeoutClientInterceptor = new DefaultTimeoutClientInterceptor(
                properties.getTimeout()
        );
        TracerClientInterceptor tracerClientInterceptor = new TracerClientInterceptor(
                tracer
        );
        List<ClientInterceptor> interceptors = new ArrayList<>(4);
        interceptors.addAll(Arrays.asList(loggingClientInterceptor, defaultTimeoutClientInterceptor,
                tracerClientInterceptor));
        if (properties.isForwardCredentials()) {
            interceptors.add(new UserCredentialsClientInterceptor());
        }
        if (tvmHeaderClientHelper != null) {
            interceptors.add(tvmHeaderClientHelper.getInterceptor(properties.getTvm().getDestinationAlias()));
        }
        return new LabeledChannel(target,
                ManagedChannelBuilder
                        .forTarget(target)
                        .intercept(interceptors)
                        .usePlaintext()
                        .maxInboundMessageSize(properties.getAdmin().getMaxMessageSize())
                        .build());
    }

    private TvmHeaderClientHelper getTvmHelper(TvmWrapper tvm) {
        if (properties.getTvm() != null && properties.getTvm().isEnabled() && tvm != null) {
            return new TvmHeaderClientHelper(tvm);
        } else {
            return null;
        }
    }
}
