package ru.yandex.travel.api.config.common;

import java.util.List;

import com.fasterxml.jackson.databind.ObjectMapper;
import io.opentracing.Tracer;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.autoconfigure.web.servlet.DispatcherServletAutoConfiguration;
import org.springframework.boot.autoconfigure.web.servlet.WebMvcProperties;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Primary;
import org.springframework.web.servlet.DispatcherServlet;
import org.springframework.web.servlet.config.annotation.InterceptorRegistry;
import org.springframework.web.servlet.config.annotation.WebMvcConfigurer;

import ru.yandex.travel.http.ReqAnsLoggerInterceptor;
import ru.yandex.travel.http.ReqAnsLoggingDispatcherServlet;

@Configuration
@RequiredArgsConstructor
@Slf4j
@EnableConfigurationProperties(ReqAnsLoggingConfigurationProperties.class)
@ConditionalOnProperty("logging.reqans.enabled")
public class ReqAnsLoggingConfiguration implements WebMvcConfigurer {
    private final ObjectMapper objectMapper;
    private final Tracer tracer;
    private final ReqAnsLoggingConfigurationProperties properties;

    @Override
    public void addInterceptors(InterceptorRegistry registry) {
        registry.addInterceptor(
                new ReqAnsLoggerInterceptor(objectMapper, tracer,
                        List.of("X-Ya-Service-Ticket", "X-Ya-Session-Key", "X-Ya-User-Ticket"))
        )
                .addPathPatterns("/api/**")
                .addPathPatterns("/1/takeout/**")
                .excludePathPatterns(properties.getExcludedPathPatterns())
                .order(Integer.MIN_VALUE);
    }

    @Primary
    @Bean(name = DispatcherServletAutoConfiguration.DEFAULT_DISPATCHER_SERVLET_BEAN_NAME)
    public DispatcherServlet dispatcherServlet(WebMvcProperties webMvcProperties) {
        DispatcherServlet dispatcherServlet = new ReqAnsLoggingDispatcherServlet();
        dispatcherServlet.setDispatchOptionsRequest(
                webMvcProperties.isDispatchOptionsRequest());
        dispatcherServlet.setDispatchTraceRequest(
                webMvcProperties.isDispatchTraceRequest());
        dispatcherServlet.setThrowExceptionIfNoHandlerFound(
                webMvcProperties.isThrowExceptionIfNoHandlerFound());
        return dispatcherServlet;
    }
}
