package ru.yandex.travel.api.config.common;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import io.sentry.Sentry;
import io.sentry.event.Event;
import io.sentry.event.EventBuilder;
import io.sentry.event.interfaces.ExceptionInterface;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.core.Ordered;
import org.springframework.web.servlet.HandlerExceptionResolver;
import org.springframework.web.servlet.ModelAndView;

/**
 * Copy of original SentryExceptionResolver with custom fingerprinting
 */
@Slf4j
@RequiredArgsConstructor
public class SentryExceptionResolver implements HandlerExceptionResolver, Ordered {

    private final StackTraceFingerprinter stackTraceFingerprinter;

    @Override
    public ModelAndView resolveException(HttpServletRequest request,
                                         HttpServletResponse response,
                                         Object handler,
                                         Exception ex) {
        String fingerprint = null;
        try {
            fingerprint = stackTraceFingerprinter.exceptionFingerprint(ex);
        } catch (Exception e) {
            log.error("Error creating fingerprint: ", e);
        }

        EventBuilder eventBuilder = new EventBuilder().withMessage(ex.getMessage())
                .withLevel(Event.Level.ERROR)
                .withSentryInterface(new ExceptionInterface(ex));

        if (fingerprint != null) {
            eventBuilder = eventBuilder.withFingerprint(fingerprint);
        }
        Sentry.getStoredClient().sendEvent(eventBuilder);

        // null = run other HandlerExceptionResolvers to actually handle the exception
        return null;
    }

    @Override
    public int getOrder() {
        // ensure this resolver runs first so that all exceptions are reported
        return Integer.MIN_VALUE;
    }
}
