package ru.yandex.travel.api.config.common;


import java.nio.charset.StandardCharsets;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableList;
import io.grpc.StatusRuntimeException;
import org.apache.commons.codec.binary.Hex;

import ru.yandex.travel.commons.grpc.ServerUtils;
import ru.yandex.travel.commons.proto.TError;


public final class StackTraceFingerprinter {
    private static final String MESSAGE_DIGEST_ALGORITHM = "MD5";
    private final ImmutableList<String> packagesToKeep;

    public StackTraceFingerprinter(ImmutableList<String> packagesToKeep) {
        this.packagesToKeep = packagesToKeep;
    }


    public String exceptionFingerprint(Throwable exception) {
        Preconditions.checkNotNull(exception, "Exception must be present");
        MessageDigest md;
        try {
            md = MessageDigest.getInstance(MESSAGE_DIGEST_ALGORITHM);
            md.reset();
        } catch (NoSuchAlgorithmException e) {
            throw new RuntimeException(e);
        }
        digestThrowable(md, exception);
        return Hex.encodeHexString(md.digest());
    }

    private void digestThrowable(MessageDigest md, Throwable throwable) {
        if (throwable.getCause() != null) {
            digestThrowable(md, throwable.getCause());
        }
        if (throwable instanceof StatusRuntimeException) {
            StatusRuntimeException statusRuntimeException = (StatusRuntimeException) throwable;
            digestStatusRuntimeException(md, statusRuntimeException);
        } else {
            digestStackTrace(md, throwable.getStackTrace());
        }
    }

    private void digestStatusRuntimeException(MessageDigest md, StatusRuntimeException statusRuntimeException) {
        if (statusRuntimeException.getStatus() != null) {
            StringBuilder key = new StringBuilder();
            key.append("status.code")
                    .append(statusRuntimeException.getStatus().getCode())
                    .append("|")
                    .append("status.description")
                    .append(statusRuntimeException.getStatus().getDescription())
                    .append("|");
            md.update(key.toString().getBytes(StandardCharsets.UTF_8));
        }
        if (statusRuntimeException.getTrailers() != null
                && statusRuntimeException.getTrailers().containsKey(ServerUtils.METADATA_ERROR_KEY)) {
            TError grpcError = statusRuntimeException.getTrailers().get(ServerUtils.METADATA_ERROR_KEY);
            if (grpcError != null) {
                StringBuilder trailerKey = new StringBuilder();
                trailerKey.append("trailerErrorCode").append(grpcError.getCode()).append("|")
                        .append("trailerErrorMessage").append(grpcError.getMessage()).append("|");
                md.update(trailerKey.toString().getBytes(StandardCharsets.UTF_8));
            }
        }
    }

    private void digestStackTrace(MessageDigest md, StackTraceElement[] stElements) {
        for (StackTraceElement ste : stElements) {
            if (ste == null) {
                continue;
            }

            if (keepClass(ste)) {
                StringBuilder key = new StringBuilder();

                key.append(ste.getClassName());
                key.append(".");
                key.append(Strings.nullToEmpty(ste.getMethodName()));
                key.append(":");
                key.append(ste.getLineNumber());

                md.update(key.toString().getBytes(StandardCharsets.UTF_8));
            }
        }
    }

    private boolean keepClass(StackTraceElement ste) {
        for (String prefix : packagesToKeep) {
            if (ste.getClassName().startsWith(prefix)) {
                return true;
            }
        }
        return false;
    }
}

