package ru.yandex.travel.api.config.common;

import java.lang.reflect.Type;
import java.math.BigDecimal;
import java.math.BigInteger;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import com.fasterxml.classmate.TypeResolver;
import org.javamoney.moneta.Money;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.Ordered;
import org.springframework.http.ResponseEntity;
import org.springframework.util.ClassUtils;
import org.springframework.web.context.request.async.DeferredResult;
import springfox.documentation.builders.AlternateTypeBuilder;
import springfox.documentation.builders.AlternateTypePropertyBuilder;
import springfox.documentation.builders.ParameterBuilder;
import springfox.documentation.builders.RequestHandlerSelectors;
import springfox.documentation.schema.AlternateTypeRule;
import springfox.documentation.schema.AlternateTypeRuleConvention;
import springfox.documentation.schema.ModelRef;
import springfox.documentation.schema.WildcardType;
import springfox.documentation.service.ApiInfo;
import springfox.documentation.service.Parameter;
import springfox.documentation.spi.DocumentationType;
import springfox.documentation.spring.web.plugins.Docket;
import springfox.documentation.swagger2.annotations.EnableSwagger2;

import ru.yandex.travel.api.models.hotels.BoundingBox;
import ru.yandex.travel.api.models.hotels.Coordinates;
import ru.yandex.travel.commons.http.CommonHttpHeaders;
import ru.yandex.travel.hotels.common.Permalink;

import static springfox.documentation.schema.AlternateTypeRules.newRule;

@Configuration
@EnableSwagger2
public class SwaggerConfiguration {

    private static final String API_ADMIN = "admin";
    private static final String API_EXTRANET = "hotels_extranet";
    private static final String API_TEST_CONTEXT = "test_context";
    private static final String API_WORKFLOW_MANAGEMENT = "workflow_management";

    private static final Set<String> KNOWN_PATHES = Set.of(
            API_EXTRANET, API_ADMIN, API_TEST_CONTEXT,
            API_WORKFLOW_MANAGEMENT);

    @Autowired
    private TypeResolver typeResolver;

    @Autowired(required = false)
    private WebConfiguration.MainHttpTvmProperties mainHttpTvmConfig;

    @Autowired(required = false)
    private WebConfiguration.AdminHttpTvmProperties adminHttpTvmConfig;

    @Autowired(required = false)
    private WebConfiguration.IdmHttpTvmProperties idmHttpTvmConfig;

    @Autowired(required = false)
    private WebConfiguration.CpaExportHttpTvmProperties cpaExportHttpTvmProperties;

    @Bean
    public AlternateTypeRuleConvention customConventions(final TypeResolver resolver) {
        return new AlternateTypeRuleConvention() {

            @Override
            public int getOrder() {
                return Ordered.HIGHEST_PRECEDENCE;
            }

            @Override
            public List<AlternateTypeRule> rules() {
                return List.of(
                        newRule(resolver.resolve(Money.class), resolver.resolve(moneyMixin())),
                        newRule(resolver.resolve(Permalink.class), resolver.resolve(BigInteger.class)),
                        newRule(resolver.resolve(BoundingBox.class), resolver.resolve(String.class)),
                        newRule(resolver.resolve(Coordinates.class), resolver.resolve(String.class))
                );
            }
        };
    }

    private Type moneyMixin() {
        return new AlternateTypeBuilder()
                .fullyQualifiedClassName(
                        String.format("%s.generated.%s",
                                Money.class.getPackage().getName(),
                                Money.class.getSimpleName()))
                .withProperties(List.of(
                        property(BigDecimal.class, "value"),
                        property(String.class, "currency")
                )).build();
    }

    private AlternateTypePropertyBuilder property(Class<?> type, String name) {
        return new AlternateTypePropertyBuilder()
                .withName(name)
                .withType(type)
                .withCanRead(true)
                .withCanWrite(true);
    }


    @Bean
    public Docket productApi() {


        Set<String> packagePrefixes = Set.of("ru.yandex.travel.api.endpoints");

        List<Parameter> globalParams = new ArrayList<>();
        globalParams.addAll(createCommonHttpHeaders(mainHttpTvmConfig != null && mainHttpTvmConfig.isEnabled()));


        return new Docket(DocumentationType.SWAGGER_2)
                .select()
                .apis(input -> Optional.ofNullable(input.declaringClass())
                        .map(clazz ->
                                packagePrefixes.stream().anyMatch(prefix ->
                                        ClassUtils.getPackageName(clazz).startsWith(prefix)))
                        .orElse(false)
                )
                .paths(selector -> KNOWN_PATHES.stream().noneMatch(selector::contains))
                .build()
                .genericModelSubstitutes(ResponseEntity.class)
                .alternateTypeRules(
                        newRule(typeResolver.resolve(DeferredResult.class,
                                        typeResolver.resolve(ResponseEntity.class, WildcardType.class)),
                                typeResolver.resolve(WildcardType.class)),
                        newRule(typeResolver.resolve(DeferredResult.class, WildcardType.class),
                                typeResolver.resolve(WildcardType.class)))
                .groupName("1. default")
                .apiInfo(new ApiInfo(
                        "Поиск Отелей На Яндексе",
                        "API портала и сервиса бронирования", null, null, null, null, null,
                        Collections.emptyList()))
                .globalOperationParameters(globalParams);

    }

    @Bean
    public Docket adminApi() {
        List<Parameter> globalParams = new ArrayList<>();
        globalParams.addAll(createCommonHttpHeaders(adminHttpTvmConfig != null && adminHttpTvmConfig.isEnabled()));

        return new Docket(DocumentationType.SWAGGER_2)
                .select()
                .apis(input -> Optional.ofNullable(input.declaringClass())
                        .map(clazz ->
                                ClassUtils.getPackageName(clazz).startsWith("ru.yandex.travel.api"))
                        .orElse(false))
                .paths(selector -> selector.contains(API_ADMIN))
                .build()
                .groupName("2. admin")
                .apiInfo(new ApiInfo(
                        "Поиск Отелей На Яндексе",
                        "Административные ручки", null, null, null, null, null,
                        Collections.emptyList()))
                .globalOperationParameters(globalParams);

    }


    @Bean
    @ConditionalOnProperty("test-context.enabled")
    public Docket testContextApi() {
        return new Docket(DocumentationType.SWAGGER_2)
                .select()
                .apis(input -> Optional.ofNullable(input.declaringClass())
                        .map(clazz ->
                                ClassUtils.getPackageName(clazz).startsWith("ru.yandex.travel.api"))
                        .orElse(false))
                .paths(selector -> selector.contains(API_TEST_CONTEXT))
                .build()
                .groupName("4. test")
                .apiInfo(new ApiInfo(
                        "Бронирование Отелей На Яндексе",
                        "Инструменты тестирования", null, null, null, null, null,
                        Collections.emptyList()));
    }

    @Bean
    public Docket cpaExportApi() {
        List<Parameter> globalParams = new ArrayList<>();
        globalParams.addAll(createCommonHttpHeaders(cpaExportHttpTvmProperties != null && cpaExportHttpTvmProperties.isEnabled()));

        return new Docket(DocumentationType.SWAGGER_2)
                .select()
                .apis(RequestHandlerSelectors.basePackage("ru.yandex.travel.api.endpoints"))
                .paths(selector -> selector.contains("cpa_export"))
                .build()
                .groupName("3. cpa export")
                .apiInfo(new ApiInfo(
                        "Бронирование путешествий на Яндексе",
                        "Экспорт данных для CPA", null, null, null, null, null,
                        Collections.emptyList()))
                .globalOperationParameters(globalParams);
    }

    @Bean
    public Docket workflowManagementApi() {

        return new Docket(DocumentationType.SWAGGER_2)
                .select()
                .apis(input -> Optional.ofNullable(input.declaringClass())
                        .map(clazz ->
                                ClassUtils.getPackageName(clazz).startsWith("ru.yandex.travel.api"))
                        .orElse(false))
                .paths(selector -> selector.contains(API_WORKFLOW_MANAGEMENT))
                .build()
                .groupName("5. workflow_management")
                .apiInfo(new ApiInfo(
                        "Бронирование путешествий на Яндексе",
                        "Ручки управления workflow", null, null, null, null, null,
                        Collections.emptyList()))
                .globalOperationParameters(
                        List.of(
                                new ParameterBuilder()
                                        .name("Authorization")
                                        .description("OAuth Token")
                                        .modelRef(new ModelRef("string"))
                                        .parameterType("header")
                                        .required(true)
                                        .build()
                        )
                );

    }

    @Bean
    public Docket extranetApi() {
        return new Docket(DocumentationType.SWAGGER_2)
                .select()
                .apis(input -> Optional.ofNullable(input.declaringClass())
                        .map(clazz ->
                                ClassUtils.getPackageName(clazz).startsWith("ru.yandex.travel.api.endpoints" +
                                        ".hotels_extranet"))
                        .orElse(false)
                )
                .paths(selector -> selector.contains(API_EXTRANET))
                .build()
                .groupName("6.extranet")
                .apiInfo(new ApiInfo(
                        "Экстранет отельера",
                        "API для сервиса Экстранет", null, null, null, null, null,
                        Collections.emptyList()))
                .globalOperationParameters(createCommonHttpHeaders(mainHttpTvmConfig != null && mainHttpTvmConfig.isEnabled()));

    }

    @Bean
    public Docket idmApi() {
        List<Parameter> parameters = null;
        if (idmHttpTvmConfig != null && idmHttpTvmConfig.isEnabled()) {
            var st = CommonHttpHeaders.HeaderType.SERVICE_TICKET;
            parameters = List.of(new ParameterBuilder()
                    .name(st.getHeader())
                    .description(st.getDescription())
                    .defaultValue(st.getDefaultSwaggerValue())
                    .modelRef(new ModelRef("string"))
                    .parameterType("header")
                    .required(st.isRequired())
                    .build());
        }
        return new Docket(DocumentationType.SWAGGER_2)
                .select()
                .apis(input -> Optional.ofNullable(input.declaringClass())
                        .map(clazz ->
                                ClassUtils.getPackageName(clazz).startsWith("ru.yandex.travel.api.endpoints.idm"))
                        .orElse(false)
                )
                .build()
                .groupName("7.IDM")
                .apiInfo(new ApiInfo(
                        "IDM",
                        "API IDM", null, null, null, null, null,
                        Collections.emptyList()))
                .globalOperationParameters(parameters);

    }

    private List<Parameter> createCommonHttpHeaders(boolean enableServiceTicket) {
        return Arrays.stream(CommonHttpHeaders.HeaderType.values())
                .filter(ht -> enableServiceTicket || ht != CommonHttpHeaders.HeaderType.SERVICE_TICKET)
                .map(ht -> new ParameterBuilder()
                        .name(ht.getHeader())
                        .description(ht.getDescription())
                        .defaultValue(ht.getDefaultSwaggerValue())
                        .modelRef(new ModelRef("string"))
                        .parameterType("header")
                        .required(ht.isRequired())
                        .build()
                ).collect(Collectors.toList());
    }
}
