package ru.yandex.travel.api.config.common;

import lombok.RequiredArgsConstructor;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.Primary;

import ru.yandex.travel.credentials.UserCredentialsAuthValidator;
import ru.yandex.travel.credentials.UserCredentialsAuthValidatorStubImpl;
import ru.yandex.travel.credentials.UserCredentialsAuthValidatorTvmImpl;
import ru.yandex.travel.credentials.UserCredentialsBuilder;
import ru.yandex.travel.credentials.UserCredentialsPassportExtractor;
import ru.yandex.travel.credentials.UserCredentialsPassportExtractorStubImpl;
import ru.yandex.travel.credentials.UserCredentialsPassportExtractorTvmImpl;
import ru.yandex.travel.credentials.UserCredentialsProperties;
import ru.yandex.travel.credentials.UserCredentialsServletFilter;
import ru.yandex.travel.credentials.UserCredentialsServletFilterRegistration;
import ru.yandex.travel.credentials.UserCredentialsValidator;
import ru.yandex.travel.grpc.interceptors.UserCredentialsServerInterceptor;
import ru.yandex.travel.tvm.TvmHelper;
import ru.yandex.travel.tvm.TvmWrapper;

@Configuration
@EnableConfigurationProperties({TvmServiceProperties.class, TvmUserInternalProperties.class, TvmUserExternalProperties.class, UserCredentialsProperties.class})
@RequiredArgsConstructor
public class TvmConfiguration {

    private final TvmServiceProperties tvmServiceProperties;
    private final TvmUserInternalProperties tvmUserInternalProperties;
    private final TvmUserExternalProperties tvmUserExternalProperties;
    private final UserCredentialsProperties userCredentialsProperties;

    @Bean
    @Primary
    @ConditionalOnProperty(value = "tvm-service.enabled")
    public TvmWrapper tvmServiceWrapper() {
        return TvmHelper.getTvmWrapper(tvmServiceProperties);
    }

    @Bean
    @ConditionalOnProperty(value = "tvm-internal.enabled")
    public TvmWrapper tvmUserInternal() {
        return TvmHelper.getTvmWrapper(tvmUserInternalProperties);
    }

    @Bean
    @ConditionalOnProperty(value = "tvm-external.enabled")
    public TvmWrapper tvmUserExternal() {
        return TvmHelper.getTvmWrapper(tvmUserExternalProperties);
    }

    @Bean
    @ConditionalOnProperty("credentials.auth-enabled")
    public UserCredentialsPassportExtractor userCredentialsExternalPassportExtractor(@Qualifier("tvmUserExternal") TvmWrapper tvm) {
        return new UserCredentialsPassportExtractorTvmImpl(tvm);
    }

    @Bean
    @ConditionalOnProperty("credentials.auth-enabled")
    public UserCredentialsPassportExtractor userCredentialsInternalPassportExtractor(@Qualifier("tvmUserInternal") TvmWrapper tvm) {
        return new UserCredentialsPassportExtractorTvmImpl(tvm);
    }

    @Bean(name = {"userCredentialsInternalPassportExtractor", "userCredentialsExternalPassportExtractor"})
    @ConditionalOnProperty(value = "credentials.auth-enabled", havingValue = "false")
    public UserCredentialsPassportExtractor userCredentialsStubPassportExtractor() {
        return new UserCredentialsPassportExtractorStubImpl();
    }

    @Bean
    @ConditionalOnProperty("credentials.auth-enabled")
    public UserCredentialsAuthValidator userCredentialsInternalAuthValidator(@Qualifier("tvmUserInternal") TvmWrapper tvm) {
        return new UserCredentialsAuthValidatorTvmImpl(tvm);
    }

    @Bean
    @ConditionalOnProperty("credentials.auth-enabled")
    public UserCredentialsAuthValidator userCredentialsExternalAuthValidator(@Qualifier("tvmUserExternal") TvmWrapper tvm) {
        return new UserCredentialsAuthValidatorTvmImpl(tvm);
    }

    @Bean(name = {"userCredentialsInternalAuthValidator", "userCredentialsExternalAuthValidator"})
    @ConditionalOnProperty(value = "credentials.auth-enabled", havingValue = "false")
    public UserCredentialsAuthValidator userCredentialsStubAuthValidator() {
        return new UserCredentialsAuthValidatorStubImpl();
    }

    @Bean
    public UserCredentialsServletFilterRegistration userCredentialsServletFilter(
            UserCredentialsBuilder userCredentialsBuilder,
            @Qualifier("userCredentialsExternalPassportExtractor") UserCredentialsPassportExtractor userCredentialsPassportExtractor
    ) {
        // we use a single UserCredentialsPassportExtractor with only external tickets support here
        // because it shouldn't be needed for internal calls from the admin app
        // (the admin app always explicitly sends the 'X-Ya-PassportId' header, no extraction from the ticket is needed)
        var registrationBean = new UserCredentialsServletFilterRegistration();
        registrationBean.setFilter(new UserCredentialsServletFilter(userCredentialsBuilder, userCredentialsPassportExtractor));
        registrationBean.setOrder(1000);// should be > order of CommonHttpHeadersServletFilterRegistration
        registrationBean.addUrlPatterns("/api/*");
        registrationBean.addUrlPatterns("/1/takeout/*");
        registrationBean.setName("userCredentialsFilter");
        return registrationBean;
    }

    @Bean
    @ConditionalOnProperty(value = "credentials.enabled")
    public UserCredentialsServerInterceptor userCredentialsInternalServerInterceptor(
            UserCredentialsBuilder userCredentialsBuilder,
            @Qualifier("userCredentialsInternalPassportExtractor") UserCredentialsPassportExtractor userCredentialsPassportExtractor,
            @Qualifier("userCredentialsInternalAuthValidator") UserCredentialsAuthValidator userCredentialsAuthValidator
    ) {
        return new UserCredentialsServerInterceptor(userCredentialsBuilder, userCredentialsPassportExtractor,
                new UserCredentialsValidator(userCredentialsAuthValidator));
    }

    @Bean
    @ConditionalOnProperty(value = "credentials.enabled")
    public UserCredentialsServerInterceptor userCredentialsExternalServerInterceptor(
            UserCredentialsBuilder userCredentialsBuilder,
            @Qualifier("userCredentialsExternalPassportExtractor") UserCredentialsPassportExtractor userCredentialsPassportExtractor,
            @Qualifier("userCredentialsExternalAuthValidator") UserCredentialsAuthValidator userCredentialsAuthValidator
    ) {
        return new UserCredentialsServerInterceptor(userCredentialsBuilder, userCredentialsPassportExtractor,
                new UserCredentialsValidator(userCredentialsAuthValidator));
    }
}
