package ru.yandex.travel.api.config.common;

import java.io.IOException;
import java.time.Instant;
import java.util.HashMap;
import java.util.Map;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

import com.fasterxml.jackson.databind.ObjectMapper;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.web.servlet.error.DefaultErrorAttributes;
import org.springframework.core.Ordered;
import org.springframework.http.MediaType;
import org.springframework.web.context.request.ServletWebRequest;
import org.springframework.web.servlet.HandlerExceptionResolver;
import org.springframework.web.servlet.ModelAndView;

@Slf4j
@RequiredArgsConstructor
public class UnhandledExceptionsResolver implements HandlerExceptionResolver, Ordered {

    private final ObjectMapper objectMapper;

    @Override
    public int getOrder() {
        return Integer.MAX_VALUE; // we ensure that it's run after all exception resolver
    }

    @Override
    public ModelAndView resolveException(HttpServletRequest request, HttpServletResponse response, Object handler,
                                         Exception ex) {
        log.error("Exception occured during {}", request.getRequestURI(), ex);

        var errorAttrs = new DefaultErrorAttributes();
        ServletWebRequest webRequest = new ServletWebRequest(request);
        response.setContentType(MediaType.APPLICATION_JSON.toString());
        Map<String, Object> errorAttributes = null;
        if (errorAttrs.getError(webRequest) != null) {
            errorAttributes = errorAttrs.getErrorAttributes(webRequest, false);
        } else {
            errorAttributes = new HashMap<>();
            errorAttributes.put("timestamp", Instant.now());
            errorAttributes.put("error", ex.getMessage());
        }

        String errorMessage = null;
        try {
            errorMessage = objectMapper.writeValueAsString(errorAttributes);
        } catch (Exception e) {
            log.error("Error printing errorDetails", e);
            errorMessage = String.format("Error occured printing details for ex: %s", ex.getMessage());
        }

        try {
            response.sendError(HttpServletResponse.SC_INTERNAL_SERVER_ERROR, errorMessage);
            return new ModelAndView();
        } catch (IOException e) {
            log.error("Exception sending error to response", e);
        }
        return null;
    }
}
