package ru.yandex.travel.api.config.common;

import java.util.Collections;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.protobuf.util.JsonFormat;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.autoconfigure.condition.ConditionalOnProperty;
import org.springframework.boot.context.properties.ConfigurationProperties;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.core.annotation.Order;
import org.springframework.http.converter.protobuf.ProtobufJsonFormatHttpMessageConverter;
import org.springframework.web.servlet.config.annotation.InterceptorRegistry;
import org.springframework.web.servlet.config.annotation.WebMvcConfigurer;

import ru.yandex.travel.PropertyUtils;
import ru.yandex.travel.credentials.UserCredentialsAuthValidator;
import ru.yandex.travel.credentials.UserCredentialsValidationHandlerInterceptor;
import ru.yandex.travel.credentials.UserCredentialsValidator;
import ru.yandex.travel.credentials.UserLoggedInHandlerInterceptor;
import ru.yandex.travel.http.BaseHttpTvmProperties;
import ru.yandex.travel.http.CommonHttpHeadersServletFilter;
import ru.yandex.travel.http.CommonHttpHeadersServletFilterRegistration;
import ru.yandex.travel.http.TvmAuthenticationInterceptor;
import ru.yandex.travel.tvm.TvmWrapper;

@Configuration
@RequiredArgsConstructor
@EnableConfigurationProperties({WebConfiguration.MainHttpTvmProperties.class,
        WebConfiguration.AdminHttpTvmProperties.class,
        WebConfiguration.CpaExportHttpTvmProperties.class,
        WebConfiguration.TakeoutHttpTvmProperties.class,
        WebConfiguration.AdministratorHttpTvmProperties.class,
        WebConfiguration.IdmHttpTvmProperties.class
})
public class WebConfiguration {

    @Configuration
    @RequiredArgsConstructor
    @ConditionalOnProperty("http-tvm-main.enabled")
    @Slf4j
    public static class TvmMainInterceptorConfiguration implements WebMvcConfigurer {
        private final MainHttpTvmProperties tvmProperties;
        private final TvmWrapper tvm2;

        @Override
        public void addInterceptors(InterceptorRegistry registry) {
            TvmAuthenticationInterceptor interceptor = new TvmAuthenticationInterceptor(tvm2,
                    PropertyUtils.stringToListOfStrings(tvmProperties.getAllowedConsumers()));
            log.info("Creating HTTP tvm interceptor for primary API calls");
            registry.addInterceptor(interceptor).addPathPatterns("/api/**")
                    .excludePathPatterns("/api/idm/**")
                    .excludePathPatterns("/api/administrator/**")
                    .excludePathPatterns("/api/admin/**")
                    .excludePathPatterns("/api/avia_country_restrictions/**")
                    .excludePathPatterns("/api/travel-orders-admin-idm/**")
                    .excludePathPatterns("/api/booking_flow/v1/admin/**")
                    .excludePathPatterns("/api/travel_orders_trust_callback/**")
                    .excludePathPatterns("/api/cpa_export/**")
                    .excludePathPatterns("/api/test_context/**")
                    .excludePathPatterns("/api/takeout/**")
                    .excludePathPatterns("/api/workflow_management/**");
        }
    }

    @Configuration
    @RequiredArgsConstructor
    @ConditionalOnProperty("http-tvm-admin.enabled")
    @Slf4j
    public static class TvmAdminInterceptorConfiguration implements WebMvcConfigurer {
        private final AdminHttpTvmProperties tvmProperties;
        private final TvmWrapper tvm2;

        @Override
        public void addInterceptors(InterceptorRegistry registry) {
            TvmAuthenticationInterceptor interceptor = new TvmAuthenticationInterceptor(tvm2,
                    PropertyUtils.stringToListOfStrings(tvmProperties.getAllowedConsumers()));
            log.info("Creating HTTP tvm interceptor for administrative API calls");
            registry.addInterceptor(interceptor)
                    .addPathPatterns("/api/admin/**")
                    .addPathPatterns("/api/travel-orders-admin-idm/**")
                    .excludePathPatterns("/api/booking_flow/v1/admin/**")
                    .excludePathPatterns("/api/admin/v1/get_order_no_auth")
                    .excludePathPatterns("/api/admin/v1/get_order_for_business_trip_doc")
            ;
        }
    }

    @Configuration
    @RequiredArgsConstructor
    @ConditionalOnProperty("http-tvm-idm.enabled")
    @Slf4j
    public static class TvmIdmInterceptorConfiguration implements WebMvcConfigurer {
        private final IdmHttpTvmProperties tvmProperties;
        private final TvmWrapper tvm2;

        @Override
        public void addInterceptors(InterceptorRegistry registry) {
            TvmAuthenticationInterceptor interceptor = new TvmAuthenticationInterceptor(tvm2,
                    PropertyUtils.stringToListOfStrings(tvmProperties.getAllowedConsumers()));
            log.info("Creating HTTP tvm interceptor for IDM API calls");
            registry.addInterceptor(interceptor)
                    .addPathPatterns("/api/idm/**");
        }
    }

    @Configuration
    @RequiredArgsConstructor
    @ConditionalOnProperty("http-tvm-takeout.enabled")
    @Slf4j
    public static class TvmTakeoutInterceptorConfiguration implements WebMvcConfigurer {
        private final TakeoutHttpTvmProperties tvmProperties;
        private final TvmWrapper tvm2;

        @Override
        public void addInterceptors(InterceptorRegistry registry) {
            TvmAuthenticationInterceptor interceptor = new TvmAuthenticationInterceptor(tvm2,
                    PropertyUtils.stringToListOfStrings(tvmProperties.getAllowedConsumers()));
            log.info("Creating HTTP tvm interceptor for takeout API calls");
            registry.addInterceptor(interceptor)
                    .addPathPatterns("/api/takeout/**")
                    .addPathPatterns("/1/takeout/**")
            ;
        }
    }

    @Configuration
    @RequiredArgsConstructor
    @ConditionalOnProperty("http-tvm-cpa-export.enabled")
    @Slf4j
    public static class TvmCpaExportInterceptorConfiguration implements WebMvcConfigurer {
        private final CpaExportHttpTvmProperties tvmProperties;
        private final TvmWrapper tvm2;

        @Override
        public void addInterceptors(InterceptorRegistry registry) {
            TvmAuthenticationInterceptor interceptor = new TvmAuthenticationInterceptor(tvm2,
                    PropertyUtils.stringToListOfStrings(tvmProperties.getAllowedConsumers()));
            log.info("Creating HTTP tvm interceptor for cpa export API calls");
            registry.addInterceptor(interceptor).addPathPatterns("/api/cpa_export/**");
        }
    }

    @Configuration
    @RequiredArgsConstructor
    @ConditionalOnProperty("http-tvm-main.enabled")
    @Slf4j
    public static class OrchestratorCallbacksInterceptorConfiguration implements WebMvcConfigurer {
        private final OrchestratorConfigurationProperties orchestratorConfigurationProperties;
        private final TvmWrapper tvm2;

        @Override
        public void addInterceptors(InterceptorRegistry registry) {
            log.info("Creating OrchestratorHttpCallbacks tvm interceptor");
            String tvmAlias = orchestratorConfigurationProperties.getTvm().getDestinationAlias();
            registry.addInterceptor(new TvmAuthenticationInterceptor(tvm2, Collections.singletonList(tvmAlias)))
                    .addPathPatterns("/api/avia_booking_flow/v1/variants/refresh");
        }
    }

    @Configuration
    @RequiredArgsConstructor
    @ConditionalOnProperty("http-tvm-administrator.enabled")
    @Slf4j
    public static class TvmAdministratorInterceptorConfiguration implements WebMvcConfigurer {
        private final AdministratorHttpTvmProperties tvmProperties;
        private final TvmWrapper tvm2;

        @Override
        public void addInterceptors(InterceptorRegistry registry) {
            TvmAuthenticationInterceptor interceptor = new TvmAuthenticationInterceptor(tvm2,
                    PropertyUtils.stringToListOfStrings(tvmProperties.getAllowedConsumers()));
            log.info("Creating HTTP tvm interceptor for API calls to Hotels Administrator");
            registry.addInterceptor(interceptor)
                    .addPathPatterns("/api/administrator/**");
        }
    }

    @Configuration
    @RequiredArgsConstructor
    @Order(1000) // it must be used after the user credentials mvc handler
    public static class UserCredentialsLoggedInInterceptorConfiguration implements WebMvcConfigurer {
        private final UserLoggedInHandlerInterceptor userLoggedInHandlerInterceptor;

        @Override
        public void addInterceptors(InterceptorRegistry registry) {
            registry.addInterceptor(userLoggedInHandlerInterceptor)
                    .addPathPatterns("/api/orders/v2/list_orders")
                    .addPathPatterns("/api/orders/v2/list_orders_next_page")
                    .addPathPatterns("/api/trips/v1/**")
                    .excludePathPatterns("/api/trips/v1/get_trip")
                    .addPathPatterns("/api/trips/orders/v1/**")
                    .excludePathPatterns("/api/trips/orders/v1/get_order")
                    .addPathPatterns("/api/admin/**")
                    .excludePathPatterns("/api/admin/v1/get_order_no_auth")
                    .excludePathPatterns("/api/admin/v1/get_order_for_business_trip_doc")
            ;
        }
    }

    @Bean
    @Order(500)
    public UserCredentialsValidationHandlerInterceptor userCredentialsExternalMvcHandlerInterceptor(
            @Qualifier("userCredentialsExternalAuthValidator") UserCredentialsAuthValidator userCredentialsAuthValidator
    ) {
        return new UserCredentialsValidationHandlerInterceptor(new UserCredentialsValidator(userCredentialsAuthValidator));
    }

    @Bean
    @Order(500)
    public UserCredentialsValidationHandlerInterceptor userCredentialsExternalWeakMvcHandlerInterceptor(
            @Qualifier("userCredentialsExternalAuthValidator") UserCredentialsAuthValidator userCredentialsAuthValidator
    ) {
        return new UserCredentialsValidationHandlerInterceptor(new UserCredentialsValidator(userCredentialsAuthValidator, false));
    }

    @Bean
    @Order(500)
    public UserCredentialsValidationHandlerInterceptor userCredentialsInternalMvcHandlerInterceptor(
            @Qualifier("userCredentialsInternalAuthValidator") UserCredentialsAuthValidator userCredentialsAuthValidator
    ) {
        return new UserCredentialsValidationHandlerInterceptor(new UserCredentialsValidator(userCredentialsAuthValidator));
    }

    @Configuration
    @Order(500)
    public static class UserCredentialsTvmAuthInterceptorConfiguration implements WebMvcConfigurer {
        private final UserCredentialsValidationHandlerInterceptor userCredentialsExternalMvcHandlerInterceptor;
        private final UserCredentialsValidationHandlerInterceptor userCredentialsExternalWeakMvcHandlerInterceptor;
        private final UserCredentialsValidationHandlerInterceptor userCredentialsInternalMvcHandlerInterceptor;

        public UserCredentialsTvmAuthInterceptorConfiguration(
                @Qualifier("userCredentialsExternalMvcHandlerInterceptor") UserCredentialsValidationHandlerInterceptor externalInterceptor,
                @Qualifier("userCredentialsExternalWeakMvcHandlerInterceptor") UserCredentialsValidationHandlerInterceptor externalWeakInterceptor,
                @Qualifier("userCredentialsInternalMvcHandlerInterceptor") UserCredentialsValidationHandlerInterceptor internalInterceptor
        ) {
            this.userCredentialsExternalMvcHandlerInterceptor = externalInterceptor;
            this.userCredentialsExternalWeakMvcHandlerInterceptor = externalWeakInterceptor;
            this.userCredentialsInternalMvcHandlerInterceptor = internalInterceptor;
        }

        @Override
        public void addInterceptors(InterceptorRegistry registry) {
            registry.addInterceptor(userCredentialsExternalMvcHandlerInterceptor)
                    .addPathPatterns("/api/avia_booking/**")
                    .addPathPatterns("/api/avia_booking_flow/**")
                    .addPathPatterns("/api/booking_flow/**")
                    .addPathPatterns("/api/hotels_booking_flow/**")
                    .addPathPatterns("/api/orders/**")
                    .addPathPatterns("/api/trains_booking_flow/**")
                    .addPathPatterns("/api/promo_codes/**")
                    .excludePathPatterns("/api/avia_booking_flow/v1/variants/**")
                    .excludePathPatterns("/api/booking_flow/v1/admin/**")
                    .addPathPatterns("/api/admin/v1/train_download_blank")
            ;
            registry.addInterceptor(userCredentialsExternalWeakMvcHandlerInterceptor)
                    .addPathPatterns("/api/hotels_portal/**");
            registry.addInterceptor(userCredentialsInternalMvcHandlerInterceptor)
                    .addPathPatterns("/api/admin/**")
                    .excludePathPatterns("/api/admin/v1/train_download_blank")
                    .excludePathPatterns("/api/admin/v1/get_order_no_auth")
                    .excludePathPatterns("/api/admin/v1/get_order_for_business_trip_doc")
            ;
        }
    }

    @ConfigurationProperties(value = "http-tvm-main", ignoreUnknownFields = false)
    public class MainHttpTvmProperties extends BaseHttpTvmProperties {
    }

    @ConfigurationProperties(value = "http-tvm-admin", ignoreUnknownFields = false)
    public class AdminHttpTvmProperties extends BaseHttpTvmProperties {
    }

    @ConfigurationProperties(value = "http-tvm-cpa-export", ignoreUnknownFields = false)
    public class CpaExportHttpTvmProperties extends BaseHttpTvmProperties {
    }

    @ConfigurationProperties(value = "http-tvm-takeout", ignoreUnknownFields = false)
    public class TakeoutHttpTvmProperties extends BaseHttpTvmProperties {
    }

    @ConfigurationProperties(value = "http-tvm-administrator", ignoreUnknownFields = false)
    public class AdministratorHttpTvmProperties extends BaseHttpTvmProperties {
    }

    @ConfigurationProperties(value = "http-tvm-idm", ignoreUnknownFields = false)
    public class IdmHttpTvmProperties extends BaseHttpTvmProperties {
    }


    @Bean
    public CommonHttpHeadersServletFilterRegistration commonHttpHeadersServletFilterRegistration() {
        var registrationBean = new CommonHttpHeadersServletFilterRegistration();
        registrationBean.setOrder(100);
        registrationBean.setFilter(new CommonHttpHeadersServletFilter());
        registrationBean.addUrlPatterns("/api/*");
        registrationBean.addUrlPatterns("/1/takeout/*");
        registrationBean.setName("commonHttpHeadersFilter");
        return registrationBean;
    }

    @Bean
    public UnhandledExceptionsResolver unhandledExceptionsResolver(ObjectMapper objectMapper) {
        return new UnhandledExceptionsResolver(objectMapper);
    }

    @Bean
    public ProtobufJsonFormatHttpMessageConverter protobufHttpMessageConverter() {
        return new ProtobufJsonFormatHttpMessageConverter(
                null,
                JsonFormat.printer()
                        .preservingProtoFieldNames()
                        .omittingInsignificantWhitespace()
                        .includingDefaultValueFields()
        );
    }
}
