package ru.yandex.travel.api.config.hotels

import io.grpc.Channel
import io.grpc.ClientInterceptor
import io.grpc.ManagedChannelBuilder
import io.grpc.health.v1.HealthGrpc
import io.opentracing.Tracer
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.beans.factory.annotation.Qualifier
import org.springframework.boot.context.properties.EnableConfigurationProperties
import org.springframework.context.annotation.Bean
import org.springframework.context.annotation.Configuration
import ru.yandex.travel.api.services.hotels.offercache.OffercacheClientFactory
import ru.yandex.travel.api.services.hotels.offercache.OffercacheService
import ru.yandex.travel.api.services.hotels.offercache.OffercacheServiceImpl
import ru.yandex.travel.commons.concurrent.FutureUtils
import ru.yandex.travel.grpc.interceptors.DefaultTimeoutClientInterceptor
import ru.yandex.travel.grpc.interceptors.LoggingClientInterceptor
import ru.yandex.travel.grpc.interceptors.TracerClientInterceptor
import ru.yandex.travel.hotels.proto.OfferCacheServiceV1Grpc
import ru.yandex.travel.hotels.proto.TPingRpcReq
import ru.yandex.travel.hotels.proto.TPingRpcRsp
import ru.yandex.travel.orders.client.ChannelState
import ru.yandex.travel.orders.client.ChannelSupplier
import ru.yandex.travel.orders.client.GrpcChannelSupplierFactory
import ru.yandex.travel.orders.client.HAGrpcChannelFactory
import ru.yandex.travel.orders.client.LabeledChannel
import java.net.InetAddress
import java.util.Objects

@Configuration
@EnableConfigurationProperties(OffercacheConfigurationProperties::class)
open class OffercacheConfiguration(val properties: OffercacheConfigurationProperties) {
    @Bean
    open fun offercacheService(
        offercacheClientFactory: OffercacheClientFactory
    ): OffercacheService = OffercacheServiceImpl(offercacheClientFactory)

    @Bean
    open fun offercacheClientFactory(
        @Qualifier("OffercacheHAGrpcChannelFactory") channelFactory: HAGrpcChannelFactory,
    ): OffercacheClientFactory {
        return OffercacheClientFactory { OfferCacheServiceV1Grpc.newFutureStub(channelFactory.roundRobinChannel) }
    }

    @Bean("OffercacheHAGrpcChannelFactory")
    open fun haGrpcChannelFactory(
        @Qualifier("OffercacheChannelSupplier") channelSupplier: ChannelSupplier,
        tracer: Tracer,
    ): HAGrpcChannelFactory {
        val builder = HAGrpcChannelFactory.Builder.newBuilder()
        return builder.withPingProducer { ch: Channel ->
            FutureUtils.buildCompletableFuture(
                OfferCacheServiceV1Grpc.newFutureStub(ch).ping(TPingRpcReq.newBuilder().build())
            ).thenApply { rsp: TPingRpcRsp ->
                if (rsp.isReady) {
                    ChannelState.READY
                } else {
                    ChannelState.NOT_READY
                }
            }
        }
            .withFailureDetectorProperties(properties.failureDetection)
            .withChannelSupplier(channelSupplier)
            .withChannelBuilder { createChannel(it, tracer) }
            .build()
    }

    @Bean("OffercacheChannelSupplier")
    open fun getChannelSupplier(properties: OffercacheConfigurationProperties): ChannelSupplier {
        return GrpcChannelSupplierFactory(properties).channelSupplier
    }

    private fun createChannel(
        target: String,
        tracer: Tracer,
    ): LabeledChannel {
        val clientFqdn = Objects.requireNonNull(InetAddress.getLocalHost().canonicalHostName)
        val loggingClientInterceptor = LoggingClientInterceptor(
            clientFqdn, target, setOf(HealthGrpc.getCheckMethod().fullMethodName), tracer
        )
        val defaultTimeoutClientInterceptor = DefaultTimeoutClientInterceptor(properties.timeout)
        val tracerClientInterceptor = TracerClientInterceptor(tracer)
        val interceptors = ArrayList<ClientInterceptor>()
        interceptors.addAll(
            listOf(
                loggingClientInterceptor, defaultTimeoutClientInterceptor,
                tracerClientInterceptor
            )
        )
        return LabeledChannel(
            target,
            ManagedChannelBuilder
                .forTarget(target)
                .intercept(interceptors)
                .usePlaintext()
                .build()
        )
    }
}

