package ru.yandex.travel.api.config.hotels;

import java.util.ArrayList;
import java.util.List;

import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;
import org.springframework.validation.Validator;

import ru.yandex.travel.commons.health.HealthCheckedSupplier;
import ru.yandex.travel.commons.messaging.CompressionSettings;
import ru.yandex.travel.commons.messaging.KeyValueStorage;
import ru.yandex.travel.commons.retry.Retry;
import ru.yandex.travel.commons.yt.ClientReplicatedYtProperties;
import ru.yandex.travel.commons.yt.ConnectionFactory;
import ru.yandex.travel.commons.yt.MultiClusterYtAdapter;
import ru.yandex.travel.commons.yt.SingleClusterYtAdapter;
import ru.yandex.travel.hotels.proto.TPingMessage;

@Configuration
@EnableConfigurationProperties(YtKeyValueProperties.class)
public class YtConfiguration {
    private YtKeyValueProperties properties;

    public YtConfiguration(YtKeyValueProperties properties) {
        this.properties = properties;
    }

    @Bean
    public static Validator configurationPropertiesValidator() {
        return new YtPropertiesValidator();
    }

    @Bean
    public ConnectionFactory getKeyValueStorageFactory() {
        return new ConnectionFactory(properties);
    }

    public List<SingleClusterYtAdapter> getSingleClusterAdapters(ClientReplicatedYtProperties properties, String purpose, ConnectionFactory connectionFactory, Retry retryHelper) {
        ArrayList<SingleClusterYtAdapter> result = new ArrayList<>();
        for (String clusterName : properties.getSinkClusters()) {
            result.add(new SingleClusterYtAdapter(clusterName, purpose, properties.getClusterConfigFor(clusterName),
                    new CompressionSettings(properties.getMessageCodec(), properties.getCompressionLevel()), retryHelper,
                    connectionFactory, true, false, true,
                    this::getPingMessage, properties.getPingLifetime(), properties.isPerInstancePingId()));
        }
        return result;
    }

    private TPingMessage getPingMessage() {
        return TPingMessage.newBuilder().build();
    }

    @Bean
    public KeyValueStorage getKeyValueStorage(ConnectionFactory connectionFactory,
                                              Retry retryHelper) {
        MultiClusterYtAdapter adapter = new MultiClusterYtAdapter(properties, "keyValueStorage",
                getSingleClusterAdapters(properties, "keyValueStorage", connectionFactory, retryHelper));
        adapter.startHealthCheckThread();
        return adapter;
    }

    @Bean
    public HealthCheckedSupplier<KeyValueStorage> storageSupplier(KeyValueStorage storage) {
        return new HealthCheckedSupplier<>(storage, "hotel_offerdata_storage");
    }
}
