package ru.yandex.travel.api.config.train;

import java.net.InetAddress;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.Set;
import java.util.concurrent.CompletableFuture;

import io.grpc.ClientInterceptor;
import io.grpc.ManagedChannelBuilder;
import io.grpc.health.v1.HealthCheckRequest;
import io.grpc.health.v1.HealthCheckResponse;
import io.grpc.health.v1.HealthGrpc;
import io.opentracing.Tracer;
import lombok.RequiredArgsConstructor;
import lombok.SneakyThrows;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.annotation.Qualifier;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.Configuration;

import ru.yandex.travel.api.services.train.TrainOfferServiceConfigurationProperties;
import ru.yandex.travel.commons.concurrent.FutureUtils;
import ru.yandex.travel.grpc.interceptors.DefaultTimeoutClientInterceptor;
import ru.yandex.travel.grpc.interceptors.LoggingClientInterceptor;
import ru.yandex.travel.grpc.interceptors.TracerClientInterceptor;
import ru.yandex.travel.grpc.interceptors.TvmHeaderClientHelper;
import ru.yandex.travel.orders.client.ChannelState;
import ru.yandex.travel.orders.client.ChannelSupplier;
import ru.yandex.travel.orders.client.GrpcChannelSupplierFactory;
import ru.yandex.travel.orders.client.HAGrpcChannelFactory;
import ru.yandex.travel.orders.client.LabeledChannel;
import ru.yandex.travel.train_offer_storage.proto.OfferStorageApiServiceV1Grpc;
import ru.yandex.travel.tvm.TvmWrapper;

@Configuration
@EnableConfigurationProperties(TrainOfferServiceConfigurationProperties.class)
@RequiredArgsConstructor
public class OfferStorageApiConfiguration {
    private final TrainOfferServiceConfigurationProperties properties;
    private final Tracer tracer;

    @Bean
    public TrainOfferClientFactory trainOfferClientFactory(
            @Qualifier("TrainOfferStorageGrpcChannelFactory") HAGrpcChannelFactory haGrpcChannelFactory) {
        return () -> OfferStorageApiServiceV1Grpc.newFutureStub(haGrpcChannelFactory.getRoundRobinChannel());
    }

    @Bean("TrainOfferStorageSupplier")
    public ChannelSupplier getChannelSupplier() {
        GrpcChannelSupplierFactory grpcChannelSupplierFactory = new GrpcChannelSupplierFactory(properties);

        return grpcChannelSupplierFactory.getChannelSupplier();
    }

    @Bean("TrainOfferStorageGrpcChannelFactory")
    public HAGrpcChannelFactory haGrpcChannelFactory(
            @Qualifier("TrainOfferStorageSupplier") ChannelSupplier channelSupplier,
            @Autowired(required = false) TvmWrapper tvm) {
        if (tvm != null) {
            tvm.validateAlias(properties.getTvm().getDestinationAlias());
        }
        HAGrpcChannelFactory.Builder builder = HAGrpcChannelFactory.Builder.newBuilder();
        if (properties.isEnableHealthCheck()) {
            builder = builder.withPingProducer(
                    channel -> FutureUtils.buildCompletableFuture(HealthGrpc
                            .newFutureStub(channel)
                            .check(HealthCheckRequest.newBuilder().build())
                    ).thenApply((rsp) -> rsp.getStatus() == HealthCheckResponse.ServingStatus.SERVING ? ChannelState.READY :
                            ChannelState.NOT_READY));
        } else {
            // to fallback from cslb to l3 balancer
            builder = builder.withPingProducer(channel -> CompletableFuture.completedFuture(ChannelState.READY));
        }
        return builder
                .withFailureDetectorProperties(properties.getFailureDetection())
                .withChannelSupplier(channelSupplier)
                .withChannelBuilder(label -> createChannel(label, getTvmHelper(tvm)))
                .build();
    }

    @SneakyThrows
    private LabeledChannel createChannel(String target, TvmHeaderClientHelper tvmHeaderClientHelper) {
        String clientFqdn = Objects.requireNonNull(InetAddress.getLocalHost().getCanonicalHostName());
        LoggingClientInterceptor loggingClientInterceptor = new LoggingClientInterceptor(
                clientFqdn, target, Set.of(HealthGrpc.getCheckMethod().getFullMethodName()), tracer
        );
        DefaultTimeoutClientInterceptor defaultTimeoutClientInterceptor = new DefaultTimeoutClientInterceptor(
                properties.getTimeout()
        );
        TracerClientInterceptor tracerClientInterceptor = new TracerClientInterceptor(
                tracer
        );
        List<ClientInterceptor> interceptors = new ArrayList<>(4);
        interceptors.addAll(Arrays.asList(loggingClientInterceptor, defaultTimeoutClientInterceptor,
                tracerClientInterceptor));
        if (tvmHeaderClientHelper != null) {
            interceptors.add(tvmHeaderClientHelper.getInterceptor(properties.getTvm().getDestinationAlias()));
        }
        return new LabeledChannel(target,
                ManagedChannelBuilder
                        .forTarget(target)
                        .intercept(interceptors)
                        .usePlaintext()
                        .build());
    }

    private TvmHeaderClientHelper getTvmHelper(TvmWrapper tvm) {
        if (properties.getTvm() != null && properties.getTvm().isEnabled() && tvm != null) {
            return new TvmHeaderClientHelper(tvm);
        } else {
            return null;
        }
    }

    public static interface TrainOfferClientFactory {
        OfferStorageApiServiceV1Grpc.OfferStorageApiServiceV1FutureStub createStub();
    }
}
