package ru.yandex.travel.api.endpoints.admin;

import java.util.UUID;

import javax.validation.Valid;

import io.grpc.StatusRuntimeException;
import io.swagger.annotations.ApiOperation;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.context.request.async.DeferredResult;

import ru.yandex.travel.api.endpoints.admin.req_rsp.AdminGetOrderReqV1;
import ru.yandex.travel.api.endpoints.admin.req_rsp.AdminGetWorkflowReqV1;
import ru.yandex.travel.api.endpoints.admin.req_rsp.AdminLinkIssueReqV1;
import ru.yandex.travel.api.endpoints.admin.req_rsp.AdminListOrdersReqV1;
import ru.yandex.travel.api.endpoints.admin.req_rsp.CalculateHotelMoneyOnlyRefundRspV1;
import ru.yandex.travel.api.endpoints.admin.req_rsp.CalculateHotelOrderRefundRspV1;
import ru.yandex.travel.api.endpoints.admin.req_rsp.ChangeEmailReqV1;
import ru.yandex.travel.api.endpoints.admin.req_rsp.ChangeEventStateReqV1;
import ru.yandex.travel.api.endpoints.admin.req_rsp.ChangePhoneReqV1;
import ru.yandex.travel.api.endpoints.admin.req_rsp.GetOrderPayloadsRspV1;
import ru.yandex.travel.api.endpoints.admin.req_rsp.ManualFullMoneyRefundReqV1;
import ru.yandex.travel.api.endpoints.admin.req_rsp.ManualMoneyRefundReqV1;
import ru.yandex.travel.api.endpoints.admin.req_rsp.ManualTrainFeeRefundReqV1;
import ru.yandex.travel.api.endpoints.admin.req_rsp.ModifyHotelOrderDetailsReqV1;
import ru.yandex.travel.api.endpoints.admin.req_rsp.ModifyOrderPayloadReqV1;
import ru.yandex.travel.api.endpoints.admin.req_rsp.ModifyOrderPayloadRspV1;
import ru.yandex.travel.api.endpoints.admin.req_rsp.MoveHotelOrderToNewContractReqV1;
import ru.yandex.travel.api.endpoints.admin.req_rsp.MoveHotelOrderToNewContractRspV1;
import ru.yandex.travel.api.endpoints.admin.req_rsp.RefundCancelledHotelOrderReqV1;
import ru.yandex.travel.api.endpoints.admin.req_rsp.RefundHotelMoneyOnlyReqV1;
import ru.yandex.travel.api.endpoints.admin.req_rsp.RefundHotelOrderReqV1;
import ru.yandex.travel.api.endpoints.admin.req_rsp.RegenerateVouchersReqV1;
import ru.yandex.travel.api.endpoints.admin.req_rsp.RegenerateVouchersRspV1;
import ru.yandex.travel.api.endpoints.admin.req_rsp.RestoreDolphinOrderReqV1;
import ru.yandex.travel.api.endpoints.admin.req_rsp.RestoreDolphinOrderRspV1;
import ru.yandex.travel.api.endpoints.admin.req_rsp.ResumePartnerPaymentsReqV1;
import ru.yandex.travel.api.endpoints.admin.req_rsp.ResumePartnerPaymentsRspV1;
import ru.yandex.travel.api.endpoints.admin.req_rsp.RetryTrustRefundReqV1;
import ru.yandex.travel.api.endpoints.admin.req_rsp.SendEventReqV1;
import ru.yandex.travel.api.endpoints.admin.req_rsp.SendUserNotificationReqV1;
import ru.yandex.travel.api.endpoints.admin.req_rsp.StopPartnerPaymentsReqV1;
import ru.yandex.travel.api.endpoints.admin.req_rsp.StopPartnerPaymentsRspV1;
import ru.yandex.travel.api.endpoints.trains_booking_flow.req_rsp.DownloadBlankReqV1;
import ru.yandex.travel.api.exceptions.GrpcError;
import ru.yandex.travel.api.infrastucture.BindFromQuery;
import ru.yandex.travel.api.infrastucture.ResponseProcessor;
import ru.yandex.travel.api.models.admin.Order;
import ru.yandex.travel.api.models.admin.OrderForBusinessTripDoc;
import ru.yandex.travel.api.models.admin.OrderLogRecords;
import ru.yandex.travel.api.models.admin.TravelAdminOrderList;
import ru.yandex.travel.api.models.admin.Workflow;
import ru.yandex.travel.api.services.common.RetryStrategyExceptionHelpers;
import ru.yandex.travel.api.services.orders.OrdersNoAuthService;
import ru.yandex.travel.api.services.orders.TrainOrdersService;
import ru.yandex.travel.api.services.orders.TrainOrdersServiceProperties;
import ru.yandex.travel.orders.commons.proto.ESnippet;

@RestController
@RequestMapping(value = "/api/admin")
@RequiredArgsConstructor
@EnableConfigurationProperties(TrainOrdersServiceProperties.class)
@Slf4j
public class TravelOrdersAdminController {
    public final ResponseProcessor responseProcessor;
    public final TravelOrdersAdminService ordersAdminService;
    private final OrdersNoAuthService ordersNoAuthService;
    private final TrainOrdersService trainOrdersService;
    private final TrainOrdersServiceProperties trainOrdersServiceProperties;

    @RequestMapping(value = "/v1/list_orders", method = RequestMethod.POST, produces = "application/json")
    @ApiOperation(value = "Get orders that accepts given filters", response = TravelAdminOrderList.class)
    public DeferredResult<TravelAdminOrderList> listOrders(@RequestBody @Valid AdminListOrdersReqV1 request) {
        return responseProcessor.replyWithFuture(
                "AdminListOrders",
                () -> ordersAdminService.listOrders(request)
        );
    }

    @RequestMapping(value = "/v1/get_order", method = RequestMethod.GET, produces = "application/json")
    @ApiOperation(value = "Get info for single order", response = Order.class)
    public DeferredResult<Order> getOrder(@RequestParam(value = "id", required = false) String id,
                                          @RequestParam(value = "prettyId", required = false) String prettyId,
                                          @RequestParam(value = "snippets", required = false) ESnippet[] snippets) {
        AdminGetOrderReqV1 request = new AdminGetOrderReqV1(id, prettyId, snippets);

        return responseProcessor.replyWithFuture(
                "AdminGetOrder",
                () -> ordersAdminService.getOrder(request)
        );
    }

    @RequestMapping(value = "/v1/get_order_no_auth", method = RequestMethod.GET, produces = "application/json")
    @ApiOperation(value = "Get info for single order", response = Order.class)
    public DeferredResult<Order> getOrderNoAuth(@RequestParam(value = "id", required = false) String id,
                                          @RequestParam(value = "snippets", required = false) ESnippet[] snippets) {
        AdminGetOrderReqV1 request = new AdminGetOrderReqV1(id, null, snippets);

        return responseProcessor.replyWithFuture(
                "AdminGetOrderNoAuth",
                () -> ordersNoAuthService.getOrder(request)
        );
    }

    @RequestMapping(value = "/v1/get_order_for_business_trip_doc", method = RequestMethod.GET, produces = "application/json")
    @ApiOperation(value = "Get data for business trip doc generation", response = OrderForBusinessTripDoc.class)
    public DeferredResult<OrderForBusinessTripDoc> getOrderForBusinessTripDoc(@RequestParam(value = "orderId") String orderId) {
        return responseProcessor.replyWithFuture(
                "AdminGetOrderForBusinessTripDoc",
                () -> ordersNoAuthService.getOrderForBusinessTripDoc(orderId)
        );
    }

    @RequestMapping(value = "/v1/link_issue", method = RequestMethod.POST, produces = "application/json")
    @ApiOperation(value = "Link startrek issue to an order", response = UUID.class)
    public DeferredResult<UUID> linkStartrekIssue(@RequestBody @Valid AdminLinkIssueReqV1 request) {
        return responseProcessor.replyWithFuture(
                "AdminLinkStartrekIssue",
                () -> ordersAdminService.linkStartrekIssue(request)
        );
    }

    @RequestMapping(value = "/v1/get_workflow", method = RequestMethod.GET, produces = "application/json")
    @ApiOperation(value = "Get info for a single workflow", response = Workflow.class)
    public DeferredResult<Workflow> getWorkflow(@RequestParam(value = "id") String id) {
        var request = new AdminGetWorkflowReqV1();
        request.setWorkflowId(id);
        return responseProcessor.replyWithFuture(
                "AdminGetWorkflow",
                () -> ordersAdminService.getWorkflow(request)
        );
    }

    @RequestMapping(value = "/v1/get_order_logs", method = RequestMethod.GET, produces = "application/json")
    @ApiOperation(value = "Get logs from ydb for order", response = OrderLogRecords.class)
    public DeferredResult<OrderLogRecords> getOrderLogs(@RequestParam(value = "id") String id,
                                                        @RequestParam(value = "level", required = false) String level,
                                                        @RequestParam(value = "logger", required = false) String logger,
                                                        @RequestParam(value = "offset", defaultValue = "0") int offset,
                                                        @RequestParam(value = "limit", defaultValue = "100") int limit,
                                                        @RequestParam(value = "searchText", required = false) String searchText) {
        return responseProcessor.replyWithFuture(
                "AdminGetOrderLogs",
                () -> ordersAdminService.getOrderLogs(id, level, logger, offset, limit, searchText)
        );
    }

    @RequestMapping(value = "/v1/send_user_notification", method = RequestMethod.POST, produces = "application/json")
    @ApiOperation(value = "Retry sending of user notification")
    public DeferredResult<Void> sendUserNotification(@RequestBody @Valid SendUserNotificationReqV1 request) {
        return responseProcessor.replyWithFuture(
                "AdminSendUserNotification",
                () -> ordersAdminService.sendUserNotification(request)
        );
    }

    @RequestMapping(value = "/v1/get_order_payloads", method = RequestMethod.GET, produces = "application/json")
    @ApiOperation(value = "Get payloads for specific order", response = GetOrderPayloadsRspV1.class)
    public DeferredResult<GetOrderPayloadsRspV1> getOrderPayloads(@RequestParam(value = "id") String id) {
        return responseProcessor.replyWithFuture(
                "AdminGetOrderPayloads",
                () -> ordersAdminService.getOrderPayloads(id)
        );
    }

    @RequestMapping(value = "/v1/modify_order_payload", method = RequestMethod.POST, produces = "application/json")
    @ApiOperation(value = "Modify payload of specific orderItem in specific order", response =
            ModifyOrderPayloadRspV1.class)
    public DeferredResult<ModifyOrderPayloadRspV1> modifyOrderPayload(@RequestBody ModifyOrderPayloadReqV1 request) {
        return responseProcessor.replyWithFuture(
                "AdminModifyOrderPayload",
                () -> ordersAdminService.modifyOrderPayload(request)
        );
    }

    @RequestMapping(value = "/v1/change_email", method = RequestMethod.POST, produces = "application/json")
    @ApiOperation(value = "Change email for order")
    public DeferredResult<Void> changeEmail(@RequestBody @Valid ChangeEmailReqV1 request) {
        return responseProcessor.replyWithFuture(
                "AdminChangeEmail",
                () -> ordersAdminService.changeEmail(request)
        );
    }

    @RequestMapping(value = "/v1/change_phone", method = RequestMethod.POST, produces = "application/json")
    @ApiOperation(value = "Change phone number for order")
    public DeferredResult<Void> changePhone(@RequestBody ChangePhoneReqV1 request) {
        return responseProcessor.replyWithFuture(
                "AdminChangePhone",
                () -> ordersAdminService.changePhone(request)
        );
    }

    @RequestMapping(value = "/v1/update_train_tickets_status", method = RequestMethod.GET, produces = "application" +
            "/json")
    @ApiOperation(value = "Update train tickets status in IM")
    public DeferredResult<Void> updateTrainTicketsStatus(@RequestParam(value = "orderId") String orderId) {
        return responseProcessor.replyWithFuture(
                "AdminUpdateTrainTicketsStatus",
                () -> ordersAdminService.updateTrainTicketsStatus(orderId)
        );
    }

    @RequestMapping(value = "/v1/manual_money_refund", method = RequestMethod.POST, produces = "application/json")
    @ApiOperation(value = "Initiate manual money refund for specified items")
    public DeferredResult<Void> manualMoneyRefund(@RequestBody @Valid ManualMoneyRefundReqV1 request) {
        return responseProcessor.replyWithFuture(
                "AdminManualMoneyRefund",
                () -> ordersAdminService.manualMoneyRefund(request)
        );
    }

    @RequestMapping(value = "/v1/manual_full_money_refund", method = RequestMethod.POST, produces = "application/json")
    @ApiOperation(value = "Initiate manual full money refund")
    public DeferredResult<Void> manualFullMoneyRefund(@RequestBody @Valid ManualFullMoneyRefundReqV1 request) {
        return responseProcessor.replyWithFuture(
                "AdminManualFullMoneyRefund",
                () -> ordersAdminService.manualFullMoneyRefund(request)
        );
    }

    @RequestMapping(value = "/v1/manual_train_fee_refund", method = RequestMethod.POST, produces = "application/json")
    @ApiOperation(value = "Initiate manual train fee refund")
    public DeferredResult<Void> manualTrainFeeRefund(@RequestBody @Valid ManualTrainFeeRefundReqV1 request) {
        return responseProcessor.replyWithFuture(
                "ManualTrainFeeRefund",
                () -> ordersAdminService.manualTrainFeeRefund(request)
        );
    }

    @RequestMapping(value = "/v1/retry_trust_refund", method = RequestMethod.POST, produces = "application/json")
    @ApiOperation(value = "Retry crashed trust refund")
    public DeferredResult<Void> retryTrustRefund(@RequestBody @Valid RetryTrustRefundReqV1 request) {
        return responseProcessor.replyWithFuture(
                "RetryTrustRefund",
                () -> ordersAdminService.retryTrustRefund(request)
        );
    }

    @RequestMapping(value = "/v1/refund_cancelled_hotel_order", method = RequestMethod.POST, produces = "application/json")
    @ApiOperation(value = "Retry crashed trust refund")
    public DeferredResult<Void> refundCancelledHotelOrder(@RequestBody @Valid RefundCancelledHotelOrderReqV1 request) {
        return responseProcessor.replyWithFuture(
                "RefundCancelledHotelOrder",
                () -> ordersAdminService.refundCancelledHotelOrder(request)
        );
    }

    @RequestMapping(value = "/v1/change_event_state", method = RequestMethod.POST, produces = "application/json")
    @ApiOperation(value = "Change state of workflow event")
    public DeferredResult<Void> changeEventState(@RequestBody @Valid ChangeEventStateReqV1 request) {
        return responseProcessor.replyWithFuture(
                "ChangeEventState",
                () -> ordersAdminService.changeEventState(request)
        );
    }

    @RequestMapping(value = "/v1/send_event", method = RequestMethod.POST, produces = "application/json")
    @ApiOperation(value = "Send new event to workflow")
    public DeferredResult<Void> sendEvent(@RequestBody @Valid SendEventReqV1 request) {
        return responseProcessor.replyWithFuture(
                "SendEvent",
                () -> ordersAdminService.sendEvent(request)
        );
    }

    @RequestMapping(value = "/v1/train_download_blank", method = RequestMethod.GET)
    @ApiOperation(value = "Получение pdf билета")
    public DeferredResult<ResponseEntity<byte[]>> downloadBlank(@BindFromQuery DownloadBlankReqV1 request) {
        return responseProcessor.replyWithFutureRetrying(
                "AdminTrainDownloadBlank",
                () -> trainOrdersService.downloadBlank(request).thenApply(b -> ResponseEntity
                        .status(HttpStatus.OK)
                        .contentType(MediaType.APPLICATION_PDF)
                        .body(b)
                ),
                RetryStrategyExceptionHelpers.defaultStatusUnavailableRetryStrategy(),
                trainOrdersServiceProperties.getDownloadBlankDuration()
        );
    }

    @RequestMapping(value = "/v1/restore_dolphin_order", method = RequestMethod.POST, produces = "application/json")
    @ApiOperation(value = "Restoring 'MANUAL_PROCESSING' Dolphin order to check if we can continue processing it")
    public DeferredResult<RestoreDolphinOrderRspV1> restoreDolphinOrder(@RequestBody @Valid RestoreDolphinOrderReqV1 request) {
        return responseProcessor.replyWithFuture(
                "AdminRestoreDolphinOrder",
                () -> ordersAdminService.restoreDolphinOrder(request)
        );
    }

    @RequestMapping(value = "/v1/regenerate_vouchers", method = RequestMethod.POST, produces = "application/json")
    @ApiOperation(value = "Regenerating all complete vouchers")
    public DeferredResult<RegenerateVouchersRspV1> regenerateVouchers(
            @RequestBody @Valid RegenerateVouchersReqV1 request) {
        return responseProcessor.replyWithFuture(
                "AdminRegenerateVouchers",
                () -> ordersAdminService.regenerateVouchers(request)
        );
    }

    @RequestMapping(value = "/v1/calculate_hotel_order_refund", method = RequestMethod.GET, produces = "application/json")
    @ApiOperation(value = "Calculate hotel order refund")
    public DeferredResult<CalculateHotelOrderRefundRspV1> calculateHotelOrderRefund(@RequestParam(value = "id") String id) {
        return responseProcessor.replyWithFuture("AdminCalculateHotelOrderRefund",
                () -> ordersAdminService.calculateHotelOrderRefund(id));
    }

    @RequestMapping(value = "/v1/refund_hotel_order", method = RequestMethod.POST, produces = "application/json")
    @ApiOperation(value = "Refund hotel order")
    public DeferredResult<Void> refundHotelOrder(@RequestBody @Valid RefundHotelOrderReqV1 request) {
        return responseProcessor.replyWithFuture(
                "AdminRefundHotelOrder",
                () -> ordersAdminService.refundHotelOrder(request)
        );
    }

    @RequestMapping(value = "/v1/calculate_hotel_money_only_refund", method = RequestMethod.GET, produces = "application/json")
    @ApiOperation(value = "Calculate hotel money only refund")
    public DeferredResult<CalculateHotelMoneyOnlyRefundRspV1> calculateHotelMoneyOnlyRefund(@RequestParam(value = "id") String id) {
        return responseProcessor.replyWithFuture("AdminCalculateHotelMoneyOnlyRefund",
                () -> ordersAdminService.calculateHotelMoneyOnlyRefund(id));
    }

    @RequestMapping(value = "/v1/refund_hotel_money_only", method = RequestMethod.POST, produces = "application/json")
    @ApiOperation(value = "Refund hotel money only")
    public DeferredResult<Void> refundHotelOrder(@RequestBody @Valid RefundHotelMoneyOnlyReqV1 request) {
        return responseProcessor.replyWithFuture(
                "AdminRefundHotelMoneyOnly",
                () -> ordersAdminService.refundHotelMoneyOnly(request)
        );
    }

    @RequestMapping(value = "/v1/modify_hotel_order_details", method = RequestMethod.POST, produces = "application/json")
    @ApiOperation(value = "Modify hotel order details")
    public DeferredResult<Void> modifyHotelOrderDetails(@RequestBody @Valid ModifyHotelOrderDetailsReqV1 request) {
        return responseProcessor.replyWithFuture(
                "AdminModifyHotelOrderDetails",
                () -> ordersAdminService.modifyHotelOrderDetails(request)
        );
    }

    @RequestMapping(value = "/v1/stop_partner_payments", method = RequestMethod.POST, produces = "application/json")
    @ApiOperation(value = "Stop payments to partner")
    public DeferredResult<StopPartnerPaymentsRspV1> stopPartnerPayments(@RequestBody @Valid StopPartnerPaymentsReqV1 request) {
        return responseProcessor.replyWithFuture(
                "AdminStopPartnerPayments",
                () -> ordersAdminService.stopPartnerPayments(request)
        );
    }

    @RequestMapping(value = "/v1/resume_partner_payments", method = RequestMethod.POST, produces = "application/json")
    @ApiOperation(value = "Resume payments to partner")
    public DeferredResult<ResumePartnerPaymentsRspV1> resumePartnerPayments(@RequestBody @Valid ResumePartnerPaymentsReqV1 request) {
        return responseProcessor.replyWithFuture(
                "AdminResumePartnerPayments",
                () -> ordersAdminService.resumePartnerPayments(request)
        );
    }

    @RequestMapping(value = "/v1/move_hotel_order_to_new_contract", method = RequestMethod.POST, produces = "application/json")
    @ApiOperation(value = "Move hotel order from one contract to another")
    public DeferredResult<MoveHotelOrderToNewContractRspV1> moveHotelOrderToNewContract(@RequestBody @Valid MoveHotelOrderToNewContractReqV1 request) {
        return responseProcessor.replyWithFuture(
                "AdminMoveHotelOrderToNewContract",
                () -> ordersAdminService.moveHotelOrderToNewContract(request)
        );
    }

    @ExceptionHandler(IllegalArgumentException.class)
    @ResponseStatus(HttpStatus.BAD_REQUEST)
    public String handleIllegalArgumentException(IllegalArgumentException e) {
        return e.getMessage();
    }

    @ExceptionHandler(StatusRuntimeException.class)
    public ResponseEntity<GrpcError> handleGrpcErrors(StatusRuntimeException ex) {
        GrpcError error = GrpcError.fromGrpcStatusRuntimeException(ex);
        return ResponseEntity.status(error.getStatus()).contentType(MediaType.APPLICATION_JSON).body(error);
    }
}
