package ru.yandex.travel.api.endpoints.admin;

import java.math.BigDecimal;
import java.time.Instant;
import java.time.LocalDate;
import java.time.ZoneOffset;
import java.util.ArrayList;
import java.util.Base64;
import java.util.List;
import java.util.Objects;
import java.util.UUID;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.ImmutableBiMap;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;
import org.springframework.boot.context.properties.EnableConfigurationProperties;
import org.springframework.stereotype.Service;

import ru.yandex.avia.booking.partners.gateways.aeroflot.model.AeroflotServicePayload;
import ru.yandex.misc.lang.StringUtils;
import ru.yandex.travel.api.endpoints.admin.req_rsp.CalculateHotelMoneyOnlyRefundRspV1;
import ru.yandex.travel.api.endpoints.admin.req_rsp.CalculateHotelOrderRefundRspV1;
import ru.yandex.travel.api.endpoints.booking_flow.model.PaymentErrorCode;
import ru.yandex.travel.api.infrastucture.JsonUtils;
import ru.yandex.travel.api.models.admin.AdminFilterValues;
import ru.yandex.travel.api.models.admin.AdminOrderGeneratedPromoCodesInfo;
import ru.yandex.travel.api.models.admin.AdminOrderPriceInfo;
import ru.yandex.travel.api.models.admin.Amenity;
import ru.yandex.travel.api.models.admin.AuthorizedUser;
import ru.yandex.travel.api.models.admin.AviaLegInfo;
import ru.yandex.travel.api.models.admin.AviaOrderItemInfo;
import ru.yandex.travel.api.models.admin.AviaSegmentInfo;
import ru.yandex.travel.api.models.admin.AviaTravellerInfo;
import ru.yandex.travel.api.models.admin.BasicHotelInfo;
import ru.yandex.travel.api.models.admin.BedGroupInfo;
import ru.yandex.travel.api.models.admin.BriefAviaOrderItemInfo;
import ru.yandex.travel.api.models.admin.BriefAviaOrderItemSegmentInfo;
import ru.yandex.travel.api.models.admin.BriefBusOrderItemInfo;
import ru.yandex.travel.api.models.admin.BriefBusPassengerInfo;
import ru.yandex.travel.api.models.admin.BriefHotelOrderItemInfo;
import ru.yandex.travel.api.models.admin.BriefPassengerInfo;
import ru.yandex.travel.api.models.admin.BriefTrainOrderItemInfo;
import ru.yandex.travel.api.models.admin.BusOrderItemInfo;
import ru.yandex.travel.api.models.admin.BusPassengerInfo;
import ru.yandex.travel.api.models.admin.BusRideInfo;
import ru.yandex.travel.api.models.admin.CancellationInfo;
import ru.yandex.travel.api.models.admin.CancellationPenalty;
import ru.yandex.travel.api.models.admin.ConfirmationInfo;
import ru.yandex.travel.api.models.admin.DefaultOrderItemInfo;
import ru.yandex.travel.api.models.admin.EventInfo;
import ru.yandex.travel.api.models.admin.Guest;
import ru.yandex.travel.api.models.admin.HotelOrderItemInfo;
import ru.yandex.travel.api.models.admin.InsuranceInfo;
import ru.yandex.travel.api.models.admin.InvoiceItemInfo;
import ru.yandex.travel.api.models.admin.LocalizedPansionInfo;
import ru.yandex.travel.api.models.admin.Order;
import ru.yandex.travel.api.models.admin.OrderFlags;
import ru.yandex.travel.api.models.admin.OrderGuestInfo;
import ru.yandex.travel.api.models.admin.OrderLogRecord;
import ru.yandex.travel.api.models.admin.OrderLogRecords;
import ru.yandex.travel.api.models.admin.OrderRefundInfo;
import ru.yandex.travel.api.models.admin.OwnerInfo;
import ru.yandex.travel.api.models.admin.PansionType;
import ru.yandex.travel.api.models.admin.PaymentAttemptInfo;
import ru.yandex.travel.api.models.admin.PaymentScheduleInfo;
import ru.yandex.travel.api.models.admin.PaymentScheduleItemInfo;
import ru.yandex.travel.api.models.admin.PendingInvoiceInfo;
import ru.yandex.travel.api.models.admin.PromoCodeApplicationResult;
import ru.yandex.travel.api.models.admin.ReceiptItem;
import ru.yandex.travel.api.models.admin.RefundInfo;
import ru.yandex.travel.api.models.admin.RequestInfo;
import ru.yandex.travel.api.models.admin.RoomInfo;
import ru.yandex.travel.api.models.admin.TrainOrderItemInfo;
import ru.yandex.travel.api.models.admin.TrainPassengerInfo;
import ru.yandex.travel.api.models.admin.TrainPlaceInfo;
import ru.yandex.travel.api.models.admin.TrainTicketInfo;
import ru.yandex.travel.api.models.admin.TravelAdminOrderList;
import ru.yandex.travel.api.models.admin.TravelAdminOrderListItem;
import ru.yandex.travel.api.models.admin.TrustRefundInfo;
import ru.yandex.travel.api.models.admin.TrustRefundItemInfo;
import ru.yandex.travel.api.models.admin.TrustRefundType;
import ru.yandex.travel.api.models.admin.Voucher;
import ru.yandex.travel.api.models.admin.Workflow;
import ru.yandex.travel.api.models.admin.WorkflowStateInfo;
import ru.yandex.travel.api.models.admin.promo.AdminMir2020PromoCampaign;
import ru.yandex.travel.api.models.admin.promo.AdminPromoCampaigns;
import ru.yandex.travel.api.models.admin.promo.AdminTaxi2020PromoCampaign;
import ru.yandex.travel.api.models.admin.promo.AdminYandexPlus;
import ru.yandex.travel.api.models.common.PromoCodeApplicationResultType;
import ru.yandex.travel.api.services.dictionaries.avia.AviaAirportDictionary;
import ru.yandex.travel.bus.model.BusReservation;
import ru.yandex.travel.bus.model.BusRide;
import ru.yandex.travel.bus.model.BusesPassenger;
import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.hotels.common.orders.ExpediaHotelItinerary;
import ru.yandex.travel.hotels.common.orders.HotelItinerary;
import ru.yandex.travel.hotels.models.booking_flow.Offer;
import ru.yandex.travel.orders.admin.proto.EMir2020PromoEligibility;
import ru.yandex.travel.orders.admin.proto.EPartnerType;
import ru.yandex.travel.orders.admin.proto.TAdminGeneratedPromoCodesInfo;
import ru.yandex.travel.orders.admin.proto.TAdminListAviaInfo;
import ru.yandex.travel.orders.admin.proto.TAdminListHotelInfo;
import ru.yandex.travel.orders.admin.proto.TAdminListItem;
import ru.yandex.travel.orders.admin.proto.TAdminOrderFlags;
import ru.yandex.travel.orders.admin.proto.TAdminOrderInfo;
import ru.yandex.travel.orders.admin.proto.TAdminOrderInvoiceInfo;
import ru.yandex.travel.orders.admin.proto.TAdminOrderItemInfo;
import ru.yandex.travel.orders.admin.proto.TAdminOrderPriceInfo;
import ru.yandex.travel.orders.admin.proto.TAdminPayment;
import ru.yandex.travel.orders.admin.proto.TAdminPaymentSchedule;
import ru.yandex.travel.orders.admin.proto.TAdminPaymentScheduleItem;
import ru.yandex.travel.orders.admin.proto.TAdminPendingInvoice;
import ru.yandex.travel.orders.admin.proto.TCalculateHotelOrderRefundRsp;
import ru.yandex.travel.orders.admin.proto.TCalculateMoneyOnlyRefundRsp;
import ru.yandex.travel.orders.admin.proto.TGetLogRecordsRsp;
import ru.yandex.travel.orders.admin.proto.TGetOrderRsp;
import ru.yandex.travel.orders.admin.proto.TGetWorkflowRsp;
import ru.yandex.travel.orders.admin.proto.TListOrdersRsp;
import ru.yandex.travel.orders.admin.proto.TOrderPromoCampaignsInfo;
import ru.yandex.travel.orders.admin.proto.TOrderTaxi2020PromoCampaignInfo;
import ru.yandex.travel.orders.commons.proto.EDisplayOrderType;
import ru.yandex.travel.orders.commons.proto.EOrderType;
import ru.yandex.travel.orders.commons.proto.EServiceType;
import ru.yandex.travel.orders.commons.proto.ESnippet;
import ru.yandex.travel.orders.proto.TUserInfo;
import ru.yandex.travel.train.model.TrainModelHelpers;
import ru.yandex.travel.train.model.TrainReservation;

@Slf4j
@Service
@RequiredArgsConstructor
@EnableConfigurationProperties(AdminMappingConfigurationProperties.class)
public class TravelOrdersAdminMapper {
    // todo(tlg-13): TRAVELBACK-1961: should be removed when the admin app stops using raw types
    static final ImmutableBiMap<EDisplayOrderType, EOrderType> DISPLAY_ORDER_TYPE_TO_ORDER_TYPE_MAPPING =
            ImmutableBiMap.<EDisplayOrderType, EOrderType>builder()
                    .put(EDisplayOrderType.DT_AVIA, EOrderType.OT_AVIA_AEROFLOT)
                    .put(EDisplayOrderType.DT_BUS, EOrderType.OT_BUS)
                    .put(EDisplayOrderType.DT_HOTEL, EOrderType.OT_HOTEL_EXPEDIA)
                    .put(EDisplayOrderType.DT_SUBURBAN, EOrderType.OT_GENERIC)
                    .put(EDisplayOrderType.DT_TRAIN, EOrderType.OT_TRAIN)
                    .put(EDisplayOrderType.DT_UNKNOWN, EOrderType.OT_UNKNOWN)
                    .build();

    private final AdminMappingConfigurationProperties config;
    private final AviaAirportDictionary airportDict;

    public TravelAdminOrderList tListOrdersRspToTravelAdminOrderList(TListOrdersRsp listOrdersResponse) {
        TravelAdminOrderList orderList = new TravelAdminOrderList();
        List<TravelAdminOrderListItem> itemList = new ArrayList<>();
        listOrdersResponse.getOrderList().forEach(listItem -> itemList.add(mapListItemFromProto(listItem)));
        orderList.setOrders(itemList);
        orderList.setQuantity(listOrdersResponse.getQuantity());
        var adminFilterValues = AdminFilterValues.builder()
                // todo(tlg-13): TRAVELBACK-1961: this field should be removed when the admin app stops using it
                .orderTypeValues(
                        listOrdersResponse.getFilterValues().getOrderTypeList()
                                .stream().distinct().sorted().collect(Collectors.toUnmodifiableList())
                )
                .displayOrderTypeValues(
                        listOrdersResponse.getFilterValues().getOrderTypeList()
                                .stream()
                                .map(type -> DISPLAY_ORDER_TYPE_TO_ORDER_TYPE_MAPPING.inverse().get(type))
                                .filter(Objects::nonNull)
                                .distinct().sorted().collect(Collectors.toUnmodifiableList())
                )
                .orderStateValues(
                        listOrdersResponse.getFilterValues().getDisplayOrderStateList().stream().distinct()
                                .sorted().collect(Collectors.toUnmodifiableList())
                )
                .partnerValues(
                        listOrdersResponse.getFilterValues().getPartnerList()
                                .stream().distinct().sorted().collect(Collectors.toUnmodifiableList())
                )
                .currentValues(AdminFilterValues.CurrentFilterValues.builder()
                        .orderIdFilter(listOrdersResponse.getFilterValues().getOrderIdFilter())
                        .prettyIdFilter(listOrdersResponse.getFilterValues().getPrettyIdFilter())
                        .providerIdFilter(listOrdersResponse.getFilterValues().getProviderIdFilter())
                        .emailFilter(listOrdersResponse.getFilterValues().getEmailFilter())
                        .phoneFilter(listOrdersResponse.getFilterValues().getPhoneFilter())
                        .orderTypeFilter(listOrdersResponse.getFilterValues().getOrderTypeFilter())
                        .orderStateFilter(listOrdersResponse.getFilterValues().getOrderStateFilter())
                        .partnerFilter(listOrdersResponse.getFilterValues().getPartnerTypeFilter())
                        .createdAtFromFilter(ProtoUtils.toInstant(listOrdersResponse.getFilterValues().getCreatedAtFromFilter()))
                        .createdAtToFilter(ProtoUtils.toInstant(listOrdersResponse.getFilterValues().getCreatedAtToFilter()))
                        .purchaseTokenFilter(listOrdersResponse.getFilterValues().getPurchaseTokenFilter())
                        .cardMaskFilter(listOrdersResponse.getFilterValues().getCardMaskFilter())
                        .rrnFilter(listOrdersResponse.getFilterValues().getRrnFilter())
                        .passengerNamesFilter(listOrdersResponse.getFilterValues().getPassengerNamesFilter())
                        .ticketNumberFilter(listOrdersResponse.getFilterValues().getTicketNumberFilter())
                        .yandexUidFilter(listOrdersResponse.getFilterValues().getYandexUidFilter())
                        .carrierFilter(listOrdersResponse.getFilterValues().getCarrierFilter())
                        .referralPartnerIdFilter(listOrdersResponse.getFilterValues().getReferralPartnerIdFilter())
                        .build())
                .build();
        orderList.setFilterValues(adminFilterValues);
        return orderList;
    }

    public TravelAdminOrderListItem mapListItemFromProto(TAdminListItem protoListItem) {
        TravelAdminOrderListItem item = new TravelAdminOrderListItem();
        item.setOrderId(protoListItem.getOrderId());
        item.setPrettyId(protoListItem.getPrettyId());
        item.setWorkflowId(protoListItem.getWorkflowId());
        item.setBroken(protoListItem.getBroken());

        item.setOrderType(protoListItem.getOrderType());
        item.setDisplayOrderType(protoListItem.getDisplayOrderType());
        item.setDisplayOrderState(protoListItem.getDisplayOrderState());
        // todo(tlg-13): TRAVELBACK-1888: tmp order list W/A will be fixed with the UI support introduced
        if (protoListItem.getOrderType() == EOrderType.OT_GENERIC
                && protoListItem.getDisplayOrderType() == EDisplayOrderType.DT_TRAIN) {
            item.setOrderType(EOrderType.OT_TRAIN);
        }

        item.setCreatedAt(ProtoUtils.toInstant(protoListItem.getCreatedAt()));
        item.setUpdatedAt(ProtoUtils.toInstant(protoListItem.getUpdatedAt()));
        if (protoListItem.hasExpiresAt()) {
            item.setExpiresAt(ProtoUtils.toInstant(protoListItem.getExpiresAt()));
        }

        List<String> clientNames = new ArrayList<>();
        List<EPartnerType> partners = new ArrayList<>();
        List<BriefHotelOrderItemInfo> hotelInfoList = new ArrayList<>();
        List<BriefAviaOrderItemInfo> aviaInfoList = new ArrayList<>();
        List<BriefTrainOrderItemInfo> trainInfoList = new ArrayList<>();
        List<BriefBusOrderItemInfo> busInfoList = new ArrayList<>();
        if (protoListItem.getPartnerInfoCount() > 0) {
            protoListItem.getPartnerInfoList().forEach(partnerInfo -> {
                partners.add(partnerInfo.getPartnerType());
                if (partnerInfo.hasPrice()) {
                    item.setPriceInfo(ProtoUtils.fromTPrice(partnerInfo.getPrice()));
                }
                switch (partnerInfo.getPartnerType()) {
                    case PT_HOTEL_EXPEDIA:
                    case PT_HOTEL_DOLPHIN:
                    case PT_HOTEL_TRAVELLINE:
                    case PT_HOTEL_BNOVO:
                    case PT_HOTEL_BRONEVIK:
                        var hotelInfo = new BriefHotelOrderItemInfo();
                        TAdminListHotelInfo protoHotelInfo = partnerInfo.getHotelInfo();
                        hotelInfo.setHotelName(protoHotelInfo.getHotelName());
                        hotelInfo.setCity(protoHotelInfo.getHotelAddress());
                        hotelInfo.setPhone(protoHotelInfo.getHotelPhone());
                        hotelInfo.setArriveDate(ProtoUtils.toLocalDate(protoHotelInfo.getCheckIn()));
                        hotelInfo.setCheckOutDate(ProtoUtils.toLocalDate(protoHotelInfo.getCheckOut()));
                        hotelInfoList.add(hotelInfo);

                        protoHotelInfo.getTravellerList().forEach(traveller -> clientNames.add(traveller.getFio()));
                        break;
                    case PT_AVIA_AEROFLOT:
                        var aviaInfo = new BriefAviaOrderItemInfo();
                        aviaInfo.setSegments(new ArrayList<>());
                        TAdminListAviaInfo protoAviaInfo = partnerInfo.getAviaInfo();
                        protoAviaInfo.getSegmentList().forEach(tSegment -> {
                            var segmentInfo = new BriefAviaOrderItemSegmentInfo();
                            segmentInfo.setFlightNumber(tSegment.getFlightNumber());
                            if (tSegment.hasDeparture()) {
                                segmentInfo.setDepartureDateTime(ProtoUtils.toLocalDateTime(tSegment.getDeparture()));
                            }
                            String departureCode = tSegment.getDepartureAirportCode();
                            String arrivalCode = tSegment.getArrivalAirportCode();
                            segmentInfo.setDirection(departureCode + " - " + arrivalCode);
                            segmentInfo.setDirectionFull(airportDict.getByIataCode(departureCode).getTitle() + " - " +
                                    airportDict.getByIataCode(arrivalCode).getTitle());
                            aviaInfo.getSegments().add(segmentInfo);
                        });
                        protoAviaInfo.getTravellerList().forEach(tPassenger -> clientNames.add(tPassenger.getFio()));
                        aviaInfoList.add(aviaInfo);
                        break;
                    case PT_TRAIN_RZHD:
                        var trainInfo = new BriefTrainOrderItemInfo();
                        var protoTrainInfo = partnerInfo.getTrainInfo();
                        trainInfo.setTrainNumber(protoTrainInfo.getTrainNumber());
                        trainInfo.setCarNumber(protoTrainInfo.getCarNumber());
                        trainInfo.setDepartureStation(protoTrainInfo.getDepartureStation());
                        trainInfo.setArrivalStation(protoTrainInfo.getArrivalStation());
                        if (protoTrainInfo.hasDeparture()) {
                            trainInfo.setDepartureDateTime(ProtoUtils.toInstant(protoTrainInfo.getDeparture()));
                        }
                        List<BriefPassengerInfo> passengers = new ArrayList<>();
                        protoTrainInfo.getTravellerList().forEach(tPassenger -> {
                            var passengerInfo = new BriefPassengerInfo();
                            passengerInfo.setFio(tPassenger.getFio());
                            passengerInfo.setAge(tPassenger.getAge());
                            passengerInfo.setPlaceNumbers(tPassenger.getPlaceNumbersList());
                            passengers.add(passengerInfo);
                        });
                        trainInfo.setPassengers(passengers);
                        trainInfoList.add(trainInfo);
                        break;
                    case PT_BUS_BUSFOR:
                    case PT_BUS_ECOLINES:
                    case PT_BUS_ETRAFFIC:
                    case PT_BUS_NOY:
                    case PT_BUS_OK:
                    case PT_BUS_RUSET:
                    case PT_BUS_SKS:
                    case PT_BUS_UNITIKI:
                    case PT_BUS_YUGAVTOTRANS:
                        var protoBusInfo = partnerInfo.getBusInfo();
                        var busInfo = BriefBusOrderItemInfo.builder()
                                .routeName(protoBusInfo.getName())
                                .passengers(protoBusInfo.getPassengerList().stream()
                                        .map(passenger -> BriefBusPassengerInfo.builder()
                                                .seatId(Strings.emptyToNull(passenger.getSeat()))
                                                .build())
                                        .collect(Collectors.toList())
                                );
                        if (protoBusInfo.hasDeparture()) {
                            busInfo.departure(ProtoUtils.toInstant(protoBusInfo.getDeparture()));
                            busInfo.departureTimeZone(protoBusInfo.getDepartureTimeZone());
                        }
                        if (protoBusInfo.hasArrival()) {
                            busInfo.arrival(ProtoUtils.toInstant(protoBusInfo.getArrival()));
                            busInfo.arrivalTimeZone(protoBusInfo.getArrivalTimeZone());
                        }
                        busInfoList.add(busInfo.build());
                        break;
                    default:
                        break;
                }
            });
        }
        item.setHotelInfo(hotelInfoList);
        item.setAviaInfo(aviaInfoList);
        item.setTrainInfo(trainInfoList);
        item.setBusInfo(busInfoList);
        item.setPartners(partners);
        item.setClientsNames(String.join(", ", clientNames));
        item.setDeferred(protoListItem.getDeferred());
        return item;
    }

    private Money calculateTrainTotalCost(TrainReservation reservation) {
        return reservation.getPassengers().stream()
                .filter(p -> p.getTicket() != null)
                .map(passenger -> TrainModelHelpers.calculateTotalCostForPassenger(reservation, passenger))
                .reduce(Money::add)
                .orElse(Money.zero(ProtoCurrencyUnit.RUB));
    }


    public Order getOrderFromProto(TGetOrderRsp protoOrder, List<ESnippet> snippets) {
        Order order = new Order();
        TAdminOrderInfo orderInfo = protoOrder.getOrderInfo();
        Preconditions.checkNotNull(orderInfo, "Admin order info is null");
        order.setId(orderInfo.getOrderId());
        order.setYandexOrderId(orderInfo.getPrettyId());
        order.setUrl(config.getOrderBaseUrl() + orderInfo.getOrderId() + "/");
        order.setAdminActionToken(orderInfo.getAdminActionToken());
        order.setStatus(orderInfo.getDisplayOrderState());
        order.setOrderType(orderInfo.getOrderType());
        order.setDisplayOrderType(orderInfo.getDisplayOrderType());
        // todo(tlg-13): TRAVELBACK-1888: tmp order actions W/A; will be fixed when the UI support is introduced
        if (orderInfo.getOrderType() == EOrderType.OT_GENERIC
                && orderInfo.getDisplayOrderType() == EDisplayOrderType.DT_TRAIN) {
            order.setOrderType(EOrderType.OT_TRAIN);
        }
        order.setCreatedAt(ProtoUtils.toInstant(orderInfo.getCreatedAt()));
        order.setUpdatedAt(ProtoUtils.toInstant(orderInfo.getUpdatedAt()));
        order.setExpiresAt(ProtoUtils.toInstant(orderInfo.getExpiresAt()));
        order.setWorkflowId(orderInfo.getWorkflowId());
        order.setWorkflowType(orderInfo.getWorkflowType());
        order.setUserActionScheduled(orderInfo.getUserActionScheduled());
        order.setBroken(orderInfo.getBroken());
        order.setPostPayEligible(orderInfo.getIsPostPayEligible());
        order.setPostPaid(orderInfo.getIsPostPaid());
        if (orderInfo.hasOrderFlags()) {
            var orderFlags = new OrderFlags();
            TAdminOrderFlags protoOrderFlags = orderInfo.getOrderFlags();
            orderFlags.setHasMaskedInfo(protoOrderFlags.getHasMaskedInfo());
            orderFlags.setCanResendSuccessfulMail(protoOrderFlags.getCanSendSuccessfulMail());
            orderFlags.setCanResendRefundMail(protoOrderFlags.getCanSendRefundMail());
            orderFlags.setCanManualRefundAllMoney(protoOrderFlags.getCanRefundAllMoney());
            orderFlags.setCanManualRefundByFiscalItem(protoOrderFlags.getCanRefundByFiscalItem());
            orderFlags.setCanManualRefundYandexFee(protoOrderFlags.getCanRefundYandexFee());
            orderFlags.setCanRestoreDolphinOrder(protoOrderFlags.getCanRestoreDolphinOrder());
            orderFlags.setCanRegenerateVouchers(protoOrderFlags.getCanRegenerateVouchers());
            orderFlags.setCanRefundHotelOrder(protoOrderFlags.getCanRefundHotelOrder());
            orderFlags.setCanAmountRefundHotelOrder(protoOrderFlags.getCanAmountRefundHotelOrder());
            orderFlags.setCanOnlyFullRefundHotelOrder(protoOrderFlags.getCanOnlyFullRefundHotelOrder());
            orderFlags.setCanRefundHotelMoneyOnly(protoOrderFlags.getCanRefundHotelMoneyOnly());
            orderFlags.setCanModifyHotelOrderDetails(protoOrderFlags.getCanModifyHotelOrderDetails());
            orderFlags.setCanRetryMoneyRefund(protoOrderFlags.getCanRetryMoneyRefund());
            order.setFlags(orderFlags);
        }
        if (orderInfo.hasServicedAt()) {
            order.setServicedAt(ProtoUtils.toInstant(orderInfo.getServicedAt()));
        }

        if (snippets != null && snippets.contains(ESnippet.S_PRIVATE_INFO)) {
            OwnerInfo ownerInfo = new OwnerInfo();
            TUserInfo protoOwner = orderInfo.getOwner();
            ownerInfo.setYandexUid(protoOwner.getYandexUid());
            ownerInfo.setPassportId(protoOwner.getPassportId());
            ownerInfo.setLogin(protoOwner.getLogin());
            ownerInfo.setPhone(protoOwner.getPhone());
            ownerInfo.setEmail(protoOwner.getEmail());
            ownerInfo.setIp(protoOwner.getIp());
            ownerInfo.setAllowsSubscription(protoOwner.getAllowsSubscription());
            order.setOwner(ownerInfo);
        }

        if (orderInfo.getOrderRefundCount() != 0) {
            var orderRefundList = new ArrayList<OrderRefundInfo>(orderInfo.getOrderRefundCount());
            orderInfo.getOrderRefundList().forEach(refund -> {
                var refundInfo = new OrderRefundInfo();
                refundInfo.setId(refund.getId());
                refundInfo.setState(refund.getState());
                refundInfo.setType(refund.getType());
                if (refund.hasRefundedAmount()) {
                    refundInfo.setRefundedAmount(ProtoUtils.fromTPrice(refund.getRefundedAmount()));
                }
                if (refund.hasCreatedAt()) {
                    refundInfo.setCreatedAt(ProtoUtils.toInstant(refund.getCreatedAt()));
                }
                if (refund.hasUpdatedAt()) {
                    refundInfo.setUpdatedAt(ProtoUtils.toInstant(refund.getUpdatedAt()));
                }
                orderRefundList.add(refundInfo);
            });
            order.setOrderRefunds(orderRefundList);
        }

        order.setHotelOrderItems(new ArrayList<>());
        order.setAviaOrderItems(new ArrayList<>());
        order.setTrainOrderItems(new ArrayList<>());
        order.setBusOrderItems(new ArrayList<>());
        order.setDefaultOrderItems(new ArrayList<>());
        orderInfo.getOrderItemList().forEach(orderItemInfo -> {
            switch (orderItemInfo.getOrderItemType()) {
                case PT_BNOVO_HOTEL:
                case PT_DOLPHIN_HOTEL:
                case PT_EXPEDIA_HOTEL:
                case PT_TRAVELLINE_HOTEL:
                case PT_BRONEVIK_HOTEL:
                    HotelItinerary itinerary = ProtoUtils.fromTJson(orderItemInfo.getPayload(), HotelItinerary.class);
                    HotelOrderItemInfo hotelInfo = mapHotelOrderItemInfo(itinerary, snippets,
                            protoOrder.getOrderInfo().getOrderId());
                    if (orderItemInfo.getOrderItemType() == EServiceType.PT_EXPEDIA_HOTEL) {
                        var expediaItinerary = ProtoUtils.fromTJson(orderItemInfo.getPayload(),
                                ExpediaHotelItinerary.class);
                        hotelInfo.setItineraryId(expediaItinerary.getExpediaItineraryId());
                    }
                    if (hotelInfo.getConfirmationInfo() == null) {
                        hotelInfo.setConfirmationInfo(new ConfirmationInfo());
                    }
                    hotelInfo.getConfirmationInfo().setDocumentUrl(protoOrder.getOrderInfo().getDocumentUrl());
                    if (orderItemInfo.hasConfirmedAt()) {
                        hotelInfo.getConfirmationInfo().setConfirmedAt(ProtoUtils.toInstant(orderItemInfo.getConfirmedAt()));
                    }
                    mapCommonFieldsForOrderItems(hotelInfo, orderItemInfo);
                    order.getHotelOrderItems().add(hotelInfo);
                    break;
                case PT_FLIGHT:
                    AeroflotServicePayload payload = ProtoUtils.fromTJson(orderItemInfo.getPayload(),
                            AeroflotServicePayload.class);
                    AviaOrderItemInfo aviaInfo = mapAviaOrderItemInfo(payload, snippets);
                    if (orderItemInfo.hasConfirmedAt()) {
                        aviaInfo.setConfirmedAt(ProtoUtils.toInstant(orderItemInfo.getConfirmedAt()));
                    }
                    mapCommonFieldsForOrderItems(aviaInfo, orderItemInfo);
                    order.getAviaOrderItems().add(aviaInfo);
                    break;
                case PT_TRAIN:
                    TrainOrderItemInfo trainInfo = mapTrainOrderInfo(orderItemInfo, snippets);
                    order.getTrainOrderItems().add(trainInfo);
                    break;
                case PT_BUS:
                    BusOrderItemInfo busInfo = mapBusOrderInfo(orderItemInfo, snippets);
                    order.getBusOrderItems().add(busInfo);
                    break;
                default:
                    var defaultOrderItemInfo = new DefaultOrderItemInfo();
                    mapCommonFieldsForOrderItems(defaultOrderItemInfo, orderItemInfo);
                    order.getDefaultOrderItems().add(defaultOrderItemInfo);
                    break;
            }
        });

        order.setPaymentInfo(orderInfo.getInvoiceList().stream()
                .map(this::getPaymentAttemptInfo).collect(Collectors.toList()));

        order.setVouchers(protoOrder.getVouchersList().stream()
                .map(v -> {
                            var voucher = new Voucher();
                            voucher.setId(v.getId());
                            voucher.setName(v.getName());
                            voucher.setUrl(v.getUrl());
                            return voucher;
                        }
                ).collect(Collectors.toList()));

        List<AuthorizedUser> authorizedUserList = protoOrder.getAuthorizedUsersList().stream()
                .map(user -> AuthorizedUser.builder()
                        .login(user.getLogin())
                        .yandexUid(user.getYandexUid())
                        .role(user.getRole())
                        .isLoggedIn(user.getLoggedIn())
                        .build())
                .collect(Collectors.toList());
        order.setAuthorizedUsers(authorizedUserList);

        if (protoOrder.getStarTrekTicketCount() > 0) {
            order.setStarTrekTicketIds(protoOrder.getStarTrekTicketList());
        }

        order.setPromoCampaigns(mapPromoCampaigns(protoOrder.getPromoCampaigns()));

        if (protoOrder.getOrderInfo().hasOrderPriceInfo()) {
            TAdminOrderPriceInfo orderPriceInfo = protoOrder.getOrderInfo().getOrderPriceInfo();
            order.setOrderPriceInfo(new AdminOrderPriceInfo());
            order.getOrderPriceInfo().setDiscountAmount(ProtoUtils.fromTPrice(orderPriceInfo.getDiscountAmount()));
            order.getOrderPriceInfo().setOriginalPrice(ProtoUtils.fromTPrice(orderPriceInfo.getOriginalPrice()));
            order.getOrderPriceInfo().setPrice(ProtoUtils.fromTPrice(orderPriceInfo.getPrice()));
            order.getOrderPriceInfo().setCurrentTotalPaymentsSum(ProtoUtils.fromTPrice(orderPriceInfo.getCurrentTotalPaymentsSum()));
            order.getOrderPriceInfo().setPromoCodeApplicationResults(
                    orderPriceInfo.getPromoCodeApplicationResultsList().stream().map(par -> {
                                PromoCodeApplicationResult result = new PromoCodeApplicationResult();
                                result.setPromoCodeId(par.getPromoCodeId());
                                result.setPromoCodeActivationId(par.getPromoCodeActivationId());
                                result.setCode(par.getCode());
                                result.setType(PromoCodeApplicationResultType.fromProto(par.getType()));
                                if (par.hasDiscountAmount()) {
                                    result.setDiscountAmount(ProtoUtils.fromTPrice(par.getDiscountAmount()));
                                }
                                return result;
                            }
                    ).collect(Collectors.toUnmodifiableList())
            );
        }

        if (protoOrder.getOrderInfo().hasGeneratedPromoCodes()) {
            TAdminGeneratedPromoCodesInfo generatedPromoCodes = protoOrder.getOrderInfo().getGeneratedPromoCodes();
            var generatedPromoCodesInfo = new AdminOrderGeneratedPromoCodesInfo();
            generatedPromoCodesInfo.setId(generatedPromoCodes.getId());
            generatedPromoCodesInfo.setCode(generatedPromoCodes.getCode());
            generatedPromoCodesInfo.setActivationsStrategy(generatedPromoCodes.getActivationsStrategy());
            if (generatedPromoCodes.hasDiscountAmount()) {
                generatedPromoCodesInfo.setDiscountAmount(ProtoUtils.fromTPrice(generatedPromoCodes.getDiscountAmount()));
            }
            generatedPromoCodesInfo.setAllowedActivationsLeft(generatedPromoCodes.getAllowedActivationsLeft());
            generatedPromoCodesInfo.setGenerationStrategy(generatedPromoCodes.getGenerationStrategy());
            if (generatedPromoCodes.hasMinTotalCost()) {
                generatedPromoCodesInfo.setMinTotalCost(ProtoUtils.fromTPrice(generatedPromoCodes.getMinTotalCost()));
            }
            if (generatedPromoCodes.hasValidFrom()) {
                generatedPromoCodesInfo.setValidFrom(ProtoUtils.toInstant(generatedPromoCodes.getValidFrom()));
            }
            if (generatedPromoCodes.hasValidTill()) {
                generatedPromoCodesInfo.setValidTill(ProtoUtils.toInstant(generatedPromoCodes.getValidTill()));
            }
            order.setGeneratedPromoCodesInfo(generatedPromoCodesInfo);
        }

        if (orderInfo.getPartnersCount() > 0) {
            order.setPartners(orderInfo.getPartnersList());
        }

        protoOrder.getOrderInfo().getPaymentsList().stream()
                .filter(TAdminPayment::hasPaymentSchedule)
                .findFirst().ifPresent(ps -> order.setPaymentSchedule(getPaymentScheduleInfo(ps.getPaymentSchedule())));
        order.setPendingInvoices(
                protoOrder.getOrderInfo().getPaymentsList().stream()
                        .filter(TAdminPayment::hasPendingInvoice)
                        .map(pi -> getPendingInvoiceInfo(pi.getPendingInvoice()))
                        .collect(Collectors.toList()));
        return order;
    }

    protected PendingInvoiceInfo getPendingInvoiceInfo(TAdminPendingInvoice pendingInvoice) {
        var builder = PendingInvoiceInfo.builder()
                .id(pendingInvoice.getId())
                .workflowId(pendingInvoice.getWorkflowId())
                .createdAt(ProtoUtils.toInstant(pendingInvoice.getCreatedAt()))
                .state(pendingInvoice.getState())
                .pendingAmount(ProtoUtils.fromTPrice(pendingInvoice.getPendingAmount()))
                .totalAmount(ProtoUtils.fromTPrice(pendingInvoice.getTotalAmount()))
                .paymentAttempts(pendingInvoice.getAttemptsList().stream().map(this::getPaymentAttemptInfo).collect(Collectors.toList()));
        if (pendingInvoice.hasPaidAt()) {
            builder.paidAt(ProtoUtils.toInstant(pendingInvoice.getPaidAt()));
        }
        return builder.build();
    }

    protected PaymentScheduleInfo getPaymentScheduleInfo(TAdminPaymentSchedule schedule) {
        return PaymentScheduleInfo.builder()
                .id(schedule.getId())
                .state(schedule.getState())
                .workflowId(schedule.getWorkflowId())
                .initialInvoice(getPendingInvoiceInfo(schedule.getInitial()))
                .items(schedule.getItemsList().stream().map(this::getPaymentScheduleItemInfo).collect(Collectors.toList()))
                .build();
    }

    protected PaymentScheduleItemInfo getPaymentScheduleItemInfo(TAdminPaymentScheduleItem item) {
        return PaymentScheduleItemInfo.builder()
                .id(item.getId())
                .createdAt(ProtoUtils.toInstant(item.getCreatedAt()))
                .paymentEndsAt(ProtoUtils.toInstant(item.getPaymentEndsAt()))
                .invoice(getPendingInvoiceInfo(item.getInvoice()))
                .penaltyIfUnpaid(ProtoUtils.fromTPrice(item.getPenaltyIfUnpaid()))
                .emailSent(item.getNotificationEmailSent())
                .ticketCreated(item.getNotificationTicketCreated())
                .build();
    }

    protected PaymentAttemptInfo getPaymentAttemptInfo(TAdminOrderInvoiceInfo invoice) {
        PaymentAttemptInfo info = new PaymentAttemptInfo();
        if (invoice.hasPaymentTs()) {
            info.setPaidAt(ProtoUtils.toInstant(invoice.getPaymentTs()));
        }
        if (invoice.hasPaymentStartTs()) {
            info.setStartedAt(ProtoUtils.toInstant(invoice.getPaymentStartTs()));
        }
        if (invoice.hasPaymentCancelTs()) {
            info.setCancelledAt(ProtoUtils.toInstant(invoice.getPaymentCancelTs()));
        }
        info.setTrustPaymentId(invoice.getTrustPaymentId());
        info.setPaymentUrl(invoice.getPaymentUrl());
        info.setOriginalPrice(invoice.getInvoiceItemList().stream()
                .map(invoiceItem -> ProtoUtils.fromTPrice(invoiceItem.getOriginalMoneyAmount()))
                .reduce(Money::add)
                .orElse(Money.of(BigDecimal.ZERO, ProtoCurrencyUnit.RUB)));
        info.setCurrentPrice(invoice.getInvoiceItemList().stream()
                .map(invoiceItem -> ProtoUtils.fromTPrice(invoiceItem.getMoneyAmount()))
                .reduce(Money::add)
                .orElse(Money.of(BigDecimal.ZERO, ProtoCurrencyUnit.RUB)));
        if (invoice.getInvoiceItemCount() != 0) {
            info.setInvoiceItems(invoice.getInvoiceItemList().stream()
                    .filter(Objects::nonNull)
                    .map(item -> new InvoiceItemInfo(
                            item.getInvoiceItemId(),
                            item.getFiscalItemId(),
                            item.getFiscalItemType(),
                            ProtoUtils.fromTPrice(item.getMoneyAmount())))
                    .collect(Collectors.toUnmodifiableList()));
        }
        info.setReceipts(invoice.getFiscalReceiptList().stream()
                .filter(Objects::nonNull)
                .map(fri -> new ReceiptItem(fri.getUrl(), fri.getType()))
                .collect(Collectors.toUnmodifiableList()));
        if (invoice.getTrustRefundCount() != 0) {
            info.setTrustRefunds(invoice.getTrustRefundList().stream()
                    .map(refund -> {
                        var refundInfo = new TrustRefundInfo();
                        refundInfo.setId(refund.getId());
                        refundInfo.setTrustRefundId(refund.getTrustRefundId());
                        if (!Strings.isNullOrEmpty(refund.getOrderRefundId())) {
                            refundInfo.setOrderRefundId(refund.getOrderRefundId());
                        }
                        refundInfo.setType(TrustRefundType.fromString(refund.getType()));
                        refundInfo.setState(refund.getState());
                        refundInfo.setCreatedAt(ProtoUtils.toInstant(refund.getCreatedAt()));
                        if (refund.hasConfirmedAt()) {
                            refundInfo.setConfirmedAt(ProtoUtils.toInstant(refund.getConfirmedAt()));
                        }
                        if (refund.getTrustRefundItemCount() != 0) {
                            refundInfo.setRefundItems(refund.getTrustRefundItemList().stream()
                                    .map(item -> {
                                        var refItemInfo = new TrustRefundItemInfo();
                                        refItemInfo.setId(item.getId());
                                        refItemInfo.setOriginalAmount(ProtoUtils.fromTPrice(item.getOriginalAmount()));
                                        refItemInfo.setTargetAmount(ProtoUtils.fromTPrice(item.getTargetAmount()));
                                        return refItemInfo;
                                    })
                                    .collect(Collectors.toList()));
                        }
                        return refundInfo;
                    })
                    .collect(Collectors.toList()));
        }
        info.setPaymentRespCode(PaymentErrorCode.fromErrorCode(invoice.getAuthorizationErrorCode()).toString());
        info.setPaymentRespDesc(invoice.getAuthorizationErrorDesc());
        info.setRrn(invoice.getRrn());
        info.setScroogeUrl(config.getScroogeBaseUrl() + invoice.getPurchaseToken());
        info.setCardNumber(invoice.getUserAccount());
        info.setApprovalCode(invoice.getApprovalCode());
        switch (invoice.getOneOfInvoiceStatesCase()) {
            case TRUSTINVOICESTATE:
                info.setStatus(invoice.getTrustInvoiceState().toString());
                break;
            case AEROFLOTINVOICESTATE:
                info.setStatus(invoice.getAeroflotInvoiceState().toString());
                break;
            case ONEOFINVOICESTATES_NOT_SET:
            default:
                break;
        }
        return info;
    }

    private AdminPromoCampaigns mapPromoCampaigns(TOrderPromoCampaignsInfo campaignsProto) {
        AdminPromoCampaigns campaignsDto = new AdminPromoCampaigns();
        if (campaignsProto.hasTaxi2020()) {
            TOrderTaxi2020PromoCampaignInfo taxi2020Proto = campaignsProto.getTaxi2020();
            campaignsDto.setTaxi2020(AdminTaxi2020PromoCampaign.builder()
                    .status(taxi2020Proto.getStatus())
                    .email(Strings.emptyToNull(taxi2020Proto.getEmail()))
                    .emailScheduledAt(ProtoUtils.toInstantFromSafe(taxi2020Proto.getEmailScheduledAt()))
                    .emailSentAt(ProtoUtils.toInstantFromSafe(taxi2020Proto.getEmailSentAt()))
                    .build());
        }
        if (campaignsProto.hasMir2020()) {
            var builder = AdminMir2020PromoCampaign.builder()
                    .offerEligibility(campaignsProto.getMir2020().getOfferEligibility())
                    .paidWithMir(campaignsProto.getMir2020().getPaidWithMir());
            if (campaignsProto.getMir2020().getPaidWithMir() && campaignsProto.getMir2020().getOfferEligibility() == EMir2020PromoEligibility.MIR_ELIGIBILE) {
                builder.cashbackAmount(campaignsProto.getMir2020().getCashbackAmount().getValue());
            }
            campaignsDto.setMir2020(builder.build());
        }
        if (campaignsProto.hasYandexPlus()) {
            campaignsDto.setYandexPlus(AdminYandexPlus.fromProto(campaignsProto.getYandexPlus()));
        }
        return campaignsDto;
    }

    private <T extends DefaultOrderItemInfo> void mapCommonFieldsForOrderItems(T orderItemInfo,
                                                                               TAdminOrderItemInfo protoOrderItemInfo) {
        orderItemInfo.setOrderItemState(protoOrderItemInfo.getItemState());
        orderItemInfo.setPartner(protoOrderItemInfo.getOrderItemType().toString());
        orderItemInfo.setPayload(JsonUtils.removePersonalData(protoOrderItemInfo.getPayload()).toString());
        orderItemInfo.setHasTestContext(protoOrderItemInfo.getHasTestContext());
        orderItemInfo.setPostPayEligible(protoOrderItemInfo.getIsPostPayEligible());
        orderItemInfo.setPostPaid(protoOrderItemInfo.getIsPostPaid());
    }

    private HotelOrderItemInfo mapHotelOrderItemInfo(HotelItinerary itinerary, List<ESnippet> snippets,
                                                     String orderId) {
        var hotelInfo = new HotelOrderItemInfo();
        Offer offer = getOffer(itinerary, orderId);
        var basicHotelInfo = new BasicHotelInfo();
        if (offer.getHotelInfo() != null) {
            basicHotelInfo.setName(offer.getHotelInfo().getName());
            basicHotelInfo.setAddress(offer.getHotelInfo().getAddress());
            basicHotelInfo.setPhone(offer.getHotelInfo().getPhone());
            basicHotelInfo.setPermalink(offer.getHotelInfo().getPermalink());
        }
        if (itinerary.getOrderDetails() != null) {
            basicHotelInfo.setProviderId(itinerary.getOrderDetails().getProviderId());
            basicHotelInfo.setOriginalId(itinerary.getOrderDetails().getOriginalId());
            if (itinerary.getOrderDetails().getPermalink() != null) {
                basicHotelInfo.setAltayPermalingUrl(config.getAltayPermalinkBaseUrl() + itinerary.getOrderDetails().getPermalink());
            }
            if (StringUtils.isNotBlank(itinerary.getOrderDetails().getProviderId()) &&
                    StringUtils.isNotBlank(itinerary.getOrderDetails().getOriginalId())) {
                basicHotelInfo.setAltaySignalsUrl(config.getAltaySignalsBaseUrl() +
                        itinerary.getOrderDetails().getProviderId() + "||*||" + itinerary.getOrderDetails().getOriginalId());
            }
        }
        hotelInfo.setBasicHotelInfo(basicHotelInfo);
        if (itinerary.getRefundRules() != null) {
            var cancellationInfoBuilder = CancellationInfo.builder();
            cancellationInfoBuilder.refundable(itinerary.getRefundRules().isRefundable());

            cancellationInfoBuilder.penalties(itinerary.getRefundRules().getRules().stream()
                    .filter(Objects::nonNull)
                    .map(rule -> {
                        var penaltyBuidler = CancellationPenalty.builder()
                                .amount(rule.getPenalty() == null ? null : rule.getPenalty().getNumberStripped())
                                .currency(rule.getPenalty() == null ? null :
                                        rule.getPenalty().getCurrency().getCurrencyCode());
                        if (rule.getStartsAt() != null) {
                            penaltyBuidler.startsAt(rule.getStartsAt());
                        }
                        if (rule.getEndsAt() != null) {
                            penaltyBuidler.endsAt(rule.getEndsAt());
                        }
                        if (rule.getType() != null) {
                            penaltyBuidler.type(CancellationPenalty.Type.fromRefundType(rule.getType()));
                        }
                        return penaltyBuidler.build();
                    })
                    .collect(Collectors.toList()));
            hotelInfo.setCancellationInfo(cancellationInfoBuilder.build());
        }
        hotelInfo.setChecksum(offer.getMetaInfo().getCheckSum());
        hotelInfo.setLabel(offer.getMetaInfo().getLabel());
        if (offer.getMetaInfo().getSearch() != null) {
            var requestInfo = new RequestInfo();
            requestInfo.setCheckinDate(offer.getMetaInfo().getSearch().getCheckIn());
            requestInfo.setCheckoutDate(offer.getMetaInfo().getSearch().getCheckOut());
            hotelInfo.setRequestInfo(requestInfo);
        }
        hotelInfo.setSessionKey(offer.getMetaInfo().getDeduplicationKey());
        hotelInfo.setTravelToken(offer.getMetaInfo().getToken());
        if (itinerary.getFiscalPrice() != null) {
            hotelInfo.setPriceInfo(itinerary.getFiscalPrice());
        }

        if (snippets != null && snippets.contains(ESnippet.S_PRIVATE_INFO)) {
            OrderGuestInfo guestInfo = new OrderGuestInfo();
            guestInfo.setCustomerEmail(itinerary.getCustomerEmail());
            guestInfo.setCustomerPhone(itinerary.getCustomerPhone());
            guestInfo.setAllowsSubscription(itinerary.isAllowsSubscription());
            guestInfo.setGuests(itinerary.getGuests().stream()
                    .map(gDto -> new Guest(gDto.getFirstName(), gDto.getLastName(), gDto.getAge(), gDto.isChild()))
                    .collect(Collectors.toList()));
            hotelInfo.setGuestInfo(guestInfo);
        }

        if (itinerary.getConfirmation() != null) {
            ConfirmationInfo confirmationInfo = new ConfirmationInfo();
            if (StringUtils.isNotBlank(itinerary.getConfirmation().getHotelConfirmationId())) {
                confirmationInfo.setConfirmationId(itinerary.getConfirmation().getHotelConfirmationId());
            } else {
                confirmationInfo.setConfirmationId(itinerary.getConfirmation().getPartnerConfirmationId());
            }
            hotelInfo.setConfirmationInfo(confirmationInfo);
        }
        if (itinerary.getRefundInfo() != null) {
            hotelInfo.setRefundInfo(new RefundInfo(itinerary.getRefundInfo().getRefundDateTime().toInstant(ZoneOffset.UTC),
                    itinerary.getRefundInfo().getPenaltyIntervalIndex(),
                    itinerary.getRefundInfo().getRefund(),
                    itinerary.getRefundInfo().getPenalty()));
        }

        List<Amenity> amenities = offer.getRoomInfo().getRoomAmenities().stream()
                .map(amenity -> new Amenity(amenity.getId(), amenity.getName()))
                .collect(Collectors.toList());
        LocalizedPansionInfo pansionInfo = new LocalizedPansionInfo(
                PansionType.BY_PROTO.getByValueOrNull(offer.getRoomInfo().getPansionInfo().getId()),
                offer.getRoomInfo().getPansionInfo().getName()
        );
        List<BedGroupInfo> bedGroups = new ArrayList<>();
        if (offer.getRoomInfo().getBedGroups() != null) {
            bedGroups = offer.getRoomInfo().getBedGroups().stream()
                    .map(bedGroup -> new BedGroupInfo(bedGroup.getDescription()))
                    .collect(Collectors.toList());
        }
        hotelInfo.setRoomInfo(RoomInfo.builder()
                .amenities(amenities)
                .pansionInfo(pansionInfo)
                .bedGroups(bedGroups)
                .build()
        );

        // TODO(tivelkov): pass orderCancellationDetails to admin
        return hotelInfo;
    }

    public Offer getOffer(HotelItinerary itinerary, String orderId) {
        return Offer.fromJsonNode(itinerary.getUiPayload());
    }

    public AviaOrderItemInfo mapAviaOrderItemInfo(AeroflotServicePayload payload, List<ESnippet> snippets) {
        var aviaInfo = new AviaOrderItemInfo();

        if (payload.getBookingRef() != null) {
            aviaInfo.setPnr(payload.getBookingRef().getPnr());
        }

        if (snippets != null && snippets.contains(ESnippet.S_PRIVATE_INFO) && payload.getTravellers() != null) {
            List<AviaTravellerInfo> travellersInfo = payload.getTravellers().stream()
                    .map(traveller -> AviaTravellerInfo.builder()
                            .firstName(traveller.getFirstName())
                            .middleName(traveller.getMiddleName())
                            .lastName(traveller.getLastName())
                            .nationality(traveller.getNationalityCode())
                            .documentType(traveller.getDocumentType())
                            .documentNumber(traveller.getDocumentNumber())
                            .dateOfBirth(traveller.getDateOfBirth())
                            .build())
                    .collect(Collectors.toList());
            aviaInfo.setPassengers(travellersInfo);
        }

        if (payload.getVariant() != null && payload.getVariant().getOffer() != null) {
            Money totalPrice = payload.getVariant().getOffer().getTotalPrice();
            if (totalPrice != null) {
                aviaInfo.setTotalPrice(totalPrice);
            }
        }

        if (payload.getVariant() != null && payload.getVariant().getOriginDestinations() != null) {
            var legs = payload.getVariant().getOriginDestinations().stream()
                    .map(origin -> {
                        List<AviaSegmentInfo> segments =
                                payload.getVariant().getOriginDestinationSegments(origin.getId()).stream()
                                        .map(segment -> AviaSegmentInfo.builder()
                                                .flightNumber(segment.getMarketingCarrier().getFlightNumber())
                                                .aircraftCode(segment.getAircraftCode())
                                                .departureAirportCode(segment.getDeparture().getAirportCode())
                                                .departureAirport(airportDict.getByIataCode(segment.getDeparture().getAirportCode()).getTitle())
                                                .arrivalAirportCode(segment.getArrival().getAirportCode())
                                                .arrivalAirport(airportDict.getByIataCode(segment.getArrival().getAirportCode()).getTitle())
                                                .departureAt(segment.getDeparture().getDateTime())
                                                .arrivalAt(segment.getArrival().getDateTime())
                                                .build())
                                        .collect(Collectors.toList());
                        return AviaLegInfo.builder()
                                .segments(segments)
                                .build();
                    })
                    .collect(Collectors.toList());
            aviaInfo.setLegs(legs);
        }

        return aviaInfo;
    }

    private TrainOrderItemInfo mapTrainOrderInfo(TAdminOrderItemInfo orderItemInfo, List<ESnippet> snippets) {
        TrainReservation trainReservation = ProtoUtils.fromTJson(orderItemInfo.getPayload(), TrainReservation.class);
        var trainInfo = new TrainOrderItemInfo();
        mapCommonFieldsForOrderItems(trainInfo, orderItemInfo);
        trainInfo.setPartnerNumber(trainReservation.getReservationNumber());
        trainInfo.setPassengersInfo(trainReservation.getPassengers().stream()
                .map(passenger -> {
                    var passengerInfoBuilder = TrainPassengerInfo.builder()
                            .citizenshipCode(passenger.getCitizenshipCode())
                            .customerId(passenger.getCustomerId())
                            .documentType(passenger.getDocumentType().getValue())
                            .sex(passenger.getSex().getValue())
                            .tariffType(passenger.getTariffCodeWithFallback());
                    if (passenger.getCategory() != null) {
                        passengerInfoBuilder.category(passenger.getCategory().getValue());
                    }
                    if (passenger.getInsurance() != null) {
                        InsuranceInfo.InsuranceInfoBuilder insuranceInfoBuilder = InsuranceInfo.builder()
                                .amount(passenger.getInsurance().getAmount())
                                .company(passenger.getInsurance().getCompany())
                                .partnerOperationId(passenger.getInsurance().getPartnerOperationId());
                        if (passenger.getInsurance().getPartnerOperationStatus() != null) {
                            insuranceInfoBuilder.partnerOperationStatus(passenger.getInsurance().getPartnerOperationStatus().getValue());
                        }
                        passengerInfoBuilder.insurance(insuranceInfoBuilder.build());
                    }
                    var ticket = passenger.getTicket();
                    if (ticket != null) {
                        TrainTicketInfo.TrainTicketInfoBuilder trainTicketInfoBuilder = TrainTicketInfo.builder()
                                .blankId(ticket.getBlankId())
                                .rawTariffType(ticket.getRawTariffName())
                                .bookedTariffCode(ticket.getBookedTariffCode())
                                .pendingElectronicRegistration(ticket.isPendingElectronicRegistration())
                                .places(ticket.getPlaces().stream()
                                        .map(place -> TrainPlaceInfo.builder()
                                                .number(place.getNumber())
                                                .type(place.getType().getValue())
                                                .build())
                                        .collect(Collectors.toList()))
                                .totalCost(ticket.calculateTotalCost())
                                .tariffAmount(ticket.getTariffAmount())
                                .serviceAmount(ticket.getServiceAmount())
                                .feeAmount(ticket.getFeeAmount())
                                .refundFeeAmount(ticket.calculateRefundFeeAmountOrNull());
                        if (ticket.getRefundStatus() != null) {
                            trainTicketInfoBuilder.refundStatus(ticket.getRefundStatus().toString());
                        }
                        if (ticket.getImBlankStatus() != null) {
                            trainTicketInfoBuilder.imBlankStatus(ticket.getImBlankStatus().getValue());
                        }
                        passengerInfoBuilder.ticket(trainTicketInfoBuilder.build());
                    }
                    if (snippets.contains(ESnippet.S_PRIVATE_INFO)) {
                        passengerInfoBuilder.firstName(passenger.getFirstName())
                                .lastName(passenger.getLastName())
                                .patronymic(passenger.getPatronymic())
                                .birthday(passenger.getBirthday())
                                .documentNumber(passenger.getDocumentNumber());
                    }
                    return passengerInfoBuilder.build();
                }).collect(Collectors.toList()));
        if (trainReservation.getInsuranceStatus() != null) {
            trainInfo.setOrderInsuranceStatus(trainReservation.getInsuranceStatus().toString());
        }
        trainInfo.setTrainNumber(trainReservation.getReservationRequestData().getTrainTicketNumber());
        trainInfo.setCarNumber(trainReservation.getCarNumber());
        trainInfo.setCarrier(trainReservation.getCarrier());
        trainInfo.setDirection(trainReservation.getReservationRequestData().getTrainTitle());
        trainInfo.setDepartureStation(trainReservation.getUiData().getStationFromTitle());
        trainInfo.setArrivalStation(trainReservation.getUiData().getStationToTitle());
        trainInfo.setDepartureAt(trainReservation.getReservationRequestData().getDepartureTime());
        trainInfo.setArrivalAt(trainReservation.getArrivalTime());
        trainInfo.setRefundTill(TrainModelHelpers.calculateCanRefundOnServiceTill(trainReservation));
        trainInfo.setERegisterStatusTill(TrainModelHelpers.calculateCanChangeElectronicRegistrationTill(trainReservation));
        return trainInfo;
    }

    private void setPassengerInfo(BusPassengerInfo.BusPassengerInfoBuilder builder,
                                  BusesPassenger passenger, boolean hasPrivateInfo) {
        builder
                .citizenship(passenger.getCitizenship())
                .citizenshipPartnerId(passenger.getCitizenshipPartnerId())
                .gender(passenger.getGender())
                .genderPartnerId(passenger.getGenderPartnerId())
                .documentType(passenger.getDocumentType())
                .documentTypePartnerId(passenger.getDocumentTypePartnerId())
                .seatId(passenger.getSeatId())
                .seatPartnerId(passenger.getSeatPartnerId())
                .ticketType(passenger.getTicketType())
                .ticketTypePartnerId(passenger.getTicketTypePartnerId());
        if (hasPrivateInfo) {
            builder
                    .birthday(passenger.getBirthday())
                    .firstName(passenger.getFirstName())
                    .middleName(Strings.nullToEmpty(passenger.getMiddleName()))
                    .lastName(passenger.getLastName())
                    .documentNumber(passenger.getDocumentNumber());
        }
    }

    public static String tryDecodeBusTicketId(String id) {
        try {
            byte[] bytes = Base64.getDecoder().decode(id);
            return new String(bytes);
        } catch (Exception e) {
            return null;
        }
    }

    private BusOrderItemInfo mapBusOrderInfo(TAdminOrderItemInfo orderItemInfo, List<ESnippet> snippets) {
        BusOrderItemInfo busInfo = new BusOrderItemInfo();
        mapCommonFieldsForOrderItems(busInfo, orderItemInfo);
        BusReservation busReservation = ProtoUtils.fromTJson(orderItemInfo.getPayload(), BusReservation.class);
        busInfo.setOfferId(busReservation.getOfferId());
        boolean hasPrivateInfo = snippets.contains(ESnippet.S_PRIVATE_INFO);
        if (busReservation.getOrder() != null) {
            busInfo.setPassengers(busReservation.getOrder().getTickets().stream().map(ticket -> {
                var passengerBuilder = BusPassengerInfo.builder()
                        .fullPrice(ticket.getTotalPrice())
                        .idParams(tryDecodeBusTicketId(ticket.getId()))
                        .blankUrl(hasPrivateInfo ? ticket.getBlankUrl() : null);
                setPassengerInfo(passengerBuilder, ticket.getPassenger(), hasPrivateInfo);
                return passengerBuilder.build();
            }).collect(Collectors.toList()));
        } else {
            busInfo.setPassengers(busReservation.getRequestPassengers().stream().map(passenger -> {
                var passengerBuilder = BusPassengerInfo.builder();
                setPassengerInfo(passengerBuilder, passenger, hasPrivateInfo);
                return passengerBuilder.build();
            }).collect(Collectors.toList()));
        }
        BusRide ride = busReservation.getRide();
        busInfo.setRide(BusRideInfo.builder()
                .arrivalTime(ride.getArrivalTime())
                .arrivalTimeZone(ride.getTitlePointTo().getTimezone())
                .benefits(ride.getBenefits())
                .bookOnly(ride.isBookOnly())
                .bus(ride.getBus())
                .busPartner(ride.getBusPartner())
                .canPayOffline(ride.isCanPayOffline())
                .carrier(ride.getCarrier())
                .carrierCode(ride.getCarrierCode())
                .departureTime(ride.getDepartureTime())
                .departureTimeZone(ride.getTitlePointFrom().getTimezone())
                .duration(ride.getDuration())
                .fee(ride.getFee())
                .freeSeats(ride.getFreeSeats())
                .onlineRefund(ride.isOnlineRefund())
                .price(ride.getPrice())
                .refundConditions(ride.getRefundConditions())
                .rideId(ride.getRideId())
                .routeName(ride.getRouteName())
                .routeNumber(ride.getRouteNumber())
                .titlePointFrom(ride.getTitlePointFrom())
                .titlePointTo(ride.getTitlePointTo())
                .pointFrom(ride.getPointFrom())
                .pointTo(ride.getPointTo())
                .supplier(ride.getSupplier())
                .supplierId(ride.getSupplierId())
                .ticketLimit(ride.getTicketLimit())
                .yandexFee(ride.getYandexFee())
                .build());
        return busInfo;
    }

    private int countPassengerAge(LocalDate passengerBirthday) {
        LocalDate today = LocalDate.now();
        int age = today.getYear() - passengerBirthday.getYear();
        if (today.getDayOfYear() < passengerBirthday.getDayOfYear()) {
            age -= 1;
        }
        return age;
    }

    public OrderLogRecords mapTGetLogRecordsToOrderLogRecords(TGetLogRecordsRsp protoRecords) {
        OrderLogRecords.OrderLogRecordsBuilder responseBuilder = OrderLogRecords.builder();
        List<OrderLogRecord> recordList = protoRecords.getLogRecordList().stream()
                .map(protoRecord -> OrderLogRecord.builder()
                        .ownerUuid(UUID.fromString(protoRecord.getOwnerId()))
                        .timestamp(Instant.ofEpochMilli(protoRecord.getTimestamp()))
                        .messageId(protoRecord.getMessageId())
                        .level(protoRecord.getLevel())
                        .logger(protoRecord.getLogger())
                        .message(protoRecord.getMessage())
                        .context(protoRecord.getContext())
                        .hostName(protoRecord.getHostName())
                        .build())
                .collect(Collectors.toList());
        responseBuilder.records(recordList);
        responseBuilder.count(protoRecords.getCount());
        return responseBuilder.build();
    }

    public Workflow mapWorkflowInfoFromProto(TGetWorkflowRsp protoWorkflow) {
        var workflow = tWorkflowRspToWorkflow(protoWorkflow);
        if (protoWorkflow.getSupervisedWorkflowsCount() > 0) {
            List<Workflow> supervisedWorkflows = new ArrayList<>();
            protoWorkflow.getSupervisedWorkflowsList()
                    .forEach(supervised -> supervisedWorkflows.add(tWorkflowRspToWorkflow(supervised)));
            workflow.setSupervisedWorkflows(supervisedWorkflows);
        }
        return workflow;
    }

    public Workflow tWorkflowRspToWorkflow(TGetWorkflowRsp protoWorkflow) {
        var workflow = Workflow.builder()
                .workflowId(protoWorkflow.getWorkflowId())
                .entityId(protoWorkflow.getEntityId())
                .entityType(protoWorkflow.getEntityType())
                .supervisorId(protoWorkflow.getSupervisorId())
                .workflowState(protoWorkflow.getWorkflowState().toString())
                .version(protoWorkflow.getVersion())
                .workflowVersion(protoWorkflow.getWorkflowVersion());

        if (protoWorkflow.getEventInfoCount() > 0) {
            workflow.events(protoWorkflow.getEventInfoList().stream()
                    .map(protoEvent -> {
                        var event = EventInfo.builder()
                                .id(protoEvent.getId())
                                .eventType(protoEvent.getEventType())
                                .state(protoEvent.getState().toString())
                                .timesTried(protoEvent.getTimesTried())
                                .createdAt(ProtoUtils.toInstant(protoEvent.getCreatedAt()))
                                .data(protoEvent.getData());
                        if (protoEvent.hasProcessedAt()) {
                            event.processedAt(ProtoUtils.toInstant(protoEvent.getProcessedAt()));
                        }
                        return event.build();
                    }).collect(Collectors.toList()));
        }

        if (protoWorkflow.getWorkflowStateTransitionCount() > 0) {
            workflow.workflowTransitions(protoWorkflow.getWorkflowStateTransitionList().stream()
                    .map(protoTransition -> WorkflowStateInfo.builder()
                            .id(protoTransition.getId())
                            .workflowId(protoTransition.getWorkflowId())
                            .fromState(protoTransition.getFromState())
                            .toState(protoTransition.getToState())
                            .createdAt(ProtoUtils.toInstant(protoTransition.getTransitionAt()))
                            .build()
                    ).collect(Collectors.toList()));
        }

        return workflow.build();
    }

    public CalculateHotelOrderRefundRspV1 mapCalculatedHotelOrderRefundMoney(TCalculateHotelOrderRefundRsp ordersRsp) {
        CalculateHotelOrderRefundRspV1 rsp = new CalculateHotelOrderRefundRspV1();
        rsp.setOrderId(ordersRsp.getOrderId());
        rsp.setOrderPrettyId(ordersRsp.getOrderPrettyId());
        rsp.setRefundAmountByRules(ProtoUtils.fromTPrice(ordersRsp.getRefundAmountByRules()));
        rsp.setPaidAmount(ProtoUtils.fromTPrice(ordersRsp.getPaidAmount()));
        rsp.setTotalAmount(ProtoUtils.fromTPrice(ordersRsp.getTotalAmount()));

        return rsp;
    }

    public CalculateHotelMoneyOnlyRefundRspV1 mapCalculatedHotelMoneyOnlyRefundMoney(TCalculateMoneyOnlyRefundRsp ordersRsp) {
        CalculateHotelMoneyOnlyRefundRspV1 rsp = new CalculateHotelMoneyOnlyRefundRspV1();
        rsp.setOrderId(ordersRsp.getOrderId());
        rsp.setOrderPrettyId(ordersRsp.getOrderPrettyId());
        rsp.setTotalAmount(ProtoUtils.fromTPrice(ordersRsp.getTotalAmount()));
        rsp.setRemainingAmount(ProtoUtils.fromTPrice(ordersRsp.getRemainingAmount()));

        return rsp;
    }
}
