package ru.yandex.travel.api.endpoints.admin;

import java.util.List;
import java.util.Map;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.protobuf.Int32Value;
import io.grpc.Metadata;
import io.grpc.Status;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.stereotype.Service;

import ru.yandex.misc.lang.StringUtils;
import ru.yandex.travel.api.endpoints.admin.req_rsp.AdminGetOrderReqV1;
import ru.yandex.travel.api.endpoints.admin.req_rsp.AdminGetWorkflowReqV1;
import ru.yandex.travel.api.endpoints.admin.req_rsp.AdminLinkIssueReqV1;
import ru.yandex.travel.api.endpoints.admin.req_rsp.AdminListOrdersReqV1;
import ru.yandex.travel.api.endpoints.admin.req_rsp.CalculateHotelMoneyOnlyRefundRspV1;
import ru.yandex.travel.api.endpoints.admin.req_rsp.CalculateHotelOrderRefundRspV1;
import ru.yandex.travel.api.endpoints.admin.req_rsp.ChangeEmailReqV1;
import ru.yandex.travel.api.endpoints.admin.req_rsp.ChangeEventStateReqV1;
import ru.yandex.travel.api.endpoints.admin.req_rsp.ChangePhoneReqV1;
import ru.yandex.travel.api.endpoints.admin.req_rsp.GetOrderPayloadsRspV1;
import ru.yandex.travel.api.endpoints.admin.req_rsp.ManualFullMoneyRefundReqV1;
import ru.yandex.travel.api.endpoints.admin.req_rsp.ManualMoneyRefundReqV1;
import ru.yandex.travel.api.endpoints.admin.req_rsp.ManualTrainFeeRefundReqV1;
import ru.yandex.travel.api.endpoints.admin.req_rsp.ModifyHotelOrderDetailsReqV1;
import ru.yandex.travel.api.endpoints.admin.req_rsp.ModifyOrderPayloadReqV1;
import ru.yandex.travel.api.endpoints.admin.req_rsp.ModifyOrderPayloadRspV1;
import ru.yandex.travel.api.endpoints.admin.req_rsp.MoveHotelOrderToNewContractReqV1;
import ru.yandex.travel.api.endpoints.admin.req_rsp.MoveHotelOrderToNewContractRspV1;
import ru.yandex.travel.api.endpoints.admin.req_rsp.RefundCancelledHotelOrderReqV1;
import ru.yandex.travel.api.endpoints.admin.req_rsp.RefundHotelMoneyOnlyReqV1;
import ru.yandex.travel.api.endpoints.admin.req_rsp.RefundHotelOrderReqV1;
import ru.yandex.travel.api.endpoints.admin.req_rsp.RegenerateVouchersReqV1;
import ru.yandex.travel.api.endpoints.admin.req_rsp.RegenerateVouchersRspV1;
import ru.yandex.travel.api.endpoints.admin.req_rsp.RestoreDolphinOrderReqV1;
import ru.yandex.travel.api.endpoints.admin.req_rsp.RestoreDolphinOrderRspV1;
import ru.yandex.travel.api.endpoints.admin.req_rsp.ResumePartnerPaymentsReqV1;
import ru.yandex.travel.api.endpoints.admin.req_rsp.ResumePartnerPaymentsRspV1;
import ru.yandex.travel.api.endpoints.admin.req_rsp.RetryTrustRefundReqV1;
import ru.yandex.travel.api.endpoints.admin.req_rsp.SendEventReqV1;
import ru.yandex.travel.api.endpoints.admin.req_rsp.SendUserNotificationReqV1;
import ru.yandex.travel.api.endpoints.admin.req_rsp.StopPartnerPaymentsReqV1;
import ru.yandex.travel.api.endpoints.admin.req_rsp.StopPartnerPaymentsRspV1;
import ru.yandex.travel.api.exceptions.TravelApiBadRequestException;
import ru.yandex.travel.api.infrastucture.JsonUtils;
import ru.yandex.travel.api.models.admin.Order;
import ru.yandex.travel.api.models.admin.OrderItemPayloadInfo;
import ru.yandex.travel.api.models.admin.OrderLogRecords;
import ru.yandex.travel.api.models.admin.TravelAdminOrderList;
import ru.yandex.travel.api.models.admin.Workflow;
import ru.yandex.travel.api.services.orders.OrchestratorAdminClientFactory;
import ru.yandex.travel.commons.concurrent.FutureUtils;
import ru.yandex.travel.commons.grpc.ServerUtils;
import ru.yandex.travel.commons.proto.EErrorCode;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.commons.proto.TError;
import ru.yandex.travel.commons.proto.TJson;
import ru.yandex.travel.orders.admin.proto.EBrokenStateEnum;
import ru.yandex.travel.orders.admin.proto.ENotificationTransportType;
import ru.yandex.travel.orders.admin.proto.ENotificationType;
import ru.yandex.travel.orders.admin.proto.TAllMoneyRefund;
import ru.yandex.travel.orders.admin.proto.TAllTrainFeeRefund;
import ru.yandex.travel.orders.admin.proto.TByFiscalItemRefund;
import ru.yandex.travel.orders.admin.proto.TCalculateHotelOrderRefundReq;
import ru.yandex.travel.orders.admin.proto.TCalculateMoneyOnlyRefundReq;
import ru.yandex.travel.orders.admin.proto.TChangeEmailReq;
import ru.yandex.travel.orders.admin.proto.TChangeEventStateReq;
import ru.yandex.travel.orders.admin.proto.TChangePhoneReq;
import ru.yandex.travel.orders.admin.proto.TGetLogRecordsReq;
import ru.yandex.travel.orders.admin.proto.TGetOrderPayloadsReq;
import ru.yandex.travel.orders.admin.proto.TGetOrderReq;
import ru.yandex.travel.orders.admin.proto.TGetWorkflowReq;
import ru.yandex.travel.orders.admin.proto.TLinkStartrekIssueReq;
import ru.yandex.travel.orders.admin.proto.TListOrdersReq;
import ru.yandex.travel.orders.admin.proto.TManualMoneyRefundReq;
import ru.yandex.travel.orders.admin.proto.TModifyHotelOrderDatesParams;
import ru.yandex.travel.orders.admin.proto.TModifyHotelOrderDetailsReq;
import ru.yandex.travel.orders.admin.proto.TModifyHotelOrderGuestsParams;
import ru.yandex.travel.orders.admin.proto.TModifyHotelOrderParams;
import ru.yandex.travel.orders.admin.proto.TModifyOrderPayloadReq;
import ru.yandex.travel.orders.admin.proto.TMoveHotelOrderToNewContractReq;
import ru.yandex.travel.orders.admin.proto.TOrderId;
import ru.yandex.travel.orders.admin.proto.TPayloadInfo;
import ru.yandex.travel.orders.admin.proto.TRefundCancelledHotelOrderReq;
import ru.yandex.travel.orders.admin.proto.TRefundHotelMoneyOnlyReq;
import ru.yandex.travel.orders.admin.proto.TRefundHotelOrderReq;
import ru.yandex.travel.orders.admin.proto.TRegenerateVoucherReq;
import ru.yandex.travel.orders.admin.proto.TRestoreDolphinOrderSecureReq;
import ru.yandex.travel.orders.admin.proto.TResumePaymentsReq;
import ru.yandex.travel.orders.admin.proto.TRetryMoneyRefundReq;
import ru.yandex.travel.orders.admin.proto.TSendEventToWorkflowReq;
import ru.yandex.travel.orders.admin.proto.TSendUserNotificationReq;
import ru.yandex.travel.orders.admin.proto.TStopPaymentsReq;
import ru.yandex.travel.orders.admin.proto.TUpdateTrainTicketsReq;
import ru.yandex.travel.orders.commons.proto.TOffsetPage;
import ru.yandex.travel.orders.services.finances.proto.EMoneyRefundMode;
import ru.yandex.travel.workflow.EWorkflowEventState;

@Service
@Slf4j
@RequiredArgsConstructor
public class TravelOrdersAdminService {
    private final TravelOrdersAdminMapper adminMapper;
    private final OrchestratorAdminClientFactory adminClientFactory;

    public CompletableFuture<Order> getOrder(AdminGetOrderReqV1 request) {
        TGetOrderReq.Builder reqBuilder = TGetOrderReq.newBuilder();
        if (!Strings.isNullOrEmpty(request.getOrderId())) {
            reqBuilder.setOrderId(request.getOrderId());
        } else {
            reqBuilder.setPrettyId(Strings.nullToEmpty(request.getPrettyId()));
        }
        if (request.getSnippets() != null && request.getSnippets().size() != 0) {
            reqBuilder.addAllSnippet(request.getSnippets());
        }
        return FutureUtils.buildCompletableFuture(adminClientFactory.createAdminFutureStub(false)
                .getOrder(reqBuilder.build()))
                .thenApply(rsp -> adminMapper.getOrderFromProto(rsp, request.getSnippets()));
    }

    public CompletableFuture<TravelAdminOrderList> listOrders(AdminListOrdersReqV1 request) {
        TListOrdersReq.Builder requestBuilder = prepareListOrdersProtoRequest(request);

        return FutureUtils.buildCompletableFuture(adminClientFactory.createAdminFutureStub(false)
                .listOrders(requestBuilder.build()))
                .thenApply(adminMapper::tListOrdersRspToTravelAdminOrderList);
    }

    public CompletableFuture<UUID> linkStartrekIssue(AdminLinkIssueReqV1 request) {
        TLinkStartrekIssueReq.Builder reqBuilder = TLinkStartrekIssueReq.newBuilder();
        var orderID = TOrderId.newBuilder();
        if (!Strings.isNullOrEmpty(request.getOrderId())) {
            orderID.setOrderId(request.getOrderId());
        } else {
            orderID.setPrettyId(Strings.nullToEmpty(request.getPrettyId()));
        }
        reqBuilder.setOrderId(orderID);
        reqBuilder.setIssueKey(request.getIssueKey());
        return FutureUtils.buildCompletableFuture(adminClientFactory.createAdminFutureStub(false)
                .linkStartrekIssue(reqBuilder.build()))
                .thenApply(rsp -> UUID.fromString(rsp.getTicketId()));
    }

    public CompletableFuture<Workflow> getWorkflow(AdminGetWorkflowReqV1 request) {
        return FutureUtils.buildCompletableFuture(adminClientFactory.createAdminFutureStub(false)
                .getWorkflow(TGetWorkflowReq.newBuilder()
                        .setWorkflowId(request.getWorkflowId())
                        .build()))
                .thenApply(adminMapper::mapWorkflowInfoFromProto);
    }

    public CompletableFuture<OrderLogRecords> getOrderLogs(String id, String level, String logger, int offset,
                                                           int limit, String searchText) {
        TGetLogRecordsReq.Builder reqBuilder = TGetLogRecordsReq.newBuilder()
                .setOrderId(id)
                .setOffset(offset)
                .setLimit(limit);
        reqBuilder.setLevel(Strings.nullToEmpty(level));
        reqBuilder.setLogger(Strings.nullToEmpty(logger));
        reqBuilder.setSearchText(Strings.nullToEmpty(searchText));
        return FutureUtils.buildCompletableFuture(adminClientFactory.createAdminFutureStub(false)
                .getLogRecords(reqBuilder.build()))
                .thenApply(adminMapper::mapTGetLogRecordsToOrderLogRecords);
    }

    public CompletableFuture<Void> sendUserNotification(SendUserNotificationReqV1 request) {
        Preconditions.checkNotNull(request.getOrderId(), "Order ID should be present");
        Preconditions.checkArgument(request.getTransport() != ENotificationTransportType.UNRECOGNIZED &&
                        request.getTransport() != ENotificationTransportType.NTT_UNKNOWN,
                "Unknown transport type");
        Preconditions.checkArgument(request.getType() != ENotificationType.UNRECOGNIZED &&
                        request.getType() != ENotificationType.NT_UNKNOWN,
                "Unknown notification type");
        return FutureUtils.buildCompletableFuture(adminClientFactory.createAdminFutureStub(true)
                .sendUserNotification(TSendUserNotificationReq.newBuilder()
                        .setOrderId(request.getOrderId())
                        .setTransport(request.getTransport())
                        .setType(request.getType())
                        .setOrderRefundId(Strings.nullToEmpty(request.getOrderRefundId()))
                        .build()))
                .thenApply(rsp -> null);
    }

    public CompletableFuture<GetOrderPayloadsRspV1> getOrderPayloads(String id) {
        return FutureUtils.buildCompletableFuture(adminClientFactory.createAdminFutureStub(false)
                .getOrderPayloads(TGetOrderPayloadsReq.newBuilder()
                        .setOrderId(id)
                        .build()))
                .thenApply(rsp -> GetOrderPayloadsRspV1.builder()
                        .orderId(rsp.getOrderid())
                        .orderPayloads(rsp.getPayloadsList().stream()
                                .map(payload -> OrderItemPayloadInfo.builder()
                                        .orderItemId(payload.getOrderItemId())
                                        .payload(JsonUtils.removePersonalData(payload.getPayload()))
                                        .version(payload.getVersion())
                                        .build())
                                .collect(Collectors.toList()))
                        .build());
    }

    public CompletableFuture<ModifyOrderPayloadRspV1> modifyOrderPayload(ModifyOrderPayloadReqV1 request) {
        return FutureUtils.buildCompletableFuture(adminClientFactory.createAdminFutureStub(true)
                .modifyOrderPayload(TModifyOrderPayloadReq.newBuilder()
                        .setOrderId(request.getOrderId())
                        .setPayload(TPayloadInfo.newBuilder()
                                .setOrderItemId(request.getOrderItemId())
                                .setPayload(ProtoUtils.toTJson(request.getPayload()))
                                .build())
                        .setComment(request.getComment() == null ? "" : request.getComment())
                        .build()))
                .thenApply(rsp -> ModifyOrderPayloadRspV1.builder()
                        .orderId(rsp.getOrderId())
                        .orderItemId(rsp.getOrderItemId())
                        .version(rsp.getVersion())
                        .build());
    }

    public CompletableFuture<Void> changeEmail(ChangeEmailReqV1 request) {
        Preconditions.checkArgument(StringUtils.isNotBlank(request.getOrderId()), "Order ID should be present");
        Preconditions.checkArgument(StringUtils.isNotBlank(request.getNewEmail()), "New email should not be empty");
        return FutureUtils.buildCompletableFuture(adminClientFactory.createAdminFutureStub(true)
                .changeEmail(TChangeEmailReq.newBuilder()
                        .setOrderId(request.getOrderId())
                        .setNewEmail(request.getNewEmail())
                        .build()))
                .thenApply(rsp -> null);
    }

    public CompletableFuture<Void> changePhone(ChangePhoneReqV1 request) {
        Preconditions.checkArgument(StringUtils.isNotBlank(request.getOrderId()), "Order ID should be present");
        Preconditions.checkArgument(StringUtils.isNotBlank(request.getNewPhone()), "New phone number should not be " +
                "empty");
        return FutureUtils.buildCompletableFuture(adminClientFactory.createAdminFutureStub(true)
                .changePhone(TChangePhoneReq.newBuilder()
                        .setOrderId(request.getOrderId())
                        .setNewPhone(request.getNewPhone())
                        .build()))
                .thenApply(rsp -> null);
    }

    public CompletableFuture<Void> updateTrainTicketsStatus(String orderId) {
        Preconditions.checkArgument(StringUtils.isNotBlank(orderId), "Order ID should be present");
        return FutureUtils.buildCompletableFuture(adminClientFactory.createAdminFutureStub(true)
                .updateTrainTickets(TUpdateTrainTicketsReq.newBuilder()
                        .setOrderId(orderId)
                        .build()))
                .thenApply(rsp -> null);
    }

    public CompletableFuture<Void> manualMoneyRefund(ManualMoneyRefundReqV1 request) {
        return FutureUtils.buildCompletableFuture(adminClientFactory.createAdminFutureStub(true)
                .manualMoneyRefund(TManualMoneyRefundReq.newBuilder()
                        .setOrderId(request.getOrderId())
                        .setAdminActionToken(request.getAdminActionToken())
                        .setReason(request.getReason())
                        .setByFiscalItemRefund(TByFiscalItemRefund.newBuilder()
                                .putAllTargetFiscalItems(request.getRefundsByFiscalItem().entrySet().stream()
                                        .collect(Collectors.toMap(Map.Entry::getKey,
                                                entry -> ProtoUtils.toTPrice(entry.getValue())))).build())
                        .build()))
                .thenApply(rsp -> null);
    }

    public CompletableFuture<Void> manualFullMoneyRefund(ManualFullMoneyRefundReqV1 request) {
        return FutureUtils.buildCompletableFuture(adminClientFactory.createAdminFutureStub(true)
                .manualMoneyRefund(TManualMoneyRefundReq.newBuilder()
                        .setOrderId(request.getOrderId())
                        .setAdminActionToken(request.getAdminActionToken())
                        .setReason(request.getReason())
                        .setAllMoneyRefund(TAllMoneyRefund.newBuilder().build())
                        .build()))
                .thenApply(rsp -> null);
    }

    public CompletableFuture<Void> manualTrainFeeRefund(ManualTrainFeeRefundReqV1 request) {
        return FutureUtils.buildCompletableFuture(adminClientFactory.createAdminFutureStub(true)
                .manualMoneyRefund(TManualMoneyRefundReq.newBuilder()
                        .setOrderId(request.getOrderId())
                        .setAdminActionToken(request.getAdminActionToken())
                        .setReason(request.getReason())
                        .setAllTrainFeeRefund(TAllTrainFeeRefund.newBuilder().build())
                        .build()))
                .thenApply(rsp -> null);
    }

    public CompletableFuture<Void> retryTrustRefund(RetryTrustRefundReqV1 request) {
        var pbReq = TRetryMoneyRefundReq.newBuilder()
                .setOrderId(request.getOrderId())
                .setAdminActionToken(request.getAdminActionToken());
        if (!Strings.isNullOrEmpty(request.getOrderRefundId())) {
            pbReq.setOrderRefundId(request.getOrderRefundId());
        }
        return FutureUtils.buildCompletableFuture(adminClientFactory.createAdminFutureStub(true)
                .retryMoneyRefund(pbReq.build()))
                .thenApply(rsp -> null);
    }

    public CompletableFuture<Void> refundCancelledHotelOrder(RefundCancelledHotelOrderReqV1 request) {
        return FutureUtils.buildCompletableFuture(adminClientFactory.createAdminFutureStub(true)
                .refundCancelledHotelOrder(TRefundCancelledHotelOrderReq.newBuilder()
                        .setOrderId(request.getOrderId())
                        .setRefundAmount(ProtoUtils.toTPrice(request.getRefundAmount()))
                        .build()))
                .thenApply(rsp -> null);
    }

    public CompletableFuture<Void> changeEventState(ChangeEventStateReqV1 request) {
        return FutureUtils.buildCompletableFuture(adminClientFactory.createAdminFutureStub(true)
                .changeEventState(TChangeEventStateReq.newBuilder()
                        .setWorkflowId(request.getWorkflowId())
                        .setEventId(request.getEventId())
                        .setNewState(EWorkflowEventState.valueOf(request.getNewEventState()))
                        .build()))
                .thenApply(rsp -> null);
    }

    public CompletableFuture<Void> sendEvent(SendEventReqV1 request) {
        return FutureUtils.buildCompletableFuture(adminClientFactory.createAdminFutureStub(true)
                .sendEventToWorkflow(TSendEventToWorkflowReq.newBuilder()
                        .setWorkflowId(request.getWorkflowId())
                        .setEventClass(request.getEventClass())
                        .setEventData(request.getEventData() != null ? ProtoUtils.toTJson(request.getEventData()) :
                                TJson.getDefaultInstance())
                        .build()))
                .thenApply(rsp -> null);
    }

    public CompletableFuture<RestoreDolphinOrderRspV1> restoreDolphinOrder(RestoreDolphinOrderReqV1 request) {
        Preconditions.checkNotNull(request.getOrderId(), "Order ID should be present");
        return FutureUtils.buildCompletableFuture(adminClientFactory.createAdminFutureStub(true)
                .restoreDolphinOrder(TRestoreDolphinOrderSecureReq.newBuilder()
                        .setOrderId(TOrderId.newBuilder().setOrderId(request.getOrderId().toString()).build())
                        .setCancel(request.getCancel() == Boolean.TRUE)
                        .build()))
                .thenApply(rsp -> RestoreDolphinOrderRspV1.builder()
                        .success(true)
                        .build());
    }

    public CompletableFuture<RegenerateVouchersRspV1> regenerateVouchers(RegenerateVouchersReqV1 request) {
        Preconditions.checkNotNull(request.getOrderId(), "Order ID should be present");
        return FutureUtils.buildCompletableFuture(adminClientFactory.createAdminFutureStub(true)
                .regenerateVouchers(TRegenerateVoucherReq.newBuilder()
                        .setOrderId(TOrderId.newBuilder().setOrderId(request.getOrderId().toString()).build())
                        .build()))
                .thenApply(rsp -> RegenerateVouchersRspV1.builder()
                        .success(true)
                        .build());
    }

    public CompletableFuture<CalculateHotelOrderRefundRspV1> calculateHotelOrderRefund(String orderId) {
        TCalculateHotelOrderRefundReq.Builder reqBuilder = TCalculateHotelOrderRefundReq.newBuilder()
                .setOrderId(TOrderId.newBuilder().setOrderId(orderId));

        return FutureUtils.buildCompletableFuture(adminClientFactory.createAdminFutureStub(true)
                .calculateHotelOrderRefund(reqBuilder.build()))
                .thenApply(adminMapper::mapCalculatedHotelOrderRefundMoney);
    }

    public CompletableFuture<Void> refundHotelOrder(RefundHotelOrderReqV1 request) {
        Preconditions.checkNotNull(request.getOrderId(), "Order ID should be present");

        return FutureUtils.buildCompletableFuture(adminClientFactory.createAdminFutureStub(true)
            .refundHotelOrder(TRefundHotelOrderReq.newBuilder()
                    .setOrderId(TOrderId.newBuilder().setOrderId(request.getOrderId().toString()).build())
                    .setRefundAmount(ProtoUtils.toTPrice(request.getRefundAmount()))
                    .setGenerateFinEvents(request.isGenerateFinEvents())
                    .setMoneyRefundMode(EMoneyRefundMode.MRM_PROMO_MONEY_FIRST)
                    .setAdminActionToken(request.getAdminActionToken())
                    .setReason(request.getReason())
                    .build())
            ).handle(this::handleEmptyResponse);
    }

    public CompletableFuture<CalculateHotelMoneyOnlyRefundRspV1> calculateHotelMoneyOnlyRefund(String orderId) {
        TCalculateMoneyOnlyRefundReq.Builder reqBuilder = TCalculateMoneyOnlyRefundReq.newBuilder()
                .setOrderId(TOrderId.newBuilder().setOrderId(orderId));

        return FutureUtils.buildCompletableFuture(adminClientFactory.createAdminFutureStub(true)
                .calculateHotelMoneyOnlyRefund(reqBuilder.build()))
                .thenApply(adminMapper::mapCalculatedHotelMoneyOnlyRefundMoney);
    }

    public CompletableFuture<Void> refundHotelMoneyOnly(RefundHotelMoneyOnlyReqV1 request) {
        Preconditions.checkNotNull(request.getOrderId(), "Order ID should be present");

        return FutureUtils.buildCompletableFuture(adminClientFactory.createAdminFutureStub(true)
                .refundHotelMoneyOnly(TRefundHotelMoneyOnlyReq.newBuilder()
                        .setOrderId(TOrderId.newBuilder().setOrderId(request.getOrderId().toString()).build())
                        .setRefundAmount(ProtoUtils.toTPrice(request.getRefundAmount()))
                        .setNewInvoiceAmount(ProtoUtils.toTPrice(request.getNewInvoiceAmount()))
                        .setRefundUserMoney(request.isRefundUserMoney())
                        .setGenerateFinEvents(request.isGenerateFinEvents())
                        .setMoneyRefundMode(EMoneyRefundMode.MRM_PROMO_MONEY_FIRST)
                        .setAdminActionToken(request.getAdminActionToken())
                        .setReason(request.getReason())
                        .build())
        ).handle(this::handleEmptyResponse);
    }

    private TListOrdersReq.Builder prepareListOrdersProtoRequest(AdminListOrdersReqV1 request) {
        TListOrdersReq.Builder requestBuilder = TListOrdersReq.newBuilder()
                .setPage(TOffsetPage.newBuilder()
                        .setOffset(request.getOffset())
                        .setLimit(request.getLimit())
                        .build())
                .setOrderIdFilter(Strings.nullToEmpty(request.getOrderIdFilter()))
                .setPrettyIdFilter(Strings.nullToEmpty(request.getPrettyIdFilter()))
                .setProviderIdFilter(Strings.nullToEmpty(request.getProviderIdFilter()))
                .setEmailFilter(Strings.nullToEmpty(request.getEmailFilter()))
                .setPhoneFilter(Strings.nullToEmpty(request.getPhoneFilter()))
                .setPurchaseTokenFilter(Strings.nullToEmpty(request.getPurchaseTokenFilter()))
                .setCardMaskFilter(Strings.nullToEmpty(request.getCardMaskFilter()))
                .setRrnFilter(Strings.nullToEmpty(request.getRrnFilter()))
                .setPassengerNamesFilter(Strings.nullToEmpty(request.getPassengerNamesFilter()))
                .setTicketNumberFilter(Strings.nullToEmpty(request.getTicketNumberFilter()))
                .setYandexUidFilter(Strings.nullToEmpty(request.getYandexUidFilter()))
                .setCarrierFilter(Strings.nullToEmpty(request.getCarrierFilter()))
                .setReferralPartnerIdFilter(Strings.nullToEmpty(request.getReferralPartnerIdFilter()));

        if (request.getCreatedAtFromFilter() != null) {
            requestBuilder.setCreatedAtFromFilter(ProtoUtils.fromInstant(request.getCreatedAtFromFilter()));
        }
        if (request.getCreatedAtToFilter() != null) {
            requestBuilder.setCreatedAtToFilter(ProtoUtils.fromInstant(request.getCreatedAtToFilter()));
        }
        if (request.getOrderTypeFilter() != null) {
            requestBuilder.setOrderTypeFilter(request.getOrderTypeFilter());
        }
        if (request.getDisplayOrderTypeFilter() != null) {
            // todo(tlg-13): TRAVELBACK-1961: should pass the values as is when the orders app is ready
            requestBuilder.setOrderTypeFilter(TravelOrdersAdminMapper
                    .DISPLAY_ORDER_TYPE_TO_ORDER_TYPE_MAPPING.get(request.getDisplayOrderTypeFilter()));
        }
        if (request.getOrderStateFilter() != null) {
            requestBuilder.setOrderStateFilter(request.getOrderStateFilter());
        }
        if (request.getPartnerFilter() != null) {
            requestBuilder.setPartnerTypeFilter(request.getPartnerFilter());
        }
        if (request.getIsBrokenFilter() != null) {
            requestBuilder.setBrokenFlagFilter(request.getIsBrokenFilter() ? EBrokenStateEnum.BS_BROKEN :
                    EBrokenStateEnum.BS_NOT_BROKEN);
        }
        if (request.getSnippets() != null) {
            requestBuilder.addAllSnippet(List.of(request.getSnippets()));
        }
        if (request.getSorters() != null) {
            requestBuilder.addAllSorter(List.of(request.getSorters()));
        }
        if (request.getPaymentScheduleTypeFilter() != null) {
            requestBuilder.setPaymentScheduleTypeFilter(request.getPaymentScheduleTypeFilter());
        }
        return requestBuilder;
    }

    public CompletableFuture<Void> modifyHotelOrderDetails(ModifyHotelOrderDetailsReqV1 request) {
        Preconditions.checkNotNull(request.getOrderId(), "Order ID should be present");

        var pbReq = TModifyHotelOrderDetailsReq.newBuilder();
        pbReq.setAdminActionToken(request.getAdminActionToken());
        var pbParams = TModifyHotelOrderParams.newBuilder();
        pbParams.setOrderId(TOrderId.newBuilder().setOrderId(request.getOrderId()).build());
        pbParams.setReason(request.getReason());
        if (request.getDates() != null) {
            pbParams.setDates(TModifyHotelOrderDatesParams.newBuilder()
                    .setCheckInDate(request.getDates().getCheckinDate().toString())
                    .setCheckOutDate(request.getDates().getCheckoutDate().toString())
                    .build());
        }
        if (request.getGuests() != null) {
            var pbGuests = TModifyHotelOrderGuestsParams.newBuilder();
            for (ModifyHotelOrderDetailsReqV1.GuestInfo guestInfo : request.getGuests().getGuests()) {
                var pbGuest = TModifyHotelOrderGuestsParams.TGuest.newBuilder();
                pbGuest.setFirstName(guestInfo.getFirstName());
                pbGuest.setLastName(guestInfo.getLastName());
                pbGuest.setChild(guestInfo.isChild());
                if (guestInfo.getAge() != null) {
                    pbGuest.setAge(Int32Value.of(guestInfo.getAge()));
                }
                pbGuests.addGuests(pbGuest);
            }
            pbParams.setGuests(pbGuests);
        }
        pbReq.setParams(pbParams);

        return FutureUtils.buildCompletableFuture(adminClientFactory.createAdminFutureStub(true)
                .modifyHotelOrderDetails(pbReq.build())
        ).handle(this::handleEmptyResponse);
    }

    public CompletableFuture<StopPartnerPaymentsRspV1> stopPartnerPayments(StopPartnerPaymentsReqV1 request) {
        TStopPaymentsReq.Builder reqBuilder = TStopPaymentsReq.newBuilder()
                .setBillingClientId(request.getBillingClientId())
                .setBillingContractId(request.getBillingContractId());

        return FutureUtils.buildCompletableFuture(adminClientFactory.createAdminFutureStub(true)
                .stopPayments(reqBuilder.build()))
                .thenApply(r -> new StopPartnerPaymentsRspV1(
                        r.getBillingClientId(),
                        r.getStopResult(),
                        r.getPaymentsEnabled()
                ));
    }

    public CompletableFuture<ResumePartnerPaymentsRspV1> resumePartnerPayments(ResumePartnerPaymentsReqV1 request) {
        TResumePaymentsReq.Builder reqBuilder = TResumePaymentsReq.newBuilder()
                .setBillingClientId(request.getBillingClientId())
                .setBillingContractId(request.getBillingContractId());

        return FutureUtils.buildCompletableFuture(adminClientFactory.createAdminFutureStub(true)
                .resumePayments(reqBuilder.build()))
                .thenApply(r -> new ResumePartnerPaymentsRspV1(
                        r.getBillingClientId(),
                        r.getResumeResult(),
                        r.getPaymentsEnabled()
                ));
    }

    public CompletableFuture<MoveHotelOrderToNewContractRspV1> moveHotelOrderToNewContract(MoveHotelOrderToNewContractReqV1 request) {
        TMoveHotelOrderToNewContractReq.Builder reqBuilder = TMoveHotelOrderToNewContractReq.newBuilder()
                .setClientId(request.getClientId())
                .setContractId(request.getContractId());

        if (request.getPayoutDate() != null) {
           reqBuilder.setPayoutDate(ProtoUtils.fromLocalDate(request.getPayoutDate()));
        }

        TOrderId.Builder tOrderId = TOrderId.newBuilder();
        if (!Strings.isNullOrEmpty(request.getOrderId())) {
            tOrderId.setOrderId(request.getOrderId());
        } else {
            tOrderId.setPrettyId(request.getPrettyId());
        }
        reqBuilder.setOrderId(tOrderId);

        return FutureUtils.buildCompletableFuture(adminClientFactory.createAdminFutureStub(false)
                .moveHotelOrderToNewContract(reqBuilder.build()))
                .thenApply(r -> new MoveHotelOrderToNewContractRspV1("ok"));
    }

    private <T> Void handleEmptyResponse(T rsp, Throwable err) {
        if (err == null) {
            return null;
        }
        Metadata metadata = Status.trailersFromThrowable(err);
        if (metadata == null) {
            throw new RuntimeException(err);
        }
        TError error = metadata.get(ServerUtils.METADATA_ERROR_KEY);
        if (error == null || !error.getCode().equals(EErrorCode.EC_FAILED_PRECONDITION)) {
            throw new RuntimeException(err);
        }
        Status status = Status.fromThrowable(err);
        throw new TravelApiBadRequestException(status.getDescription());
    }
}
