package ru.yandex.travel.api.endpoints.administrator;

import javax.validation.Valid;

import io.grpc.StatusRuntimeException;
import lombok.RequiredArgsConstructor;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.context.request.async.DeferredResult;

import ru.yandex.travel.api.config.hotels.administrator.AdministratorAdminInterfaceProvider;
import ru.yandex.travel.api.endpoints.administrator.req_rsp.ChangeHotelConnectionStateReq;
import ru.yandex.travel.api.endpoints.administrator.req_rsp.ChangeHotelConnectionStateRsp;
import ru.yandex.travel.api.endpoints.administrator.req_rsp.GetHotelConnectionRsp;
import ru.yandex.travel.api.exceptions.GrpcError;
import ru.yandex.travel.api.infrastucture.ResponseProcessor;
import ru.yandex.travel.hotels.proto.EPartnerId;

@RestController
@RequiredArgsConstructor
@RequestMapping(value = "/api/administrator/")
public class AdministratorAdminController {
    private final ResponseProcessor responseProcessor;
    private final AdministratorAdminInterfaceProvider administratorAdminInterfaceProvider;
    private final AdministratorAdminService administratorAdminService;

    @ExceptionHandler(IllegalArgumentException.class)
    @ResponseStatus(HttpStatus.BAD_REQUEST)
    public ResponseEntity<String> handleIllegalArgumentException(IllegalArgumentException e) {
        return ResponseEntity.badRequest().contentType(MediaType.TEXT_PLAIN).body(e.getMessage());
    }

    @ExceptionHandler(StatusRuntimeException.class)
    public ResponseEntity<GrpcError> handleGrpcErrors(StatusRuntimeException ex) {
        GrpcError error = GrpcError.fromGrpcStatusRuntimeException(ex);
        return ResponseEntity.status(error.getStatus()).contentType(MediaType.APPLICATION_JSON).body(error);
    }

    @RequestMapping(value = "/get_hotel_connection", method = RequestMethod.GET)
    public DeferredResult<GetHotelConnectionRsp> getHotelConnection(
            @RequestParam("partnerId") int partnerId,
            @RequestParam("hotelCode") String hotelCode) {
        return responseProcessor.replyWithFuture("AdministratorGetHotelConnectionDetails", () ->
                administratorAdminService.getHotelConnection(EPartnerId.forNumber(partnerId), hotelCode));
    }

    @RequestMapping(value = "/change_hotel_connection_state", method = RequestMethod.POST, consumes = "application/json")
    public DeferredResult<ChangeHotelConnectionStateRsp> changeHotelConnectionState(@RequestBody @Valid ChangeHotelConnectionStateReq req) {
        return responseProcessor.replyWithFuture("AdministratorChangeConnectionStatus", () ->
                administratorAdminService.changeHotelConnectionState(req));
    }
}
