package ru.yandex.travel.api.endpoints.administrator

import org.springframework.beans.factory.annotation.Autowired
import org.springframework.stereotype.Service
import ru.yandex.travel.api.config.hotels.administrator.AdministratorAdminInterfaceProvider
import ru.yandex.travel.api.endpoints.administrator.req_rsp.*
import ru.yandex.travel.api.services.orders.OrchestratorAdminClientFactory
import ru.yandex.travel.commons.concurrent.FutureUtils
import ru.yandex.travel.hotels.administrator.grpc.proto.GetHotelConnectionReq
import ru.yandex.travel.hotels.proto.EPartnerId
import ru.yandex.travel.orders.admin.proto.TGetPaymentsStateReq
import java.util.concurrent.CompletableFuture
import ru.yandex.travel.hotels.administrator.grpc.proto.ChangeHotelConnectionStateReq as ChangeHotelConnectionStateReqGrpc
import ru.yandex.travel.hotels.administrator.grpc.proto.GetHotelConnectionRsp as GetHotelConnectionRspGrpc

@Service
class AdministratorAdminService @Autowired constructor(
    private val administratorApiFactory: AdministratorAdminInterfaceProvider,
    private val orchestratorApiFactory: OrchestratorAdminClientFactory,
) {
    fun getHotelConnection(partnerId: EPartnerId, hotelCode: String): CompletableFuture<GetHotelConnectionRsp> {
        return FutureUtils.buildCompletableFuture(
            administratorApiFactory.provideInternalInterface()
                .getHotelConnection(
                    GetHotelConnectionReq.newBuilder()
                        .setPartnerId(partnerId)
                        .setHotelCode(hotelCode)
                        .build()))
            .thenCompose { hotelConnRsp ->
                if (!hotelConnRsp.hasBillingClientId()) {
                    return@thenCompose CompletableFuture.completedFuture(
                        createGetHotelConnectionRsp(hotelConnRsp, false))
                }

                return@thenCompose FutureUtils.buildCompletableFuture(
                    orchestratorApiFactory.createAdminFutureStub(false)
                        .getPaymentsState(
                            TGetPaymentsStateReq.newBuilder()
                                .setBillingClientId(hotelConnRsp.billingClientId)
                                .build())
                ).handle{ getPaymentsStateRsp, ex ->
                     createGetHotelConnectionRsp(
                         hotelConnRsp,
                         paymentsEnabled = if (ex == null) getPaymentsStateRsp.paymentsEnabled else false)
                }
            }
    }

    private fun createGetHotelConnectionRsp(hotelConn: GetHotelConnectionRspGrpc, paymentsEnabled: Boolean): GetHotelConnectionRsp {
        return GetHotelConnectionRsp(
            hotelConnectionState = HotelConnectionState.convertFromProto(hotelConn.connectionState),
            permalink = if (hotelConn.hasPermalink()) hotelConn.permalink else null,
            slug = if (hotelConn.hasPermalink()) "permalink/${hotelConn.permalink}" else "",
            stTicket = hotelConn.stTicket,
            paperAgreement = hotelConn.paperAgreement,
            paymentsEnabled = paymentsEnabled,
            hotelName = hotelConn.hotelName,
            billingClientId = if (hotelConn.hasBillingClientId()) hotelConn.billingClientId else null,
            billingContractId = if (hotelConn.hasBillingClientId()) hotelConn.billingContractId else null,
            unpublishedReason = UnpublishedReason.convertFromProto(hotelConn.unpublishedReason),
            accountantEmails = hotelConn.accountantEmailsList,
            contractPersonEmails = hotelConn.contractPersonEmailsList,
        )
    }

    fun changeHotelConnectionState(req: ChangeHotelConnectionStateReq): CompletableFuture<ChangeHotelConnectionStateRsp> {
        val reqBuilder = ChangeHotelConnectionStateReqGrpc.newBuilder()
            .setPartnerId(EPartnerId.forNumber(req.partnerId))
            .setHotelCode(req.hotelCode)
            .setConnectionState(HotelConnectionState.convertToProto(req.connectionState))
        req.unpublishedReason?.let{reqBuilder.unpublishedReason = UnpublishedReason.convertToProto(req.unpublishedReason)}

        return FutureUtils.buildCompletableFuture(
            administratorApiFactory.provideInternalInterface().changeHotelConnectionState(reqBuilder.build()))
            .thenApply { response -> ChangeHotelConnectionStateRsp(
                previousState = HotelConnectionState.convertFromProto(response.previousState),
                newState = HotelConnectionState.convertFromProto(response.newState),
            )}
    }
}
