package ru.yandex.travel.api.endpoints.avia.landings;

import java.util.concurrent.CompletableFuture;

import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.PropertyNamingStrategy;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Component;

import ru.yandex.travel.api.endpoints.avia.landings.exceptions.LandingNotFoundException;
import ru.yandex.travel.api.endpoints.avia.landings.req_rsp.LandingRouteReq;
import ru.yandex.travel.api.endpoints.avia.landings.req_rsp.LandingRouteRsp;
import ru.yandex.travel.api.models.crosslinks.CrosslinksHotelsBlock;
import ru.yandex.travel.api.models.hotels.converters.BoundingBoxSerializersModule;
import ru.yandex.travel.api.models.hotels.converters.PermalinkSerializersModule;
import ru.yandex.travel.api.services.avia.api_gateway.AviaApiGatewayService;
import ru.yandex.travel.api.services.crosslinks.CrosslinksService;
import ru.yandex.travel.api.services.hotels.slug.RegionSlugService;
import ru.yandex.travel.commons.http.CommonHttpHeaders;
import ru.yandex.travel.commons.http.apiclient.HttpApiException;
import ru.yandex.travel.credentials.UserCredentials;

@Component
@RequiredArgsConstructor
@Slf4j
public class LandingRouteImpl {
    private final AviaApiGatewayService apiGatewayService;
    private final RegionSlugService regionSlugService;
    private final CrosslinksService crosslinksService;

    public CompletableFuture<LandingRouteRsp> getLandingRoute(LandingRouteReq request, CommonHttpHeaders commonHttpHeaders,
                                                              UserCredentials userCredentials) {
        var landingRouteFuture = apiGatewayService.landingRoute(
                request.getFromSlug(),
                request.getToSlug(),
                request.getNationalVersion(),
                request.getLang()
        ).exceptionally(e -> {
            if (e.getCause() != null) {
                if (e.getCause() instanceof HttpApiException
                        && ((HttpApiException) e.getCause()).getStatusCode() == HttpStatus.NOT_FOUND.value()) {
                    throw new LandingNotFoundException(e.getCause().getMessage());
                } else {
                    throw new RuntimeException(e.getCause());
                }
            } else {
                throw new RuntimeException(e);
            }
        });

        var hotelsBlockFuture = getHotelsBlock(
                request,
                commonHttpHeaders,
                userCredentials
        ).exceptionally(e -> {
            log.error("Unable to get hotelsBlock");
            return null;
        });

        return CompletableFuture.allOf(landingRouteFuture, hotelsBlockFuture).thenApply(ignored -> {
            var landingRoute = landingRouteFuture.join();
            var hotelsBlock = hotelsBlockFuture.join();

            ArrayNode blocks = (ArrayNode) landingRoute.getBlocks();

            if (hotelsBlock != null && hotelsBlock.getData().isHasData()) {
                try {
                    ObjectMapper mapper = new ObjectMapper();
                    mapper.setPropertyNamingStrategy(PropertyNamingStrategy.LOWER_CAMEL_CASE);
                    mapper.registerModule(new PermalinkSerializersModule());
                    mapper.registerModule(new BoundingBoxSerializersModule());
                    ObjectNode hotelNode = mapper.createObjectNode();
                    hotelNode.set("type", mapper.valueToTree(hotelsBlock.getType()));
                    hotelNode.set("data", mapper.valueToTree(hotelsBlock.getData()));
                    blocks.add(hotelNode);
                } catch (Exception e) {
                    log.error("Unable to add hotelsBlock", e);
                }
            }

            return LandingRouteRsp.builder()
                    .seoInfo(landingRoute.getSeoInfo())
                    .blocks(blocks)
                    .build();
        });
    }

    private CompletableFuture<CrosslinksHotelsBlock> getHotelsBlock(LandingRouteReq request,
                                                                          CommonHttpHeaders commonHttpHeaders,
                                                                          UserCredentials userCredentials) {
        int geoId;
        try {
            geoId = regionSlugService.getGeoIdBySlug(request.getToSlug());
        } catch (Exception e) {
            log.error("Unable to get geoId by slug {}.", request.getToSlug(), e);
            return CompletableFuture.completedFuture(null);
        }

        return crosslinksService.getHotelsBlock(geoId, request.getNationalVersion(), commonHttpHeaders, userCredentials);
    }
}
