package ru.yandex.travel.api.endpoints.avia_booking_flow;

import java.util.ArrayList;
import java.util.NoSuchElementException;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import javax.money.MonetaryException;
import javax.validation.ConstraintViolation;
import javax.validation.ValidationException;
import javax.validation.Validator;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.JsonNode;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import io.grpc.Metadata;
import io.grpc.Status;
import io.grpc.StatusRuntimeException;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.boot.autoconfigure.condition.ConditionalOnBean;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.context.request.async.DeferredResult;

import ru.yandex.avia.booking.service.dto.AeroflotStateDTO;
import ru.yandex.avia.booking.service.dto.CompositeOrderStateDTO;
import ru.yandex.avia.booking.service.dto.OrderDTO;
import ru.yandex.avia.booking.service.dto.VariantCheckResponseDTO;
import ru.yandex.avia.booking.service.dto.VariantCheckToken;
import ru.yandex.avia.booking.service.dto.form.CreateOrderForm;
import ru.yandex.avia.booking.service.dto.form.InitOrderPaymentForm;
import ru.yandex.travel.api.config.avia.AviaBookingConfiguration;
import ru.yandex.travel.api.endpoints.avia_booking_flow.req_rsp.AviaOrderPaymentConfirmationReqV1;
import ru.yandex.travel.api.endpoints.generic_booking_flow.model.GetOrderRequestSource;
import ru.yandex.travel.api.endpoints.trips.req_rsp.SelectOrdersReqV1;
import ru.yandex.travel.api.infrastucture.ResponseProcessor;
import ru.yandex.travel.api.services.avia.fares.AviaFareRulesException;
import ru.yandex.travel.api.services.avia.fares.AviaUnknownFareFamilyException;
import ru.yandex.travel.api.services.avia.legacy.AviaLegacyJsonMapper;
import ru.yandex.travel.api.services.avia.orders.AviaOrderService;
import ru.yandex.travel.api.services.avia.variants.AviaVariantNotSupportedException;
import ru.yandex.travel.api.services.avia.variants.AviaVariantService;
import ru.yandex.travel.api.services.avia.variants.AviaVariantsNotFoundException;
import ru.yandex.travel.api.services.common.RetryStrategyExceptionHelpers;
import ru.yandex.travel.api.services.trips.TripsProvider;
import ru.yandex.travel.commons.experiments.ExperimentDataProvider;
import ru.yandex.travel.commons.experiments.OrderExperiments;
import ru.yandex.travel.commons.grpc.ServerUtils;
import ru.yandex.travel.commons.http.CommonHttpHeaders;
import ru.yandex.travel.credentials.UserCredentials;
import ru.yandex.travel.komod.trips.common.TripsUtils;

@RestController
@RequestMapping(value = "/api/avia_booking_flow/")
@RequiredArgsConstructor
@ConditionalOnBean(AviaBookingConfiguration.class)
@Slf4j
public class AviaBookingFlowController {
    private final ResponseProcessor responseProcessor;
    private final AviaVariantService variantService;
    private final AviaOrderService orderService;
    private final Validator validator;
    private final TripsProvider tripsProvider;
    private final ExperimentDataProvider experimentDataProvider;

    @PostMapping("/v1/variants/availability_check")
    public DeferredResult<VariantCheckResponseDTO> checkVariantAndGetUrl(@RequestBody JsonNode jsonBody) {
        return responseProcessor.replyWithFuture("AviaBookingFlowCheckVariantAndGetUrl",
                () -> variantService.checkAvailability(jsonBody)
                        .thenApply(this::toLegacyJson));
    }

    @GetMapping("/v1/variants")
    public DeferredResult<JsonNode> getVariantInfo(@RequestParam String id) {
        VariantCheckToken token = VariantCheckToken.fromRaw(id);
        return responseProcessor.replyWithFuture("AviaBookingFlowGetVariantInfo",
                () -> variantService.getVariantInfoFuture(token.getAvailabilityCheckId()));
    }

    // todo(tlg-13) temporary WA for transition from GET to POST
    @RequestMapping(value = "/v1/variants/refresh", method = {RequestMethod.GET, RequestMethod.POST})
    public DeferredResult<JsonNode> refresh(@RequestParam String id) {
        VariantCheckToken token = VariantCheckToken.fromRaw(id);
        return responseProcessor.replyWithFuture("AviaBookingFlowRefresh",
                () -> variantService.reCheckAvailability(token.getAvailabilityCheckId()));
    }

    @PostMapping("/v1/orders")
    public DeferredResult<OrderDTO> createOrder(@RequestBody JsonNode createOrderFormJson) {
        CreateOrderForm createOrderForm = bindJson(createOrderFormJson, CreateOrderForm.class);
        validate(createOrderForm, "CreateOrderForm");

        setUserIP(createOrderForm);
        setUserAgent(createOrderForm);

        return responseProcessor.replyWithFutureRetrying("AviaBookingFlowCreateOrder",
                () -> orderService.createOrder(createOrderForm, currentUser()).thenApply(this::toLegacyJson),
                RetryStrategyExceptionHelpers.defaultStatusUnavailableRetryStrategy()
        );
    }

    private static void setUserIP(CreateOrderForm createOrderForm) {
        CommonHttpHeaders headers = CommonHttpHeaders.get();
        String userIP = headers.getUserIP();
        if (!Strings.isNullOrEmpty(userIP)) {
            createOrderForm.setUserIp(userIP);
        }
    }

    private static void setUserAgent(CreateOrderForm createOrderForm) {
        CommonHttpHeaders headers = CommonHttpHeaders.get();
        String userAgent = headers.getRealUserAgent();
        if (!Strings.isNullOrEmpty(userAgent)) {
            createOrderForm.setUserAgent(userAgent);
        }
    }

    @GetMapping("/v1/orders/info")
    public DeferredResult<OrderDTO> getOrder(@RequestParam UUID id, @RequestParam GetOrderRequestSource source) {
        var userCredentials = UserCredentials.get();
        var orderExperiments = experimentDataProvider.getInstance(OrderExperiments.class, CommonHttpHeaders.get());
        return responseProcessor.replyWithFutureRetrying(
                "AviaBookingFlowGetOrder", () -> {
                    var needRefreshState = orderExperiments.isRefreshAeroflotOrderState() && source == GetOrderRequestSource.ORDER_PAGE;
                    var getOrderByIdFuture = orderService.getOrder(id, needRefreshState);
                    var getTripIdByOrderIdFuture = tripsProvider.getTripIdByOrderId(id, userCredentials);
                    return CompletableFuture.allOf(getOrderByIdFuture, getTripIdByOrderIdFuture)
                            .thenApply(__ -> {
                                var order = getOrderByIdFuture.join();
                                TripsUtils.fillTripId(order, getTripIdByOrderIdFuture.join());
                                return order;
                            })
                            .thenApply(this::toLegacyJson);
                },
                RetryStrategyExceptionHelpers.defaultStatusUnavailableRetryStrategy()
        );
    }

    @GetMapping("/v1/orders/state")
    public DeferredResult<CompositeOrderStateDTO> getCompositeState(@RequestParam UUID id) {
        return responseProcessor.replyWithFutureRetrying("AviaBookingFlowGetCompositeState",
                () -> orderService.getCompositeOrderState(id).thenApply(this::toLegacyJson),
                RetryStrategyExceptionHelpers.defaultStatusUnavailableRetryStrategy()
        );
    }

    @PostMapping("/v1/orders/aeroflotState")
    public DeferredResult<ArrayList<AeroflotStateDTO>> getAeroflotState(@RequestBody JsonNode jsonNode) {
        SelectOrdersReqV1 req = bindJson(jsonNode, SelectOrdersReqV1.class);
        if (req == null || req.getOrderIds().isEmpty()) {
            throw new ValidationException("OrderIds is null or empty");
        }
        return responseProcessor.replyWithFutureRetrying("AviaBookingFlowGetAeroflotState",
                () -> orderService.getAeroflotState(req.getOrderIds()).thenApply(this::toLegacyJson),
                RetryStrategyExceptionHelpers.defaultStatusUnavailableRetryStrategy()
        );
    }

    @PostMapping("/v1/orders/payment")
    public DeferredResult<Void> initPaymentAsync(@RequestParam UUID id,
                                                 @RequestBody JsonNode initOrderPaymentFormJson) {
        InitOrderPaymentForm initOrderPaymentForm = bindJson(initOrderPaymentFormJson, InitOrderPaymentForm.class);
        var orderExperiments = experimentDataProvider.getInstance(OrderExperiments.class, CommonHttpHeaders.get());
        initOrderPaymentForm.setEnableNewCommonPaymentForm(orderExperiments.isNewCommonPaymentWebForm());
        validate(initOrderPaymentForm, "InitOrderPaymentForm");
        return responseProcessor.replyWithFutureRetrying("AviaBookingFlowInitPaymentAsync",
                () -> orderService.startPaymentAsync(id, initOrderPaymentForm),
                RetryStrategyExceptionHelpers.defaultStatusUnavailableRetryStrategy()
        );
    }

    @PostMapping("/v1/orders/confirmation_complete")
    public DeferredResult<Void> confirmationComplete(@RequestBody AviaOrderPaymentConfirmationReqV1 confirmation) {
        return responseProcessor.replyWithFutureRetrying("AviaBookingFlowInitPaymentAsync",
                // todo(tlg-13): a mock impl, should be completed in TRAVELBACK-1529
                () -> CompletableFuture.completedFuture(null),
                RetryStrategyExceptionHelpers.defaultStatusUnavailableRetryStrategy()
        );
    }

    private UserCredentials currentUser() {
        return Preconditions.checkNotNull(UserCredentials.get(), "UserCredentials are not set");
    }

    private <T> T bindJson(JsonNode data, Class<T> type) {
        try {
            return AviaLegacyJsonMapper.objectMapper.treeToValue(data, type);
        } catch (JsonProcessingException e) {
            throw new RuntimeException(e);
        }
    }

    @SuppressWarnings("unchecked")
    private <T> T toLegacyJson(Object object) {
        try {
            // we do this pseudo cast to an incompatible type only to keep the simple DeferredResult<PojoType> api
            // method signatures
            return (T) AviaLegacyJsonMapper.objectMapper.writeValueAsString(object);
        } catch (JsonProcessingException e) {
            throw new RuntimeException(e);
        }
    }

    private void validate(Object object, String context) {
        Set<ConstraintViolation<Object>> errors = validator.validate(object);
        if (errors != null && !errors.isEmpty()) {
            String validationErrorMessages = errors.stream()
                    .map(e -> String.format("%s.%s: %s", e.getRootBeanClass().getSimpleName(), e.getPropertyPath(),
                            e.getMessage()))
                    .collect(Collectors.joining("; "));
            throw new ValidationException("Invalid " + context + ": " + validationErrorMessages);
        }
    }

    @ExceptionHandler(value = {
            ValidationException.class,
            IllegalArgumentException.class,
            IllegalStateException.class,
            NoSuchElementException.class,
            MonetaryException.class,
    })
    public ResponseEntity<Object> handleException(Exception e) {
        return ResponseEntity.badRequest().contentType(MediaType.TEXT_PLAIN).body(e.getMessage());
    }

    @ExceptionHandler
    public ResponseEntity<Object> handleException(AviaVariantsNotFoundException e) {
        return ResponseEntity.status(HttpStatus.NOT_FOUND).contentType(MediaType.TEXT_PLAIN).body(e.getMessage());
    }

    @ExceptionHandler({
            AviaFareRulesException.class,
            AviaUnknownFareFamilyException.class,
            AviaVariantNotSupportedException.class
    })
    public ResponseEntity<Object> handleUnsupportedException(Exception e) {
        return ResponseEntity.status(HttpStatus.UNPROCESSABLE_ENTITY).contentType(MediaType.TEXT_PLAIN).body(e.getMessage());
    }

    @ExceptionHandler
    public ResponseEntity<Object> handle(StatusRuntimeException e) {
        Metadata metadata = e.getTrailers();
        log.info("GRPC exception: e.msg={}, statusDetails={}",
                e.getMessage(), metadata != null ? metadata.get(ServerUtils.METADATA_ERROR_KEY) : null);
        if (e.getStatus().getCode() == Status.Code.UNAVAILABLE) {
            return ResponseEntity.status(HttpStatus.SERVICE_UNAVAILABLE).contentType(MediaType.TEXT_PLAIN).body(e.getMessage());
        } else {
            return ResponseEntity.badRequest().contentType(MediaType.TEXT_PLAIN).body(e.getMessage());
        }
    }
}
