package ru.yandex.travel.api.endpoints.booking_flow;

import io.grpc.StatusRuntimeException;
import lombok.RequiredArgsConstructor;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.context.request.async.DeferredResult;

import ru.yandex.travel.api.endpoints.booking_flow.model.HotelOrderDto;
import ru.yandex.travel.api.endpoints.booking_flow.model.OrderStatus;
import ru.yandex.travel.api.exceptions.GrpcError;
import ru.yandex.travel.api.infrastucture.ResponseProcessor;
import ru.yandex.travel.api.services.common.RetryStrategyExceptionHelpers;
import ru.yandex.travel.api.services.orders.OrdersNoAuthService;

@RestController
@RequestMapping(value = "/api/booking_flow/v1/admin")
@RequiredArgsConstructor
public class BookingAdminController {
    @ExceptionHandler(IllegalArgumentException.class)
    @ResponseStatus(HttpStatus.BAD_REQUEST)
    public ResponseEntity<String> handleIllegalArgumentException(IllegalArgumentException e) {
        return ResponseEntity.badRequest().contentType(MediaType.TEXT_PLAIN).body(e.getMessage());
    }

    @ExceptionHandler(StatusRuntimeException.class)
    public ResponseEntity<GrpcError> handleGrpcErrors(StatusRuntimeException ex) {
        GrpcError error = GrpcError.fromGrpcStatusRuntimeException(ex);
        return ResponseEntity.status(error.getStatus()).contentType(MediaType.APPLICATION_JSON).body(error);
    }

    private final ResponseProcessor responseProcessor;
    private final OrdersNoAuthService ordersNoAuthService;

    @RequestMapping(value = "/get_order", method = RequestMethod.GET, produces = "application/json")
    public DeferredResult<ResponseEntity<HotelOrderDto>> getOrder(@RequestParam("id") String id) {
        return responseProcessor.replyWithFutureRetrying("BookingFlowAdminGetOrder",
                () -> ordersNoAuthService.getHotelOrderInfo(id)
                .thenApply(order -> {
                    if (order == null) {
                        return ResponseEntity.notFound().build();
                    } else {
                        // This method is being called by voucher-generator service only.
                        // That service should not actually care about the status of the order, however it fails
                        // if the order status is "payment failed", but it may likely have this state
                        // if the deferred payment has failed.
                        // So the if-clause below should return the status to confirmed to trick the voucher-generator.
                        if (order.getStatus() == OrderStatus.PAYMENT_FAILED) {
                            order.setStatus(OrderStatus.CONFIRMED);
                        }
                    }
                    return ResponseEntity.ok(order);
                }), RetryStrategyExceptionHelpers.defaultStatusUnavailableRetryStrategy()
        );
    }
}
