package ru.yandex.travel.api.endpoints.booking_flow;

import java.time.Instant;
import java.util.List;
import java.util.UUID;
import java.util.concurrent.CompletableFuture;

import javax.validation.Valid;

import io.grpc.StatusRuntimeException;
import io.swagger.annotations.ApiOperation;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.RequestParam;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.context.request.async.DeferredResult;

import ru.yandex.travel.api.endpoints.booking_flow.model.CreateV1HotelOrderRequest;
import ru.yandex.travel.api.endpoints.booking_flow.model.HotelOrderDto;
import ru.yandex.travel.api.endpoints.booking_flow.model.OfferDto;
import ru.yandex.travel.api.endpoints.booking_flow.model.OrderStatus;
import ru.yandex.travel.api.endpoints.booking_flow.model.PdfLink;
import ru.yandex.travel.api.endpoints.booking_flow.model.RefundCalculation;
import ru.yandex.travel.api.endpoints.booking_flow.model.RefundRequest;
import ru.yandex.travel.api.endpoints.booking_flow.model.StartInvoicePaymentRequest;
import ru.yandex.travel.api.endpoints.booking_flow.model.StartOrderPaymentRequest;
import ru.yandex.travel.api.endpoints.booking_flow.req_rsp.CancelOrderReqV1;
import ru.yandex.travel.api.endpoints.booking_flow.req_rsp.EstimateDiscountReqV1;
import ru.yandex.travel.api.endpoints.booking_flow.req_rsp.EstimateDiscountRspV1;
import ru.yandex.travel.api.endpoints.booking_flow.req_rsp.OrderStatusRspV1;
import ru.yandex.travel.api.endpoints.hotels_portal.ExtraVisitAndUserParamsUtils;
import ru.yandex.travel.api.exceptions.ExpiredTravelTokenException;
import ru.yandex.travel.api.exceptions.GrpcError;
import ru.yandex.travel.api.exceptions.HotelInfoNotFoundException;
import ru.yandex.travel.api.exceptions.InputError;
import ru.yandex.travel.api.exceptions.InvalidInputException;
import ru.yandex.travel.api.infrastucture.ResponseProcessor;
import ru.yandex.travel.api.services.common.RetryStrategyExceptionHelpers;
import ru.yandex.travel.api.services.hotels_booking_flow.BookingFlowContext;
import ru.yandex.travel.api.services.hotels_booking_flow.HotelOrdersService;
import ru.yandex.travel.api.services.hotels_booking_flow.OfferService;
import ru.yandex.travel.api.services.hotels_booking_flow.models.HotelOrder;
import ru.yandex.travel.api.services.promo.YandexPlusService;
import ru.yandex.travel.api.services.trips.TripsProvider;
import ru.yandex.travel.commons.experiments.ExperimentDataProvider;
import ru.yandex.travel.commons.experiments.OrderExperiments;
import ru.yandex.travel.commons.http.CommonHttpHeaders;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.credentials.UserCredentials;
import ru.yandex.travel.hotels.common.token.TokenCodec;
import ru.yandex.travel.hotels.common.token.TravelToken;
import ru.yandex.travel.hotels.models.booking_flow.OfferState;
import ru.yandex.travel.komod.trips.common.TripsUtils;


@RestController
@RequestMapping(value = "/api/booking_flow")
@RequiredArgsConstructor
@Slf4j
public class BookingFlowController {
    @ExceptionHandler(IllegalArgumentException.class)
    @ResponseStatus(HttpStatus.BAD_REQUEST)
    public ResponseEntity<String> handleIllegalArgumentException(IllegalArgumentException e) {
        return ResponseEntity.badRequest().contentType(MediaType.TEXT_PLAIN).body(e.getMessage());
    }

    @ExceptionHandler(StatusRuntimeException.class)
    public ResponseEntity<GrpcError> handleGrpcErrors(StatusRuntimeException ex) {
        GrpcError error = GrpcError.fromGrpcStatusRuntimeException(ex);
        return ResponseEntity.status(error.getStatus()).contentType(MediaType.APPLICATION_JSON).body(error);
    }

    private final ResponseProcessor responseProcessor;
    private final OfferService offerService;
    private final HotelOrdersService ordersService;
    private final TokenCodec tokenCodec;
    private final DtoMapper dtoConverter;
    private final BookingFlowOfferInvalidator bookingFlowOfferInvalidator;
    private final ExperimentDataProvider experimentDataProvider;
    private final TripsProvider tripsProvider;
    private final YandexPlusService yandexPlusService;

    @RequestMapping(value = "/v1/get_order_info_by_token", method = RequestMethod.GET, produces = "application/json")
    public DeferredResult<ResponseEntity<OfferDto>> getOrderInfoByToken(
            @RequestParam("token") String token,
            @RequestParam(value = "label", required = false) String label,
            @RequestParam("customerIp") String customerIp,
            @RequestParam("customerUserAgent") String customerUserAgent,
            @RequestParam(value = "debug", defaultValue = "false") boolean debug
    ) {
        BookingFlowContext context = BookingFlowContext.builder()
                .createdAt(Instant.now())
                .token(token)
                .offerLabel(label)
                .deduplicationKey(ProtoUtils.randomId())
                .stage(BookingFlowContext.Stage.GET_OFFER)
                .userIp(customerIp)
                .userAgent(customerUserAgent)
                .headers(CommonHttpHeaders.get())
                .userCredentials(UserCredentials.get())
                .build();
        CompletableFuture<Integer> yandexPlusBalanceFuture = yandexPlusService.getYandexPlusBalanceWithoutException(
                context.getUserCredentials().getPassportId(),
                context.getUserCredentials().getUserIp(),
                YandexPlusService.DEFAULT_PLUS_CURRENCY
        );

        return responseProcessor.replyWithFuture("BookingFlowV1GetOrderInfoByToken",
                () -> yandexPlusBalanceFuture.thenCompose(yandexPlusBalance ->
                        offerService.getOffer(context, yandexPlusBalance).thenApply(offer -> {
                            OfferDto info = dtoConverter.buildOfferDto(offer);
                            var pageParams = ExtraVisitAndUserParamsUtils.initParamsMap();
                            var basicHotelInfo = info.getBasicHotelInfo();
                            if (basicHotelInfo != null) {
                                ExtraVisitAndUserParamsUtils.fillPermalink(pageParams, basicHotelInfo.getPermalink());
                            }
                            ExtraVisitAndUserParamsUtils.fillSearchParams(pageParams, info.getRequestInfo());
                            ExtraVisitAndUserParamsUtils.fillPartner(pageParams, info.getPartnerId());
                            ExtraVisitAndUserParamsUtils.fillBoolean(pageParams, "deferredPaymentAvailable", info.getDeferredPaymentSchedule() != null);
                            info.setExtraVisitAndUserParams(ExtraVisitAndUserParamsUtils.createForBookingPage(pageParams));
                            if (debug) {
                                info.setPartnerHotelInfo(null);
                                info.setPartnerRoomInfo(null);
                            }
                            if (offer.checkOfferState() != OfferState.READY) {
                                bookingFlowOfferInvalidator.invalidate(context);
                            }
                            switch (offer.checkOfferState()) {
                                case MISSING_DATA:
                                    throw new HotelInfoNotFoundException(offer.getMetaInfo().getToken(), "Some data is " +
                                            "missing");
                                case PRICE_CONFLICT:
                                    return ResponseEntity.status(HttpStatus.CONFLICT).body(info);
                                case READY:
                                    return ResponseEntity.ok(info);
                                default:
                                    throw new AssertionError("We should not be here");
                            }
                        })));
    }

    @RequestMapping(value = "/v1/get_order_status", method = RequestMethod.GET, produces = "application/json")
    @ApiOperation("Получение статуса заказа (используется для поллинга)")
    public DeferredResult<OrderStatusRspV1> getOrderStatus(@RequestParam("id") String id) {
        return responseProcessor.replyWithFutureRetrying("BookingFlowGetOrderStatus",
                () -> ordersService.getOrderStatus(id),
                RetryStrategyExceptionHelpers.defaultStatusUnavailableRetryStrategy()
        );
    }


    @RequestMapping(value = "/v1/create_order", method = RequestMethod.POST, consumes = "application/json",
            produces = "application/json")
    public DeferredResult<ResponseEntity<HotelOrderDto>> createOrder(@RequestBody @Valid CreateV1HotelOrderRequest request) {
        BookingFlowContext context = BookingFlowContext.builder()
                .createdAt(Instant.now())
                .token(request.getToken())
                .offerLabel(request.getLabel())
                .deduplicationKey(request.getSessionKey())
                .stage(BookingFlowContext.Stage.CREATE_ORDER)
                .userIp(request.getCustomerIp())
                .userAgent(request.getCustomerUserAgent())
                .userCredentials(UserCredentials.get())
                .orderCreationData(dtoConverter.buildOrderCreationData(request))
                .promoCodes(request.getPromoCodes())
                .appliedPromoCampaigns(request.getAppliedPromoCampaigns())
                .subscriptionParams(request.getSubscriptionParams())
                .headers(CommonHttpHeaders.get())
                .paymentTestContextToken(request.getPaymentTestContextToken())
                .build();

        return responseProcessor.replyWithFutureRetrying("BookingFlowV1CreateOrder",
                () -> ordersService.createOrder(context).thenApply(newOrder -> {
                    HotelOrderDto result = dtoConverter.buildOrderDto(newOrder, false);
                    if (newOrder.getOfferInfo().checkOfferState() != OfferState.READY) {
                        bookingFlowOfferInvalidator.invalidate(context);
                    }
                    switch (newOrder.getOfferInfo().checkOfferState()) {
                        case MISSING_DATA:
                            return ResponseEntity.status(HttpStatus.NOT_FOUND).body(result);
                        case PRICE_CONFLICT:
                            return ResponseEntity.status(HttpStatus.CONFLICT).body(result);
                        case READY:
                            if (result.getStatus() == OrderStatus.FAILED) {
                                return ResponseEntity.status(HttpStatus.CONFLICT).body(result);
                            } else {
                                return ResponseEntity.ok(result);
                            }
                        default:
                            throw new AssertionError("We should not be here");
                    }
                }), RetryStrategyExceptionHelpers.defaultStatusUnavailableRetryStrategy(),
                null,
                request.getSessionKey()
        );
    }

    @RequestMapping(value = "/v1/estimate_discount", method = RequestMethod.POST,
            produces = "application/json", consumes = "application/json")
    @ApiOperation("Предварительный расчет скидки по промокоду")
    public DeferredResult<EstimateDiscountRspV1> estimateDiscount(@RequestBody @Valid EstimateDiscountReqV1 request) {
        BookingFlowContext context = BookingFlowContext.builder()
                .createdAt(Instant.now())
                .token(request.getToken())
                .offerLabel(request.getLabel())
                .deduplicationKey(request.getSessionKey())
                .stage(BookingFlowContext.Stage.ESTIMATE_DISCOUNT)
                .userIp(request.getCustomerIp())
                .userAgent(request.getCustomerUserAgent())
                .userCredentials(UserCredentials.get())
                .orderCreationData(dtoConverter.buildOrderCreationData(request))
                .headers(CommonHttpHeaders.get())
                .promoCodes(request.getPromoCodes())
                .appliedPromoCampaigns(request.getAppliedPromoCampaigns())
                .build();
        return responseProcessor.replyWithFutureRetrying("BookingFlowV1EstimateDiscount",
                () -> ordersService.estimateDiscount(context).thenApply(dtoConverter::buildEstimateDiscountDto),
                RetryStrategyExceptionHelpers.defaultStatusUnavailableRetryStrategy()
        );
    }

    @ExceptionHandler({InvalidInputException.class})
    @ResponseStatus(HttpStatus.BAD_REQUEST)
    public List<InputError> handleInputErrors(InvalidInputException ex) {
        return ex.getErrors();
    }

    @ExceptionHandler({ExpiredTravelTokenException.class})
    @ResponseStatus(HttpStatus.NOT_FOUND)
    public ResponseEntity<ExpiredTravelTokenException.TokenResponse404> handleExpiredTravelToken(ExpiredTravelTokenException ex) {
        return ex.getResponseEntity();
    }

    @ExceptionHandler({HotelInfoNotFoundException.class})
    @ResponseStatus(HttpStatus.NOT_FOUND)
    public ResponseEntity<ExpiredTravelTokenException.TokenResponse404> handleHotelInfoNotFound(HotelInfoNotFoundException ex) {
        var token = ex.getToken();
        TravelToken tokenObject = tokenCodec.decode(token);
        return ResponseEntity.status(HttpStatus.NOT_FOUND)
                .body(new ExpiredTravelTokenException.TokenResponse404(tokenObject, ex.getMessage()));
    }


    @RequestMapping(value = "/v1/get_order", method = RequestMethod.GET, produces = "application/json")
    public DeferredResult<ResponseEntity<HotelOrderDto>> getOrder(@RequestParam("id") String id,
                                                                  @RequestParam(value = "debug", defaultValue =
                                                                          "false") boolean debug) {

        var userCredentials = UserCredentials.get();
        return responseProcessor.replyWithFutureRetrying("BookingFlowV1GetOrder",
                () -> {
                    var getOrderByIdFuture = ordersService.getOrderById(id)
                            .thenApply(order -> mapHotelOrderDto(debug, order));
                    var getTripIdByOrderIdFuture = tripsProvider.getTripIdByOrderId(
                            UUID.fromString(id), userCredentials);
                    return CompletableFuture.allOf(getOrderByIdFuture, getTripIdByOrderIdFuture)
                            .thenApply(ignored -> {
                                var oldOrder = getOrderByIdFuture.join();
                                TripsUtils.fillTripId(oldOrder, getTripIdByOrderIdFuture.join());
                                return oldOrder;
                            })
                            .thenApply(oldOrder -> {
                                if (oldOrder == null) {
                                    return ResponseEntity.notFound().build();
                                }
                                return ResponseEntity.ok(oldOrder);
                            });
                },
                RetryStrategyExceptionHelpers.defaultStatusUnavailableRetryStrategy());
    }

    private HotelOrderDto mapHotelOrderDto(boolean debug, HotelOrder order) {
        if (order == null) {
            return null;
        }
        HotelOrderDto oldOrder = dtoConverter.buildOrderDto(order, false);
        if (debug) {
            oldOrder.getOrderInfo().setPartnerRoomInfo(null);
            oldOrder.getOrderInfo().setPartnerHotelInfo(null);
        }
        return oldOrder;
    }

    @RequestMapping(value = "/v1/start_payment", method = RequestMethod.POST, produces = "application/json",
            consumes = "application/json")
    public DeferredResult<Void> startOrderPayment(@RequestBody StartOrderPaymentRequest request) {
        return responseProcessor.replyWithFutureRetrying(
                "BookingFlowV1StartPayment",
                () -> ordersService.payOrder(
                        request.getOrderId(),
                        request.getReturnUrl(),
                        request.getCustomerSource(),
                        request.getPaymentTestContextToken(),
                        experimentDataProvider.getInstance(OrderExperiments.class, CommonHttpHeaders.get())
                ),
                RetryStrategyExceptionHelpers.defaultStatusUnavailableRetryStrategy()
        );
    }

    @RequestMapping(value = "/v1/start_invoice_payment", method = RequestMethod.POST, produces = "application/json",
            consumes = "application/json")
    public DeferredResult<Void> startInvoicePayment(@RequestBody StartInvoicePaymentRequest request) {
        return responseProcessor.replyWithFutureRetrying(
                "BookingFlowV1StartInvoicePayment",
                () -> ordersService.payInvoice(
                        request.getOrderId(),
                        request.getInvoiceId(),
                        request.getReturnUrl(),
                        request.getCustomerSource(),
                        request.getPaymentTestContextToken(),
                        experimentDataProvider.getInstance(OrderExperiments.class, CommonHttpHeaders.get())
                ),
                RetryStrategyExceptionHelpers.defaultStatusUnavailableRetryStrategy()
        );
    }

    @RequestMapping(value = "/v1/calculate_refund", method = RequestMethod.GET, produces = "application/json")
    public DeferredResult<RefundCalculation> calculateRefund(@RequestParam(value = "id") String id) {
        return responseProcessor.replyWithFutureRetrying("BookingFlowV1CalculateRefund",
                () -> ordersService.calculateRefund(id),
                RetryStrategyExceptionHelpers.defaultStatusUnavailableRetryStrategy()
        );
    }

    @RequestMapping(value = "/v1/start_refund", method = RequestMethod.POST, produces = "application/json", consumes =
            "application/json")
    public DeferredResult<Void> startRefund(@RequestParam(value = "id") String id,
                                            @RequestBody @Valid RefundRequest request) {
        return responseProcessor.replyWithFutureRetrying("BookingFlowV1StartRefund",
                () -> ordersService.startRefund(id, request.getRefundToken()),
                RetryStrategyExceptionHelpers.defaultStatusUnavailableRetryStrategy()
        );
    }


    @RequestMapping(value = "/v1/cancel_order", method = RequestMethod.POST,
            produces = "application/json", consumes = "application/json")
    @ApiOperation("Отмена по запросу пользователя")
    public DeferredResult<Void> cancelOrder(@RequestBody @Valid CancelOrderReqV1 request) {
        return responseProcessor.replyWithFutureRetrying("BookingFlowV1CancelOrder",
                () -> ordersService.cancelOrder(request),
                RetryStrategyExceptionHelpers.defaultStatusUnavailableRetryStrategy()
        );
    }


    @RequestMapping(value = "/v1/get_order_pdf_link", method = RequestMethod.GET, produces = "application/json")
    public DeferredResult<PdfLink> getOrderPdfLink(@RequestParam(value = "id") String id) {
        return responseProcessor.replyWithFutureRetrying(
                "BookingFlowV1GetOrderPdfLink",
                () -> ordersService.getOrderById(id).thenApply(order -> new PdfLink(
                        order.getDocumentUrl(), order.getBusinessTripDocUrl()
                )),
                RetryStrategyExceptionHelpers.defaultStatusUnavailableRetryStrategy()
        );
    }

    @RequestMapping(value = "/v1/generate_business_trip_pdf", method = RequestMethod.GET, produces = "application/json")
    public DeferredResult<Void> generateBusinessTripPdf(@RequestParam(value = "id") String id) {
        return responseProcessor.replyWithFutureRetrying(
                "BookingFlowV1GenerateBusinessTripPdf",
                () -> ordersService.generateBusinessTripPdf(id),
                RetryStrategyExceptionHelpers.defaultStatusUnavailableRetryStrategy()
        );
    }
}
