package ru.yandex.travel.api.endpoints.booking_flow;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.temporal.ChronoUnit;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;
import org.springframework.stereotype.Service;

import ru.yandex.travel.api.endpoints.booking_flow.model.BedGroupDto;
import ru.yandex.travel.api.endpoints.booking_flow.model.CancellationInfo;
import ru.yandex.travel.api.endpoints.booking_flow.model.ConfirmationInfo;
import ru.yandex.travel.api.endpoints.booking_flow.model.CreateV1HotelOrderRequest;
import ru.yandex.travel.api.endpoints.booking_flow.model.CurrentPaymentDto;
import ru.yandex.travel.api.endpoints.booking_flow.model.ExtraFeeRate;
import ru.yandex.travel.api.endpoints.booking_flow.model.GeoHotelInfo;
import ru.yandex.travel.api.endpoints.booking_flow.model.HotelCharges;
import ru.yandex.travel.api.endpoints.booking_flow.model.HotelOrderDto;
import ru.yandex.travel.api.endpoints.booking_flow.model.ImageDto;
import ru.yandex.travel.api.endpoints.booking_flow.model.LegalInfo;
import ru.yandex.travel.api.endpoints.booking_flow.model.LegalInfoItem;
import ru.yandex.travel.api.endpoints.booking_flow.model.LocalizedPansionInfo;
import ru.yandex.travel.api.endpoints.booking_flow.model.NextPaymentDto;
import ru.yandex.travel.api.endpoints.booking_flow.model.NightlyRate;
import ru.yandex.travel.api.endpoints.booking_flow.model.OfferDto;
import ru.yandex.travel.api.endpoints.booking_flow.model.OrderGuestInfoDto;
import ru.yandex.travel.api.endpoints.booking_flow.model.OrderStatus;
import ru.yandex.travel.api.endpoints.booking_flow.model.PartnerHotelInfoDto;
import ru.yandex.travel.api.endpoints.booking_flow.model.PaymentDto;
import ru.yandex.travel.api.endpoints.booking_flow.model.PaymentErrorDto;
import ru.yandex.travel.api.endpoints.booking_flow.model.PaymentScheduleDto;
import ru.yandex.travel.api.endpoints.booking_flow.model.PriceInfo;
import ru.yandex.travel.api.endpoints.booking_flow.model.RateDto;
import ru.yandex.travel.api.endpoints.booking_flow.model.ReceiptItem;
import ru.yandex.travel.api.endpoints.booking_flow.model.RequestInfo;
import ru.yandex.travel.api.endpoints.booking_flow.model.RoomContentDto;
import ru.yandex.travel.api.endpoints.booking_flow.model.ScheduledPaymentDto;
import ru.yandex.travel.api.endpoints.booking_flow.model.TotalsBreakdown;
import ru.yandex.travel.api.endpoints.booking_flow.model.promo.AppliedPromoCampaignsDto;
import ru.yandex.travel.api.endpoints.booking_flow.model.promo.PromoCampaignsDto;
import ru.yandex.travel.api.endpoints.booking_flow.req_rsp.EstimateDiscountReqV1;
import ru.yandex.travel.api.endpoints.booking_flow.req_rsp.EstimateDiscountRspV1;
import ru.yandex.travel.api.endpoints.hotels_portal.breadcrumbs.BreadcrumbsService;
import ru.yandex.travel.api.models.hotels.Coordinates;
import ru.yandex.travel.api.models.hotels.HotelOffer;
import ru.yandex.travel.api.services.hotels.slug.HotelSlugService;
import ru.yandex.travel.api.services.hotels_booking_flow.models.EstimateDiscountResult;
import ru.yandex.travel.api.services.hotels_booking_flow.models.HotelOrder;
import ru.yandex.travel.api.services.hotels_booking_flow.models.OrderCreationData;
import ru.yandex.travel.api.services.hotels_booking_flow.models.PaymentInfo;
import ru.yandex.travel.hotels.common.CancellationPenalty;
import ru.yandex.travel.hotels.common.Permalink;
import ru.yandex.travel.hotels.common.orders.BaseRate;
import ru.yandex.travel.hotels.common.orders.Guest;
import ru.yandex.travel.hotels.common.orders.RefundInfo;
import ru.yandex.travel.hotels.common.refunds.RefundRule;
import ru.yandex.travel.hotels.common.refunds.RefundRules;
import ru.yandex.travel.hotels.common.refunds.RefundType;
import ru.yandex.travel.hotels.common.schedule.OfferPaymentSchedule;
import ru.yandex.travel.hotels.common.schedule.OfferPaymentScheduleItem;
import ru.yandex.travel.hotels.models.booking_flow.BreakdownType;
import ru.yandex.travel.hotels.models.booking_flow.Image;
import ru.yandex.travel.hotels.models.booking_flow.Offer;
import ru.yandex.travel.hotels.models.booking_flow.PartnerHotelInfo;
import ru.yandex.travel.hotels.models.booking_flow.RateStatus;
import ru.yandex.travel.hotels.models.booking_flow.promo.PromoCampaignsInfo;

import static ru.yandex.travel.api.services.common.PromoCampaignUtils.createPromoCampaignsDto;

@Service
@Slf4j
@RequiredArgsConstructor
public class DtoMapper {
    static final String S = "70px";
    static final String M = "350px";
    static final String L = "1000px";

    private final HotelSlugService hotelSlugService;
    private final BreadcrumbsService breadcrumbsService;

    public static RateDto moneyToRateDto(Money money) {
        RateDto result = new RateDto();
        result.setAmount(money.getNumberStripped().setScale(2, RoundingMode.HALF_UP).toString());
        result.setCurrency(money.getCurrency().getCurrencyCode());
        return result;
    }

    public HotelOrderDto buildOrderDto(HotelOrder order, boolean fillDocumentUrl) {
        HotelOrderDto orderDto = new HotelOrderDto();
        orderDto.setId(order.getId());
        orderDto.setYandexOrderId(order.getPrettyId());
        OfferDto offerDto = buildOfferDto(order.getOfferInfo());
        orderDto.setOrderInfo(offerDto);
        if (offerDto.getCancellationInfo() != null) {
            List<CancellationPenalty> allPenalties =
                    new ArrayList<>(offerDto.getCancellationInfo().getPenalties());
            if (!allPenalties.isEmpty()) {
                //set startsAt of the first penalty as createdDate of the order
                CancellationPenalty firstPenalty = allPenalties.get(0);
                allPenalties.set(0, new CancellationPenalty(
                        order.getCreatedAt().truncatedTo(ChronoUnit.MINUTES),
                        firstPenalty.getEndsAt(),
                        firstPenalty.getType(),
                        firstPenalty.getAmount(),
                        firstPenalty.getCurrency()
                ));

                //set endsAt for last penalty as end ot the day of max(checkInDate, lastPenalty.startsAt)
                CancellationPenalty lastPenalty = allPenalties.get(allPenalties.size() - 1);
                LocalDateTime endsAt;
                if (lastPenalty.getStartsAt().toLocalDate().compareTo(order.getOfferInfo().getMetaInfo().getSearch().getCheckIn()) >= 0) {
                    endsAt = lastPenalty.getStartsAt().toLocalDate().atTime(23, 59, 59);
                } else {
                    endsAt = order.getOfferInfo().getMetaInfo().getSearch().getCheckIn().atTime(23, 59, 59);
                }
                allPenalties.set(allPenalties.size() - 1, new CancellationPenalty(
                        lastPenalty.getStartsAt(),
                        endsAt,
                        lastPenalty.getType(),
                        lastPenalty.getAmount(),
                        lastPenalty.getCurrency()
                ));
            }
            CancellationInfo unfiltered = new CancellationInfo();
            unfiltered.setPenalties(allPenalties);
            unfiltered.setRefundable(allPenalties.size() > 0);
            offerDto.setCancellationInfoUnfiltered(unfiltered);
            CancellationInfo filtered = offerDto.getCancellationInfo();
            if (order.getStatus() == OrderStatus.CONFIRMED) {
                filtered = filtered.removeObsolete();
            }
            offerDto.setCancellationInfo(filtered);
        }
        orderDto.setGuestInfo(new OrderGuestInfoDto());
        orderDto.getGuestInfo().setCustomerEmail(order.getGuestInfo().getCustomerEmail());
        orderDto.getGuestInfo().setCustomerPhone(order.getGuestInfo().getCustomerPhone());
        orderDto.getGuestInfo().setAllowsSubscription(order.getGuestInfo().isAllowsSubscription());
        orderDto.getGuestInfo().setGuests(order.getGuestInfo().getGuests().stream()
                .map(g -> {
                    Guest guest = new Guest();

                    guest.setFirstName(g.getFirstName());
                    guest.setLastName(g.getLastName());
                    guest.setAge(g.getAge());
                    guest.setChild(g.isChild());

                    return guest;
                })
                .collect(Collectors.toList()));
        orderDto.getGuestInfo().setNumAdults((int) order.getGuestInfo().getGuests().stream().filter(guest -> !guest.isChild()).count());
        orderDto.getGuestInfo().setChildAges(order.getGuestInfo().getGuests().stream()
                .filter(Guest::isChild)
                .map(Guest::getAge)
                .collect(Collectors.toList()));
        orderDto.setStatus(order.getStatus());
        orderDto.setDisplayState(order.getDisplayState());
        orderDto.setOrderCancellationDetails(order.getOrderCancellationDetails());
        orderDto.setPayment(buildPaymentDto(order.getPayment()));
        orderDto.setCanGenerateBusinessTripDoc(order.getCanGenerateBusinessTripDoc());
        if (order.getConfirmationId() != null) {
            var confirmationInfo =
                    ConfirmationInfo.builder().confirmationId(order.getConfirmationId());
            if (fillDocumentUrl) {
                confirmationInfo.documentUrl(order.getDocumentUrl());
            }
            orderDto.setConfirmationInfo(confirmationInfo.build());
        }
        if (order.getRefundInfo() != null) {
            orderDto.setRefundInfo(new RefundInfo());
            if (order.getRefundInfo().getRefund() != null) {
                orderDto.getRefundInfo().setRefund(
                        new BaseRate(
                                order.getRefundInfo().getRefund().getAmount(),
                                order.getRefundInfo().getRefund().getCurrency()));
            }
            if (order.getRefundInfo().getPenalty() != null) {
                orderDto.getRefundInfo().setPenalty(
                        new BaseRate(
                                order.getRefundInfo().getPenalty().getAmount(),
                                order.getRefundInfo().getPenalty().getCurrency()));
            }
            orderDto.getRefundInfo().setPenaltyIntervalIndex(order.getRefundInfo().getPenaltyIntervalIndex());
            orderDto.getRefundInfo().setRefundDateTime(order.getRefundInfo().getRefundDateTime());
            orderDto.getRefundInfo().setReason(order.getRefundInfo().getReason());
        }
        orderDto.setOrderPriceInfo(order.getOrderPriceInfo());
        orderDto.setAppliedPromoCampaigns(AppliedPromoCampaignsDto.convertFromModel(order.getAppliedPromoCampaigns()));
        return orderDto;
    }

    private PaymentDto buildPaymentDto(PaymentInfo payment) {
        if (payment != null) {
            var paymentDtoBuilder = PaymentDto.builder()
                    .mayBeStarted(payment.isMayBeStarted())
                    .mayBeCancelled(payment.isMayBeCancelled())
                    .usesDeferredPayments(payment.isUsesDeferredPayment())
                    .usesZeroFirstPayment(payment.isUsesZeroFirstPayment())
                    .amountPaid(payment.getAmountPaid())
                    .receipts(payment.getReceipts().stream()
                            .map(r -> new ReceiptItem(r.getUrl(), r.getType()))
                            .collect(Collectors.toUnmodifiableList()));
            if (payment.getCurrent() != null) {
                paymentDtoBuilder.current(CurrentPaymentDto.builder()
                        .type(payment.getCurrent().getPaymentType())
                        .amount(payment.getCurrent().getAmount())
                        .paymentUrl(payment.getCurrent().getPaymentUrl())
                        .build());
            }
            if (payment.getError() != null) {
                paymentDtoBuilder.error(PaymentErrorDto.builder()
                        // 'amount' used to be 'total' but due to TRAVELBACK-2837 it was made 'real money only'
                        .amount(payment.getError().getAmountMarkup().getCard())
                        .totalAmount(payment.getError().getAmount())
                        .cardAmount(payment.getError().getAmountMarkup().getCard())
                        .yandexAccountAmount(payment.getError().getAmountMarkup().getYandexAccount())
                        .code(payment.getError().getCode())
                        .build());
            }
            if (payment.getNext() != null && payment.getNext().getPaymentEndsAt() != null && payment.getNext().getPaymentEndsAt().isAfter(Instant.now())) {
                paymentDtoBuilder.next(NextPaymentDto.builder()
                        .amount(payment.getNext().getAmount())
                        .paymentEndsAt(payment.getNext().getPaymentEndsAt())
                        .penaltyIfUnpaid(payment.getNext().getPenaltyIfUnpaid())
                        .build());
            }
            return paymentDtoBuilder.build();
        } else {
            return null;
        }
    }

    public OfferDto buildOfferDto(Offer offer) {
        OfferDto dto = new OfferDto();
        copyMetadataToDto(offer, dto);
        copyBasicHotelInfoToDto(offer, dto);
        copyPartnerHotelInfoToDto(offer, dto);
        copyStayInfoToDto(offer, dto);
        copyRateInfoToDto(offer, dto);
        copyRoomInfoToDto(offer, dto);
        copyRefundRulesToDto(offer, dto);
        copyLegalInfoToDto(offer, dto);
        copyPromoCampaignsInfoToDto(offer, dto);
        dto.setPartnerId(offer.getMetaInfo().getSearch().getPartnerId());
        dto.setDirectPartner(offer.getMetaInfo().getDirectPartner());
        dto.setDeferredPaymentSchedule(mapOfferPaymentSchedule(offer.getDeferredPaymentSchedule()));
        dto.setAllGuestsRequired(offer.isAllGuestsRequired());
        dto.setAllowPostPay(offer.isAllowPostPay());
        return dto;
    }

    public OrderCreationData buildOrderCreationData(CreateV1HotelOrderRequest old) {
        return OrderCreationData.builder()
                .customerEmail(old.getCustomerEmail())
                .customerPhone(old.getCustomerPhone())
                .allowsSubscription(old.isAllowsSubscription())
                .guests(old.getGuests())
                .token(old.getToken())
                .label(old.getLabel())
                .checksum(old.getChecksum())
                .selectedBedGroupIndex(old.getSelectedBedGroupIndex())
                .useDeferredPayments(old.isUseDeferredPayments())
                .usePostPay(old.isUsePostPay())
                .build();
    }

    public OrderCreationData buildOrderCreationData(EstimateDiscountReqV1 req) {
        return OrderCreationData.builder()
                .guests(List.of())
                .token(req.getToken())
                .label(req.getLabel())
                .checksum(req.getChecksum())
                .selectedBedGroupIndex(req.getSelectedBedGroupIndex())
                .build();
    }


    private void copyLegalInfoToDto(Offer offer, OfferDto dto) {
        LegalInfo oldLegalInfo = new LegalInfo();
        dto.setLegalInfo(oldLegalInfo);
        oldLegalInfo.setHotel(buildLegalItemDto(offer.getLegalInfo().getHotel()));
        oldLegalInfo.setPartner(buildLegalItemDto(offer.getLegalInfo().getPartner()));
        oldLegalInfo.setYandex(buildLegalItemDto(offer.getLegalInfo().getYandex()));
    }

    private LegalInfoItem buildLegalItemDto(ru.yandex.travel.hotels.models.booking_flow.LegalInfo.LegalInfoItem newItem) {
        LegalInfoItem oldItem = new LegalInfoItem();
        if (newItem != null) {
            oldItem.setName(newItem.getName());
            oldItem.setOgrn(newItem.getOgrn());
            oldItem.setActualAddress(newItem.getActualAddress());
            oldItem.setLegalAddress(newItem.getLegalAddress());
            oldItem.setWorkingHours(newItem.getWorkingHours());
            oldItem.setRegistryNumber(newItem.getRegistryNumber());
        }
        return oldItem;
    }

    private void copyRefundRulesToDto(Offer offer, OfferDto dto) {
        if (offer.getRefundRules() != null) {
            dto.setCancellationInfo(mapCancellationInfo(offer.getRefundRules()));
            dto.setRefundInfo(mapRefundRules(offer.getRefundRules()));
        }
    }

    private void copyRoomInfoToDto(Offer offer, OfferDto dto) {
        if (offer.getRoomInfo() != null) {
            RoomContentDto roomContent = new RoomContentDto();
            dto.setPartnerRoomInfo(roomContent);
            roomContent.setName(offer.getRoomInfo().getName());
            roomContent.setDescriptions(new RoomContentDto.Descriptions());
            roomContent.getDescriptions().setOverview(offer.getRoomInfo().getDescription());
            if (offer.getRoomInfo().getRoomAmenities() != null) {
                try {
                    roomContent.setAmenities(offer.getRoomInfo().getRoomAmenities().stream().collect(Collectors.toMap(
                            a -> a.getId(), a -> {
                                var amenityDto = new RoomContentDto.Amenity();
                                amenityDto.setId(BigDecimal.valueOf(Double.parseDouble(a.getId())));
                                amenityDto.setName(a.getName());
                                return amenityDto;
                            }
                    )));
                } catch (Exception ex) {
                    log.warn("Unable to map room amenities", ex);
                    roomContent.setAmenities(new HashMap<>());
                }
            }
            roomContent.setImages(new ArrayList<>());
            if (offer.getRoomInfo().getImages() != null) {
                for (Image newImage : offer.getRoomInfo().getImages()) {
                    var imageDto = mapImage(newImage);
                    if (imageDto != null) {
                        roomContent.getImages().add(imageDto);
                    }
                }
            }
            dto.setPansionInfo(new LocalizedPansionInfo());
            dto.getPansionInfo().setId(offer.getRoomInfo().getPansionInfo().getId());
            dto.getPansionInfo().setName(offer.getRoomInfo().getPansionInfo().getName());
            if (offer.getRoomInfo().getBedGroups() != null) {
                dto.setBedGroups(offer.getRoomInfo().getBedGroups().stream().map(
                        newBedGroup -> {
                            BedGroupDto oldBedGroup = new BedGroupDto();
                            oldBedGroup.setId(String.valueOf(newBedGroup.getId()));
                            oldBedGroup.setDescription(newBedGroup.getDescription());
                            return oldBedGroup;
                        })
                        .filter(bg -> {
                            if (offer.getMetaInfo().getSelectedBedGroupIndex() == null) {
                                return true;
                            } else {
                                return String.valueOf(offer.getMetaInfo().getSelectedBedGroupIndex()).equals(bg.getId());
                            }
                        })
                        .collect(Collectors.toList()));
            } else {
                dto.setBedGroups(Collections.emptyList());
            }
        }
    }

    private ScheduledPaymentDto mapOfferScheduledPayment(OfferPaymentScheduleItem payment) {
        if (payment == null) {
            return null;
        }
        var dto = new ScheduledPaymentDto();
        dto.setPaymentEndsAt(payment.getPaymentEndsAt());
        if (payment.getAmount() != null) {
            dto.setAmount(payment.getAmount());
        }
        if (payment.getRatio() != null && payment.isRatioIsStandard()) {
            dto.setPercentage(payment.getRatio().multiply(BigDecimal.valueOf(100)).setScale(0, RoundingMode.HALF_UP).intValue());
        }
        if (payment.getPenaltyIfUnpaid() != null) {
            dto.setPenaltyIfUnpaid(payment.getPenaltyIfUnpaid());
        }
        return dto;
    }

    private PaymentScheduleDto mapOfferPaymentSchedule(OfferPaymentSchedule schedule) {
        if (schedule == null) {
            return null;
        }
        PaymentScheduleDto dto = new PaymentScheduleDto();
        dto.setInitialPayment(mapOfferScheduledPayment(schedule.getInitialPayment()));
        dto.setDeferredPayments(new ArrayList<>(schedule.getDeferredPayments().size()));
        dto.setZeroFirstPayment(schedule.getInitialPayment().getAmount().isZero());
        for (var p : schedule.getDeferredPayments()) {
            dto.getDeferredPayments().add(mapOfferScheduledPayment(p));
        }
        return dto;
    }

    private ImageDto mapImage(Image newImage) {
        Map<String, ImageDto.Link> links = new HashMap<>();
        if (newImage.getL() != null) {
            var link = new ImageDto.Link();
            link.setMethod("GET");
            link.setHref(newImage.getL());
            links.put(L, link);
        }
        if (newImage.getM() != null) {
            var link = new ImageDto.Link();
            link.setMethod("GET");
            link.setHref(newImage.getM());
            links.put(M, link);
        }
        if (newImage.getS() != null) {
            var link = new ImageDto.Link();
            link.setMethod("GET");
            link.setHref(newImage.getS());
            links.put(S, link);
        }
        ImageDto imageDto = new ImageDto();
        if (links.size() > 0) {
            imageDto.setLinks(links);
            return imageDto;
        } else {
            return null;
        }
    }

    private void copyRateInfoToDto(Offer offer, OfferDto dto) {
        var ri = offer.getRateInfo();
        PriceInfo priceInfo = new PriceInfo();
        if (ri != null && ri.getStatus() != RateStatus.SOLD_OUT) {
            dto.setRateInfo(priceInfo);
            dto.setPriceMatch(ri.getStatus() == RateStatus.CONFIRMED);
            priceInfo.setHotelCharges(new HotelCharges());
            if (ri.getBaseRateBreakdown() != null) {
                List<NightlyRate> breakdown = ri.getBaseRateBreakdown().stream().map(r -> {
                    NightlyRate nr = new NightlyRate();
                    nr.setBase(new RateDto(r.getAmount(), r.getCurrency()));
                    return nr;
                }).collect(Collectors.toList());
                if (ri.getBreakdownType() == null || ri.getBreakdownType() == BreakdownType.NIGHT) {
                    priceInfo.getHotelCharges().setNightly(breakdown);
                    priceInfo.getHotelCharges().setDaily(Collections.emptyList());

                } else if (ri.getBreakdownType() == BreakdownType.DAY) {
                    priceInfo.getHotelCharges().setDaily(breakdown);
                    priceInfo.getHotelCharges().setNightly(Collections.emptyList());
                } else {
                    priceInfo.getHotelCharges().setNightly(Collections.emptyList());
                    priceInfo.getHotelCharges().setDaily(Collections.emptyList());
                }
            } else {
                priceInfo.getHotelCharges().setNightly(Collections.emptyList());
                priceInfo.getHotelCharges().setDaily(Collections.emptyList());
            }
            priceInfo.getHotelCharges().setTotals(new TotalsBreakdown());
            priceInfo.getHotelCharges().getTotals().setBase(new RateDto(ri.getBaseRate().getAmount(),
                    ri.getBaseRate().getCurrency()));
            if (ri.getTaxesAndFees() != null) {
                priceInfo.getHotelCharges().getTotals().setTaxesAndFeesSum(new RateDto(ri.getTaxesAndFees().getAmount(), ri.getTaxesAndFees().getCurrency()));
            }
            priceInfo.getHotelCharges().getTotals().setGrand(new RateDto(ri.getTotalRate().getAmount(),
                    ri.getTotalRate().getCurrency()));
            if (offer.getDiscountInfo() != null && offer.getDiscountInfo().isApplied()) {
                priceInfo.getHotelCharges().getTotals().setStrikeThrough(priceInfo.getHotelCharges().getTotals().getGrand());
                priceInfo.getHotelCharges().getTotals().setDiscount(RateDto.fromMoney(offer.getDiscountInfo().getDiscountAmount()));
                priceInfo.getHotelCharges().getTotals().setGrand(RateDto.fromMoney(offer.getDiscountInfo().getDiscountedPrice()));
            }
            Money priceAfterPlusWithdraw = ri.getPriceAfterPlusWithdraw();
            if (priceAfterPlusWithdraw != null) {
                priceInfo.getHotelCharges().getTotals().setPriceAfterPlusWithdraw(priceAfterPlusWithdraw);
            }
            if (ri.getExtraFees() != null) {
                priceInfo.setExtraCharges(ri.getExtraFees().stream().map(extraFee -> {
                    ExtraFeeRate efr = new ExtraFeeRate();
                    switch (extraFee.getType()) {
                        case RESORT_FEE:
                            efr.setType(ExtraFeeRate.Type.RESORT_FEE);
                            break;
                        case OTHER_FEE:
                            efr.setType(ExtraFeeRate.Type.MANDATORY_FEE);
                            break;
                    }
                    efr.setPayable(new RateDto(extraFee.getAmount(), extraFee.getCurrency()));
                    return efr;
                }).collect(Collectors.toList()));
            } else {
                priceInfo.setExtraCharges(Collections.emptyList());
            }
        } else {
            dto.setPriceMatch(false);
        }
    }

    private void copyStayInfoToDto(Offer offer, OfferDto dto) {
        var propertyContent = dto.getPartnerHotelInfo();
        if (propertyContent != null) {
            propertyContent.setCheckin(new PartnerHotelInfoDto.Checkin());
            propertyContent.setCheckout(new PartnerHotelInfoDto.Checkout());
            propertyContent.setFees(new PartnerHotelInfoDto.Fees());
            propertyContent.setPolicies(new PartnerHotelInfoDto.Policies());
            if (offer.getStayInfo() != null) {
                propertyContent.getCheckin().setBeginTime(offer.getStayInfo().getCheckInStartTime());
                propertyContent.getCheckin().setEndTime(offer.getStayInfo().getCheckInEndTime());
                propertyContent.getCheckout().setTime(offer.getStayInfo().getCheckOutEndTime());
                List<String> instructions = offer.getStayInfo().getStayInstructions();
                if (instructions.size() > 0) {
                    propertyContent.getFees().setMandatory(instructions.get(0));
                }
                if (instructions.size() > 1) {
                    propertyContent.getCheckin().setSpecialInstructions(instructions.get(1));
                }
                if (instructions.size() > 2) {
                    propertyContent.getCheckin().setInstructions(instructions.get(2));
                }
                if (instructions.size() > 3) {
                    propertyContent.getFees().setOptional(instructions.get(3));
                }
                if (instructions.size() > 4) {
                    propertyContent.getPolicies().setKnowBeforeYouGo(instructions.get(4));
                }
            }
        }
    }

    private void copyPartnerHotelInfoToDto(Offer offer, OfferDto dto) {
        if (offer.getHotelInfo() != null && offer.getPartnerHotelInfo() != null) {
            PartnerHotelInfoDto partnerInfoDto = PartnerHotelInfoDto.builder().build();
            dto.setPartnerHotelInfo(partnerInfoDto);
            PartnerHotelInfo partnerHotelInfo = offer.getPartnerHotelInfo();
            partnerInfoDto.setPropertyId(offer.getMetaInfo().getSearch().getOriginalId());
            partnerInfoDto.setName(partnerHotelInfo.getName());
            partnerInfoDto.setPhone(partnerHotelInfo.getPhone());
            String[] addrParts = partnerHotelInfo.getAddress().split(", ", 3);
            PartnerHotelInfoDto.Address address = new PartnerHotelInfoDto.Address();
            if (addrParts.length > 0) {
                address.setCountryCode(addrParts[0]);
            }
            if (addrParts.length > 1) {
                address.setCity(addrParts[1]);
            }
            if (addrParts.length > 2) {
                address.setLine1(addrParts[2]);
            }
            partnerInfoDto.setAddress(address);
            if (partnerHotelInfo.getCoordinates() != null) {
                partnerInfoDto.setLocation(new PartnerHotelInfoDto.Location());
                partnerInfoDto.getLocation().setCoordinates(new PartnerHotelInfoDto.Coordinates());
                partnerInfoDto.getLocation().getCoordinates().setLatitude(BigDecimal.valueOf(partnerHotelInfo.getCoordinates().getLatitude()));
                partnerInfoDto.getLocation().getCoordinates().setLongitude(BigDecimal.valueOf(partnerHotelInfo.getCoordinates().getLongitude()));
            }
            if (partnerHotelInfo.getStars() != null) {
                PartnerHotelInfoDto.Ratings ratings = new PartnerHotelInfoDto.Ratings();
                ratings.setProperty(new PartnerHotelInfoDto.Rating());
                ratings.getProperty().setType("Star");
                ratings.getProperty().setRating(String.valueOf((double) partnerHotelInfo.getStars()));
                partnerInfoDto.setRatings(ratings);
            }
            if (partnerHotelInfo.getAmenities() != null) {
                try {
                    partnerInfoDto.setAmenities(
                            partnerHotelInfo.getAmenities().stream().collect(Collectors.toMap(a -> a.getId(),
                                    a -> {
                                        var oldAmenity = new PartnerHotelInfoDto.Amenity();
                                        oldAmenity.setId(BigDecimal.valueOf(Long.parseLong(a.getId())));
                                        oldAmenity.setName(a.getName());
                                        return oldAmenity;
                                    })));
                } catch (Exception ex) {
                    log.warn("Unable to convert amenity list", ex);
                    partnerInfoDto.setAmenities(new HashMap<>());
                }
            }
            partnerInfoDto.setDescriptions(new PartnerHotelInfoDto.Descriptions());
            if (partnerHotelInfo.getDescriptions() != null) {
                partnerInfoDto.getDescriptions().setLocation(partnerHotelInfo.getDescriptions().stream()
                        .filter(tb -> Set.of("Расположение", "Транспортная доступность").contains(tb.getHeader())).findFirst()
                        .map(PartnerHotelInfo.TextBlock::getText).orElse(null));
                partnerInfoDto.getDescriptions().setRooms(partnerHotelInfo.getDescriptions().stream()
                        .filter(tb -> Set.of("Размещение", "Описание").contains(tb.getHeader())).findFirst()
                        .map(PartnerHotelInfo.TextBlock::getText).orElse(null));
                partnerInfoDto.getDescriptions().setDining(partnerHotelInfo.getDescriptions().stream()
                        .filter(tb -> "Питание".equals(tb.getHeader())).findFirst()
                        .map(PartnerHotelInfo.TextBlock::getText).orElse(null));
                partnerInfoDto.getDescriptions().setAmenities(partnerHotelInfo.getDescriptions().stream()
                        .filter(tb -> "Услуги отеля".equals(tb.getHeader())).findFirst()
                        .map(PartnerHotelInfo.TextBlock::getText).orElse(null));
                partnerInfoDto.getDescriptions().setBusinessAmenities(partnerHotelInfo.getDescriptions().stream()
                        .filter(tb -> "Услуги для бизнеса".equals(tb.getHeader())).findFirst()
                        .map(PartnerHotelInfo.TextBlock::getText).orElse(null));
            }
            if (partnerHotelInfo.getImages() != null) {
                partnerInfoDto.setImages(partnerHotelInfo.getImages().stream().map(this::mapImage).filter(Objects::nonNull).collect(Collectors.toList()));
            }
        }
    }

    private void copyBasicHotelInfoToDto(Offer offer, OfferDto dto) {
        if (offer.getHotelInfo() != null) {
            GeoHotelInfo basicGeo = new GeoHotelInfo();
            basicGeo.setPermalink(offer.getHotelInfo().getPermalink());
            basicGeo.setHotelSlug(hotelSlugService.findMainSlugByPermalink(Permalink.of(basicGeo.getPermalink())));
            basicGeo.setName(offer.getHotelInfo().getName());
            basicGeo.setAddress(offer.getHotelInfo().getAddress());
            basicGeo.setStars(offer.getHotelInfo().getStars());
            basicGeo.setRating(offer.getHotelInfo().getRating());
            basicGeo.setPhone(offer.getHotelInfo().getPhone());
            if (offer.getHotelInfo().getCoordinates() != null) {
                basicGeo.setCoordinates(new GeoHotelInfo.Coordinates(
                        offer.getHotelInfo().getCoordinates().getLongitude(),
                        offer.getHotelInfo().getCoordinates().getLatitude()));
            }
            basicGeo.setImageUrlTemplate(offer.getHotelInfo().getImageUrlTemplate());
            basicGeo.setWorkingHours(offer.getHotelInfo().getWorkingHours());
            basicGeo.setLegalInfo(new GeoHotelInfo.LegalInfo(
                    "", // inn is not set in new model
                    offer.getLegalInfo().getHotel().getOgrn(),
                    offer.getLegalInfo().getHotel().getLegalAddress()));
            basicGeo.setLocationType(offer.getHotelInfo().getLocationType());
            if (offer.getHotelInfo().getCoordinates() != null) {
                basicGeo.setBreadcrumbs(breadcrumbsService.getHotelBreadcrumbs(
                        "ru",
                        Coordinates.of(
                                offer.getHotelInfo().getCoordinates().getLatitude(),
                                offer.getHotelInfo().getCoordinates().getLongitude()
                        )
                ));
            }
            dto.setBasicHotelInfo(basicGeo);
        }
    }

    private void copyPromoCampaignsInfoToDto(Offer offer, OfferDto dto) {
        if (offer.getPromoCampaignsInfo() == null) {
            return;
        }
        PromoCampaignsInfo campaigns = offer.getPromoCampaignsInfo();
        PromoCampaignsDto campaignsDto = createPromoCampaignsDto(campaigns);
        dto.setPromoCampaigns(campaignsDto);
    }


    public CancellationInfo mapCancellationInfo(RefundRules rules) {
        CancellationInfo res = new CancellationInfo();
        if (!rules.isRefundable()) {
            res.setRefundable(false);
            res.setPenalties(Collections.emptyList());
        } else {
            res.setRefundable(true);
            res.setPenalties(rules.getRules().stream().map(RefundRule::toCancellationPenalty).collect(Collectors.toUnmodifiableList()));
        }
        return res;
    }

    public HotelOffer.CancellationInfo mapRefundRules(RefundRules rules) {
        HotelOffer.CancellationInfo res = new HotelOffer.CancellationInfo();
        res.setHasFreeCancellation(rules.isFullyRefundable());
        if (rules.getRules().isEmpty()) {
            res.setRefundType(RefundType.NON_REFUNDABLE);
        } else {
            res.setRefundType(rules.getRules().get(0).getType());
        }
        res.setRefundRules(rules.getRules());
        return res;
    }

    private void copyMetadataToDto(Offer offer, OfferDto dto) {
        if (offer.getMetaInfo() != null) {
            dto.setRequestInfo(RequestInfo.builder()
                    .checkinDate(offer.getMetaInfo().getSearch().getCheckIn())
                    .checkoutDate(offer.getMetaInfo().getSearch().getCheckOut())
                    .numAdults(offer.getMetaInfo().getSearch().getAdults())
                    .childAges(offer.getMetaInfo().getSearch().getChildren())
                    .selectedBedGroupIndex(offer.getMetaInfo().getSelectedBedGroupIndex())
                    .build());
            dto.setTravelToken(offer.getMetaInfo().getToken());
            dto.setLabel(offer.getMetaInfo().getLabel());
            dto.setChecksum(offer.getMetaInfo().getCheckSum());
            dto.setSessionKey(offer.getMetaInfo().getDeduplicationKey());
        }
    }

    public EstimateDiscountRspV1 buildEstimateDiscountDto(EstimateDiscountResult discountResult) {
        return EstimateDiscountRspV1.builder()
                .codeApplicationResults(discountResult.getCodeApplicationResults())
                .discountAmount(discountResult.getDiscountAmount())
                .discountedAmount(discountResult.getDiscountedAmount())
                .discountPlusPoints(discountResult.getDiscountPlusPoints())
                .originalAmount(discountResult.getOriginalAmount())
                .promoCampaigns(discountResult.getPromoCampaigns())
                .deferredPaymentSchedule(mapOfferPaymentSchedule(discountResult.getDeferredPaymentSchedule()))
                .build();
    }
}
