package ru.yandex.travel.api.endpoints.booking_flow.model;

import java.time.LocalDateTime;
import java.time.ZoneOffset;
import java.util.Collections;
import java.util.List;
import java.util.stream.Collectors;

import com.fasterxml.jackson.annotation.JsonIgnoreProperties;
import lombok.Data;

import ru.yandex.travel.hotels.common.CancellationPenalty;

@Data
@JsonIgnoreProperties(ignoreUnknown = true)
public class CancellationInfo {
    public final static int HIGHLIGHT_IN_DAYS = 3;
    private boolean refundable;
    private List<CancellationPenalty> penalties;

    public CancellationInfo removeObsolete() {
        CancellationInfo copy = new CancellationInfo();
        if (!isRefundable() || penalties == null) {
            copy.setRefundable(false);
            copy.setPenalties(Collections.emptyList());
            return copy;
        }
        List<CancellationPenalty> newPenalties =
                penalties.stream().filter(cp -> cp.getEndsAt() == null || cp.getEndsAt().isAfter(LocalDateTime.now(ZoneOffset.UTC))).collect(Collectors.toList());
        if (newPenalties.stream().allMatch(np -> np.getType() == CancellationPenalty.Type.FULL_PRICE)) {
            copy.setRefundable(false);
            copy.setPenalties(Collections.emptyList());
        } else {
            copy.setRefundable(true);
            copy.setPenalties(newPenalties);
        }
        return copy;
    }

    public boolean isHighlighted() {
        CancellationInfo actualized = this.removeObsolete();
        return !actualized.isRefundable() || actualized.getPenalties() == null || actualized.penalties.size() <= 1 ||
                (actualized.getPenalties().get(1).getStartsAt() != null &&
                        actualized.getPenalties().get(1).getStartsAt().isBefore(
                                LocalDateTime.now(ZoneOffset.UTC).plusDays(HIGHLIGHT_IN_DAYS)));
    }

    public CancellationPenalty getPenaltyForDateTime(LocalDateTime dateTime) {
        return penalties.stream()
                .filter(penalty -> penalty.matchesDateTime(dateTime))
                .findFirst()
                .orElse(CancellationPenalty.fullPenalty(dateTime));
    }

    public boolean hasFreeCancellation() {
        CancellationInfo actualized = this.removeObsolete();
        if (actualized.isRefundable() && actualized.getPenalties() != null && actualized.getPenalties().size() > 0) {
            return (actualized.getPenalties().get(0).getType() == CancellationPenalty.Type.NO_PENALTY);
        }
        return false;
    }

}
