package ru.yandex.travel.api.endpoints.buses;

import javax.validation.Valid;

import io.grpc.StatusRuntimeException;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.jetbrains.annotations.NotNull;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.context.request.async.DeferredResult;

import ru.yandex.travel.api.endpoints.buses.req_rsp.BusDirectionReqV1;
import ru.yandex.travel.api.endpoints.buses.req_rsp.BusDirectionRspV1;
import ru.yandex.travel.api.endpoints.buses.req_rsp.BusErrorResponse;
import ru.yandex.travel.api.exceptions.GrpcError;
import ru.yandex.travel.api.infrastucture.ResponseProcessor;
import ru.yandex.travel.api.services.buses.BusesApiService;
import ru.yandex.travel.api.spec.buses.CreateRideOfferRequest;
import ru.yandex.travel.api.spec.buses.CreateRideOfferResponse;
import ru.yandex.travel.commons.http.CommonHttpHeaders;
import ru.yandex.travel.credentials.UserCredentials;
import ru.yandex.travel.hotels.common.HotelNotFoundException;

@RestController
@RequestMapping(path = "/api/buses")
@RequiredArgsConstructor
@Slf4j
public class BusesController {
    private final ResponseProcessor responseProcessor;
    private final BusesApiService busesApiService;
    private final BusesSearchImpl busesSearch;

    @ExceptionHandler(HotelNotFoundException.class)
    @ResponseStatus(HttpStatus.NOT_FOUND)
    public ResponseEntity<BusErrorResponse> handleHotelNotFoundException(HotelNotFoundException e) {
        var rsp = new BusErrorResponse(e.getMessage());
        return ResponseEntity.status(HttpStatus.NOT_FOUND).body(rsp);
    }

    @ExceptionHandler(IllegalArgumentException.class)
    @ResponseStatus(HttpStatus.BAD_REQUEST)
    public ResponseEntity<String> handleIllegalArgumentException(@NotNull IllegalArgumentException e) {
        return ResponseEntity.badRequest().contentType(MediaType.TEXT_PLAIN).body(e.getMessage());
    }

    @ExceptionHandler(StatusRuntimeException.class)
    public ResponseEntity<GrpcError> handleGrpcErrors(StatusRuntimeException ex) {
        GrpcError error = GrpcError.fromGrpcStatusRuntimeException(ex);
        return ResponseEntity.status(error.getStatus()).contentType(MediaType.APPLICATION_JSON).body(error);
    }

    @PostMapping(path = "/v1/create_ride_offer", produces = MediaType.APPLICATION_JSON_VALUE)
    public DeferredResult<CreateRideOfferResponse> createRideOffer(@RequestBody CreateRideOfferRequest request) {
        return responseProcessor.replyWithFuture(
                "BusesRideDetails",
                () -> busesApiService.createRideOffer(request, CommonHttpHeaders.get())
        );
    }

    @GetMapping(path = "/v1/direction", produces = MediaType.APPLICATION_JSON_VALUE)
    public DeferredResult<BusDirectionRspV1> direction(@Valid BusDirectionReqV1 request) {
        return responseProcessor.replyWithFuture(
                "busesDirection",
                () -> busesSearch.direction(request, CommonHttpHeaders.get(), UserCredentials.get())
        );
    }
}
