package ru.yandex.travel.api.endpoints.buses;

import java.text.SimpleDateFormat;
import java.time.Instant;
import java.time.LocalDateTime;
import java.time.ZoneId;
import java.time.ZoneOffset;
import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Date;
import java.util.List;
import java.util.Optional;
import java.util.UUID;
import java.util.concurrent.CancellationException;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionException;
import java.util.stream.Collectors;
import java.util.stream.Stream;

import lombok.AllArgsConstructor;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;
import org.jetbrains.annotations.NotNull;
import org.springframework.http.HttpStatus;
import org.springframework.stereotype.Service;
import org.springframework.web.server.ResponseStatusException;

import ru.yandex.travel.api.endpoints.buses.req_rsp.BusDirectionReqV1;
import ru.yandex.travel.api.endpoints.buses.req_rsp.BusDirectionRspV1;
import ru.yandex.travel.api.models.crosslinks.CrosslinksHotelsBlockData;
import ru.yandex.travel.api.models.rasp.morda_backend.ParseContext;
import ru.yandex.travel.api.models.train.CrossLink;
import ru.yandex.travel.api.services.buses.BusDirectionMapper;
import ru.yandex.travel.api.services.buses.BusesFacadeService;
import ru.yandex.travel.api.services.buses.BusesFacadeService.BUS_SEARCH_RESULT_CRITERIA;
import ru.yandex.travel.api.services.crosslinks.CrosslinksService;
import ru.yandex.travel.api.services.hotels.slug.RegionSlugService;
import ru.yandex.travel.api.services.hotels.static_pages.RegionPagesStorage;
import ru.yandex.travel.api.services.train.TrainCrosslinkService;
import ru.yandex.travel.api.services.train.TrainHttpProxyCacheService;
import ru.yandex.travel.api.spec.buses.PointType;
import ru.yandex.travel.bus.model.BusPointsHttp;
import ru.yandex.travel.bus.model.BusRideHttp;
import ru.yandex.travel.bus.service.BusesHttpApiService;
import ru.yandex.travel.commons.http.CommonHttpHeaders;
import ru.yandex.travel.credentials.UserCredentials;
import ru.yandex.travel.hotels.proto.region_pages.TRegionPage;

@Service
@AllArgsConstructor
@Slf4j
public class BusesSearchImpl {
    private final RegionSlugService regionSlugService;
    private final CrosslinksService crosslinksService;
    private final RegionPagesStorage regionPagesStorage;
    private final TrainHttpProxyCacheService cachedMordaBackendService;
    private final TrainCrosslinkService trainCrosslinkService;
    private final BusesHttpApiService busesHttpApiService;
    private final BusesFacadeService busesFacadeService;
    private final BusDirectionMapper busDirectionMapper;

    @AllArgsConstructor
    @Data
    public static class BusRidesHttpInfo {
        private String when;
        private BusRideHttp[] rides;

        public static BusRidesHttpInfo noData() {
            return new BusRidesHttpInfo(null, new BusRideHttp[0]);
        }
    }

    public CompletableFuture<BusDirectionRspV1> direction(@NotNull BusDirectionReqV1 request,
                                                          CommonHttpHeaders commonHttpHeaders,
                                                          UserCredentials userCredentials) {

        var requestId = UUID.randomUUID().toString();

        CompletableFuture<ParseContext> geoContextFuture = cachedMordaBackendService.parseContext(
                request.getFromSlug(),
                request.getToSlug(),
                "ru",
                "bus",
                requestId
        ).thenApply(geoContext -> {
            if (geoContext.getErrors().length > 0) {
                log.error("Exception occurred when getting geoContext for bus page: " + geoContext.getErrors().toString());
                throw new ResponseStatusException(HttpStatus.BAD_REQUEST);
            }
            return geoContext;
        });

        CompletableFuture<CrossLink[]> crossLinkFuture = geoContextFuture.thenCompose(geoContext -> trainCrosslinkService.crosslinks(
                geoContext.getFrom().getKey(),
                geoContext.getTo().getKey(),
                "bus",
                requestId
        ));

        CompletableFuture<BusRidesHttpInfo> ridesFuture = geoContextFuture.thenCompose(geoContext -> getBusRidesByCalendar(geoContext, requestId));
        CompletableFuture<BusPointsHttp> pointsFuture = ridesFuture.thenCompose(busRidesHttpInfo -> getBusPoints(busRidesHttpInfo, requestId));
        CompletableFuture<CrosslinksHotelsBlockData> hotelsFuture = geoContextFuture.thenCompose(geoContext -> getCrossLinksHotels(request, commonHttpHeaders, userCredentials));

        return CompletableFuture.allOf(geoContextFuture, crossLinkFuture, pointsFuture, hotelsFuture).handle((ignored, ex) -> {
            ParseContext geoContext = geoContextFuture.join();
            BusRidesHttpInfo busRidesInfo = ridesFuture.join();

            CrossLink[] crossLinks;
            try {
                crossLinks = crossLinkFuture.join();
            } catch (CancellationException | CompletionException exception) {
                crossLinks = new CrossLink[0];
                log.error("Exception occurred when getting buses cross links for bus page:", exception);
            }

            CrosslinksHotelsBlockData hotelsBlocks = null;
            try {
                hotelsBlocks = hotelsFuture.join();
            } catch (CancellationException | CompletionException exception) {
                log.error("Exception occurred when getting hotels for bus page:", exception);
            }

            BusPointsHttp busPoints;
            try {
                busPoints = pointsFuture.join();
            } catch (CancellationException | CompletionException exception) {
                busPoints = new BusPointsHttp(new BusPointsHttp.Point[0]);
                log.error("Exception occurred when getting points coordinates for bus page:", exception);
            }

            String day = busRidesInfo.getWhen();
            BusRideHttp[] busRides = busRidesInfo.getRides();

            return busDirectionMapper.map(geoContext, day, busRides, busPoints, crossLinks, hotelsBlocks);
        });
    }

    private CompletableFuture<BusRidesHttpInfo> getBusRidesByCalendar(ParseContext geoContext, String requestId) {
        return busesHttpApiService.calendar(
                geoContext.getFrom().getKey(),
                geoContext.getTo().getKey(),
                todayWithDaysDelta(28),
                requestId
        ).thenCompose(calendar -> {
            String[] days = calendar.keySet().toArray(new String[0]);

            List<String> daysWithRide = Arrays
                    .stream(days)
                    .filter(day -> calendar.get(day).getRideCount() > 0)
                    .sorted()
                    .limit(3)
                    .collect(Collectors.toList());

            if (daysWithRide.size() > 0) {
                return flexibleSearchBusRides(geoContext, daysWithRide, 1, requestId);
            }

            // check cache exists
            if (days.length > 2) {
                return CompletableFuture.completedFuture(BusRidesHttpInfo.noData());
            }

            // empty cache
            return flexibleSearchBusRides(geoContext, firstThreeDays(), 1, requestId);
        });
    }

    private CompletableFuture<BusRidesHttpInfo> flexibleSearchBusRides(ParseContext geoContext, List<String> days, int attempt, String requestId) {
        if (attempt > 3 || days.size() < attempt) {
            return CompletableFuture.completedFuture(BusRidesHttpInfo.noData());
        }
        String day = days.get(attempt - 1);
        return busesFacadeService.searchBusRides(
                        geoContext.getFrom().getKey(),
                        geoContext.getTo().getKey(),
                        day,
                        BUS_SEARCH_RESULT_CRITERIA.MORE_AS_POSSIBLE,
                        requestId
        ).thenCompose(rides -> {
            BusRideHttp[] filteredRides = filterExpiredRides(geoContext, rides);
            if (filteredRides == null || filteredRides.length == 0) {
                return flexibleSearchBusRides(geoContext, days,attempt + 1, requestId);
            }
            return CompletableFuture.completedFuture(new BusRidesHttpInfo(day, filteredRides));
        });
    }

    private CompletableFuture<BusPointsHttp> getBusPoints(BusRidesHttpInfo busRidesHttpInfo, String requestId) {
        Stream<BusRideHttp.Point> pointsFromStream = Arrays.stream(busRidesHttpInfo.getRides()).map(ride -> ride.getFrom());
        Stream<BusRideHttp.Point> pointsToStream = Arrays.stream(busRidesHttpInfo.getRides()).map(ride -> ride.getTo());

        String points = Stream.concat(pointsFromStream, pointsToStream)
                .filter(point -> {
                    return  point.getType() != null && point.getId() != null && (
                            point.getType() == PointType.Enum.SETTLEMENT.getNumber() ||
                            point.getType() == PointType.Enum.STATION.getNumber()
                        );
                })
                .map(point -> {
                    // it seems type is always STATION, but not sure
                    String prefix = point.getType() == PointType.Enum.STATION.getNumber() ? "s" : "c";
                    return prefix + String.valueOf(point.getId());
                })
                .distinct()
                .collect(Collectors.joining(","));

        if (points == null || points.equals("")) {
            return CompletableFuture.completedFuture(new BusPointsHttp(new BusPointsHttp.Point[0]));
        }

        return busesHttpApiService.points("code:" + points, requestId);
    }

    private BusRideHttp[] filterExpiredRides(ParseContext geoContext, BusRideHttp[] rides) {
        if (rides == null || rides.length == 0) {
            return rides;
        }
        ZoneId zoneId = ZoneId.of(geoContext.getFrom().getTimezone());
        ZonedDateTime now = ZonedDateTime.now(zoneId);
        return Arrays.stream(rides).filter(ride -> {
            Instant epoch = Instant.ofEpochSecond(ride.getDepartureTime());
            // departureTime is unix time in local timezone (not UTC)
            LocalDateTime localDepartureDateTime = LocalDateTime.ofInstant(epoch, ZoneOffset.UTC);
            ZonedDateTime departureDateTime = ZonedDateTime.of(localDepartureDateTime, zoneId);
            return now.isBefore(departureDateTime);
        }).toArray(BusRideHttp[]::new);
    }

    private List<String> firstThreeDays() {
        return new ArrayList<String>(Arrays.asList(
                todayWithDaysDelta(0),
                todayWithDaysDelta(1),
                todayWithDaysDelta(2)
        ));
    }

    private String todayWithDaysDelta(int delta) {
        Calendar c = Calendar.getInstance();
        c.setTime(new Date());
        c.add(Calendar.DATE, delta);
        Date date = c.getTime();
        SimpleDateFormat formatter = new SimpleDateFormat("yyyy-MM-dd");
        return formatter.format(date);
    }

    private CompletableFuture<CrosslinksHotelsBlockData> getCrossLinksHotels(BusDirectionReqV1 request,
                                                                             CommonHttpHeaders commonHttpHeaders,
                                                                             UserCredentials userCredentials) {
        int geoId;
        try {
            geoId = regionSlugService.getGeoIdBySlug(request.getToSlug());
        } catch (Exception exc) {
            log.error("Unable to get geoId by slug {}. Request to hotel cross links cancelled.", request.getToSlug(), exc);
            return CompletableFuture.completedFuture(null);
        }

        Optional<TRegionPage> result = regionPagesStorage.tryGetRegionPage(geoId);
        if (!result.isPresent()) {
            return CompletableFuture.completedFuture(null);
        }

        return crosslinksService.getHotelsBlockData(geoId, "ru", commonHttpHeaders, userCredentials);
    }
}
