package ru.yandex.travel.api.endpoints.cpa;

import io.grpc.StatusRuntimeException;
import lombok.RequiredArgsConstructor;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.validation.annotation.Validated;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.context.request.async.DeferredResult;

import ru.yandex.travel.api.endpoints.cpa.req_rsp.CpaBoyOrdersReq;
import ru.yandex.travel.api.endpoints.cpa.req_rsp.CpaBoyOrdersRsp;
import ru.yandex.travel.api.endpoints.cpa.req_rsp.CpaOrderSnapshotReqV2;
import ru.yandex.travel.api.endpoints.cpa.req_rsp.CpaOrderSnapshotRspV2;
import ru.yandex.travel.api.exceptions.GrpcError;
import ru.yandex.travel.api.infrastucture.BindFromQuery;
import ru.yandex.travel.api.infrastucture.ResponseProcessor;
import ru.yandex.travel.api.services.common.RetryStrategyExceptionHelpers;

@RestController
@RequestMapping(value = "/api/cpa_export")
@RequiredArgsConstructor
public class CpaExportController {
    @ExceptionHandler(IllegalArgumentException.class)
    @ResponseStatus(HttpStatus.BAD_REQUEST)
    public ResponseEntity<String> handleIllegalArgumentException(IllegalArgumentException e) {
        return ResponseEntity.badRequest().contentType(MediaType.TEXT_PLAIN).body(e.getMessage());
    }

    @ExceptionHandler(StatusRuntimeException.class)
    public ResponseEntity<GrpcError> handleGrpcErrors(StatusRuntimeException ex) {
        GrpcError error = GrpcError.fromGrpcStatusRuntimeException(ex);
        return ResponseEntity.status(error.getStatus()).contentType(MediaType.APPLICATION_JSON).body(error);
    }

    private final ResponseProcessor responseProcessor;
    private final CpaExportImpl cpaExportImpl;

    @RequestMapping(value = "/v2/get_cpa_order_snapshots", method = RequestMethod.GET, produces = "application/json")
    public DeferredResult<CpaOrderSnapshotRspV2> getCpaOrderSnapshots(@Validated @BindFromQuery CpaOrderSnapshotReqV2 req) {
        return responseProcessor.replyWithFutureRetrying(
                "CpaExportGetCpaOrderSnapshotsV2", () -> cpaExportImpl.getCpaOrderSnapshotsV2(req),
                RetryStrategyExceptionHelpers.defaultStatusUnavailableRetryStrategy()
        );
    }

    @RequestMapping(value = "/v1/get_boy_orders", method = RequestMethod.GET, produces = "application/json")
    public DeferredResult<CpaBoyOrdersRsp> getBoyOrders(@Validated @BindFromQuery CpaBoyOrdersReq req) {
        return responseProcessor.replyWithFutureRetrying(
                "CpaExportGetCpaBoyOrders", () -> cpaExportImpl.getCpaBoyOrders(req),
                RetryStrategyExceptionHelpers.defaultStatusUnavailableRetryStrategy()
        );
    }
}
