package ru.yandex.travel.api.endpoints.cpa;

import java.math.BigDecimal;
import java.util.Arrays;
import java.util.Map;
import java.util.concurrent.CompletableFuture;
import java.util.stream.Collectors;

import com.google.common.base.Strings;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.javamoney.moneta.Money;
import org.springframework.stereotype.Service;

import ru.yandex.travel.api.endpoints.cpa.req_rsp.CpaBoyOrdersReq;
import ru.yandex.travel.api.endpoints.cpa.req_rsp.CpaBoyOrdersRsp;
import ru.yandex.travel.api.endpoints.cpa.req_rsp.CpaOrderSnapshotReqV2;
import ru.yandex.travel.api.endpoints.cpa.req_rsp.CpaOrderSnapshotRspV2;
import ru.yandex.travel.api.models.cpa.AviaFlightDto;
import ru.yandex.travel.api.models.cpa.AviaOrderSnapshotDto;
import ru.yandex.travel.api.models.cpa.BnovoOrderSnapshotDto;
import ru.yandex.travel.api.models.cpa.BoyOrderSnapshotDto;
import ru.yandex.travel.api.models.cpa.BronevikOrderSnapshotDto;
import ru.yandex.travel.api.models.cpa.BusOrderSnapshotDto;
import ru.yandex.travel.api.models.cpa.CpaItemState;
import ru.yandex.travel.api.models.cpa.CpaOrderDisplayType;
import ru.yandex.travel.api.models.cpa.CpaOrderStatus;
import ru.yandex.travel.api.models.cpa.CpaRefundReason;
import ru.yandex.travel.api.models.cpa.DeferredPaymentEligibility;
import ru.yandex.travel.api.models.cpa.DolphinOrderSnapshotDto;
import ru.yandex.travel.api.models.cpa.ExpediaOrderSnapshotDto;
import ru.yandex.travel.api.models.cpa.HotelOrderSnapshotDto;
import ru.yandex.travel.api.models.cpa.OrderSnapshotDto;
import ru.yandex.travel.api.models.cpa.SuburbanOrderSnapshotDto;
import ru.yandex.travel.api.models.cpa.TrainOrderSnapshotDto;
import ru.yandex.travel.api.models.cpa.TravellineOrderSnapshotDto;
import ru.yandex.travel.api.services.orders.OrchestratorClientFactory;
import ru.yandex.travel.api.services.orders.TrainOrderStatusMappingService;
import ru.yandex.travel.bus.model.BusReservation;
import ru.yandex.travel.bus.model.BusRide;
import ru.yandex.travel.bus.model.BusTicketStatus;
import ru.yandex.travel.bus.model.BusTicketType;
import ru.yandex.travel.bus.model.BusesTicketRefund;
import ru.yandex.travel.commons.concurrent.FutureUtils;
import ru.yandex.travel.commons.proto.ProtoCurrencyUnit;
import ru.yandex.travel.commons.proto.ProtoUtils;
import ru.yandex.travel.orders.commons.proto.EOrderType;
import ru.yandex.travel.orders.commons.proto.EServiceType;
import ru.yandex.travel.orders.cpa.ECpaItemState;
import ru.yandex.travel.orders.cpa.TAviaExtraData;
import ru.yandex.travel.orders.cpa.TBoyOrderSnapshot;
import ru.yandex.travel.orders.cpa.TBoyOrdersReq;
import ru.yandex.travel.orders.cpa.TBoyOrdersRsp;
import ru.yandex.travel.orders.cpa.TBusExtraData;
import ru.yandex.travel.orders.cpa.TListSnapshotsReqV2;
import ru.yandex.travel.orders.cpa.TListSnapshotsRspV2;
import ru.yandex.travel.orders.cpa.TOrderSnapshot;
import ru.yandex.travel.orders.cpa.TSuburbanExtraData;
import ru.yandex.travel.orders.cpa.TTrainExtraData;
import ru.yandex.travel.orders.proto.EOrderRefundState;
import ru.yandex.travel.orders.proto.EOrderRefundType;
import ru.yandex.travel.orders.proto.TUserInfo;
import ru.yandex.travel.train.model.CarStorey;
import ru.yandex.travel.train.model.CarType;
import ru.yandex.travel.train.model.InsuranceStatus;
import ru.yandex.travel.train.model.PassengerCategory;
import ru.yandex.travel.train.model.TrainModelHelpers;
import ru.yandex.travel.train.model.TrainReservation;
import ru.yandex.travel.train.model.TrainTicketRefundStatus;
import ru.yandex.travel.train.model.refund.TicketRefund;

import static java.util.stream.Collectors.toList;

@Service
@Slf4j
@RequiredArgsConstructor
public class CpaExportImpl {
    private static final BigDecimal TRAIN_INSURANCE_FEE_COEFFICIENT = BigDecimal.valueOf(0.65); // TRAINS-5082
    private static final Map<CarType, String> TRAIN_CPA_COACH_TYPES = Map.of(
            CarType.RESERVED_SEAT, "platzkarte",
            CarType.COMPARTMENT, "compartment",
            CarType.LUXURY, "suite",
            CarType.SHARED, "common",
            CarType.SEDENTARY, "sitting",
            CarType.SOFT, "soft"
    );
    private final TrainOrderStatusMappingService trainOrderStatusMappingService;
    private final OrchestratorClientFactory orchestratorClientFactory;

    public CompletableFuture<CpaBoyOrdersRsp> getCpaBoyOrders(CpaBoyOrdersReq request) {
        TBoyOrdersReq.Builder reqBuilder = TBoyOrdersReq.newBuilder();
        reqBuilder.setUpdatedAtFrom(ProtoUtils.fromLocalDateTime(request.getUpdatedAtFromUtc()));
        if (request.getUpdatedAtToUtc() != null) {
            reqBuilder.setUpdatedAtTo(ProtoUtils.fromLocalDateTime(request.getUpdatedAtToUtc()));
        }
        reqBuilder.setMaxSnapshots(request.getLimit());
        return FutureUtils.buildCompletableFuture(
                        orchestratorClientFactory.createCpaOrderSnapshotFutureStub().getBoyOrders(reqBuilder.build()))
                .thenApply(this::mapTBoyOrdersRsp);
    }

    public CompletableFuture<CpaOrderSnapshotRspV2> getCpaOrderSnapshotsV2(CpaOrderSnapshotReqV2 request) {
        TListSnapshotsReqV2.Builder reqBuilder = TListSnapshotsReqV2.newBuilder();
        reqBuilder.setUpdatedAtFrom(ProtoUtils.fromLocalDateTime(request.getUpdatedAtFromUtc()));
        if (request.getUpdatedAtToUtc() != null) {
            reqBuilder.setUpdatedAtTo(ProtoUtils.fromLocalDateTime(request.getUpdatedAtToUtc()));
        }
        reqBuilder.setMaxSnapshots(request.getLimit());
        reqBuilder.setOrderType(request.getOrderType().getOrderType());
        reqBuilder.setServiceType(request.getOrderType().getServiceType());
        reqBuilder.setAddPersonalData(request.isAddPersonalData());
        return FutureUtils.buildCompletableFuture(
                        orchestratorClientFactory.createCpaOrderSnapshotFutureStub().getSnapshotsV2(reqBuilder.build()))
                .thenApply(r -> mapTListSnapshotRspV2(r,
                        request.getOrderType().getOrderType(),
                        request.getOrderType().getServiceType(),
                        request.isAddPersonalData()));
    }

    private CpaBoyOrdersRsp mapTBoyOrdersRsp(TBoyOrdersRsp resp) {
        CpaBoyOrdersRsp result = new CpaBoyOrdersRsp();
        result.setOrderSnapshots(resp.getOrderSnapshotsList().stream()
                .map(this::mapBoyOrder)
                .collect(toList()));
        result.setHasMore(resp.getHasMore());
        return result;
    }

    private CpaOrderSnapshotRspV2 mapTListSnapshotRspV2(TListSnapshotsRspV2 resp,
                                                        EOrderType orderType,
                                                        EServiceType serviceType,
                                                        boolean addPersonalData) {
        CpaOrderSnapshotRspV2 result = new CpaOrderSnapshotRspV2();
        result.setOrderSnapshots(resp.getOrderSnapshotsList().stream()
                .map(snapshot -> mapOrderSnapshotWrapper(snapshot, orderType, serviceType, addPersonalData))
                .collect(toList()));
        result.setHasMore(resp.getHasMore());
        return result;
    }

    private BoyOrderSnapshotDto mapBoyOrder(TBoyOrderSnapshot orderSnapshot) {
        BoyOrderSnapshotDto result = new BoyOrderSnapshotDto();
        result.setCreatedAt(ProtoUtils.toInstant(orderSnapshot.getCreatedAt()));
        result.setUpdatedAt(ProtoUtils.toInstant(orderSnapshot.getUpdatedAt()));
        result.setAmount(ProtoUtils.fromTPrice(orderSnapshot.getAmount()));
        result.setLabel(orderSnapshot.getLabel());
        result.setPartnerOrderId(orderSnapshot.getPartnerOrderId());
        result.setTravelOrderId(orderSnapshot.getTravelOrderId());
        result.setOrderStatus(CpaOrderStatus.fromProto(orderSnapshot.getCpaOrderStatus()));
        result.setDisplayType(CpaOrderDisplayType.BY_PROTO.getByValueOrNull(orderSnapshot.getDisplayType()));
        result.setPromoActions(orderSnapshot.getPromoActionList());
        result.setPromoCodes(orderSnapshot.getPromoCodeList());
        if (orderSnapshot.hasDiscountAmount()) {
            result.setDiscountAmount(ProtoUtils.fromTPrice(orderSnapshot.getDiscountAmount()));
        }
        if (orderSnapshot.hasProfit()) {
            result.setProfit(ProtoUtils.fromTPrice(orderSnapshot.getProfit()));
        }
        result.setUsesDeferredPayment(orderSnapshot.getUsesDeferredPayment());
        if (orderSnapshot.hasAmountReceiveFromUser()) {
            result.setAmountReceivedFromUser(ProtoUtils.fromTPrice(orderSnapshot.getAmountReceiveFromUser()));
        }
        result.setPostPayEligible(orderSnapshot.getPostPayEligible());
        result.setPostPayUsed(orderSnapshot.getPostPayUsed());
        return result;
    }

    private OrderSnapshotDto mapOrderSnapshotWrapper(TOrderSnapshot orderSnapshot, EOrderType orderType,
                                                     EServiceType serviceType, boolean addPersonalData) {
        try {
            return mapOrderSnapshot(orderSnapshot, orderType, serviceType, addPersonalData);
        } catch (Exception e) {
            throw new RuntimeException("Failed to convert the order for CPA export; " +
                    "id " + orderSnapshot.getTravelOrderId(), e);
        }
    }

    private OrderSnapshotDto mapOrderSnapshot(TOrderSnapshot orderSnapshot, EOrderType orderType,
                                              EServiceType serviceType, boolean addPersonalData) {
        OrderSnapshotDto result;
        switch (orderType) {
            case OT_HOTEL_EXPEDIA:
                HotelOrderSnapshotDto hotelOrderSnapshot;
                switch (serviceType) {
                    case PT_EXPEDIA_HOTEL:
                        ExpediaOrderSnapshotDto expediaOrder = new ExpediaOrderSnapshotDto();
                        expediaOrder.setConfirmationId(orderSnapshot.getHotelExtraData().getExpediaExtraData().getConfirmationId());
                        expediaOrder.setItineraryId(orderSnapshot.getHotelExtraData().getExpediaExtraData().getItineraryId());
                        hotelOrderSnapshot = expediaOrder;
                        break;
                    case PT_DOLPHIN_HOTEL:
                        DolphinOrderSnapshotDto dolphinOrder = new DolphinOrderSnapshotDto();
                        dolphinOrder.setCode(orderSnapshot.getHotelExtraData().getDolphinExtraData().getCode());
                        hotelOrderSnapshot = dolphinOrder;
                        break;
                    case PT_TRAVELLINE_HOTEL:
                        TravellineOrderSnapshotDto travellineOrder = new TravellineOrderSnapshotDto();
                        travellineOrder.setConfirmationId(orderSnapshot.getHotelExtraData().getTravellineExtraData().getConfirmationId());
                        hotelOrderSnapshot = travellineOrder;
                        break;
                    case PT_BNOVO_HOTEL:
                        BnovoOrderSnapshotDto bnovoOrderSnapshotDto = new BnovoOrderSnapshotDto();
                        bnovoOrderSnapshotDto.setConfirmationId(orderSnapshot.getHotelExtraData().getTravellineExtraData().getConfirmationId());
                        hotelOrderSnapshot = bnovoOrderSnapshotDto;
                        break;
                    case PT_BRONEVIK_HOTEL:
                        hotelOrderSnapshot = new BronevikOrderSnapshotDto();
                        break;
                    default:
                        throw new IllegalArgumentException("Unsupported service type: " + serviceType);
                }
                hotelOrderSnapshot.setCheckInDate(ProtoUtils.toLocalDate(orderSnapshot.getHotelExtraData().getCheckInDate()));
                hotelOrderSnapshot.setCheckOutDate(ProtoUtils.toLocalDate(orderSnapshot.getHotelExtraData().getCheckOutDate()));
                hotelOrderSnapshot.setHotelName(orderSnapshot.getHotelExtraData().getHotelName());
                hotelOrderSnapshot.setHotelCountryName(orderSnapshot.getHotelExtraData().getHotelCountryName());
                hotelOrderSnapshot.setHotelCityName(orderSnapshot.getHotelExtraData().getHotelCityName());
                hotelOrderSnapshot.setPermalink(Long.toUnsignedString(orderSnapshot.getHotelExtraData().getPermalink()));
                hotelOrderSnapshot.setMir2020Eligible(orderSnapshot.getHotelExtraData().getMir2020Eligible());
                hotelOrderSnapshot.setRefundReason(CpaRefundReason.fromProto(orderSnapshot.getRefundReason()));
                if (orderSnapshot.hasAmountPayable()) {
                    hotelOrderSnapshot.setAmountPayable(ProtoUtils.fromTPrice(orderSnapshot.getAmountPayable()));
                }
                if (orderSnapshot.hasYandexPlusCpaInfo()) {
                    hotelOrderSnapshot.setYandexPlusCpaInfo(orderSnapshot.getYandexPlusCpaInfo());
                }
                if (orderSnapshot.hasWhiteLabelCpaInfo()) {
                    hotelOrderSnapshot.setWhiteLabelCpaInfo(orderSnapshot.getWhiteLabelCpaInfo());
                }
                result = hotelOrderSnapshot;
                break;
            case OT_AVIA_AEROFLOT:
                AviaOrderSnapshotDto aviaOrder = new AviaOrderSnapshotDto();
                TAviaExtraData aviaExtraData = orderSnapshot.getAviaExtraData();
                aviaOrder.setFlights(aviaExtraData.getFlightsList().stream()
                        .map(f -> AviaFlightDto.builder()
                                .from(f.getFrom())
                                .to(f.getTo())
                                .departureAt(ProtoUtils.toLocalDateTime(f.getDepartureAt()))
                                .arrivalAt(ProtoUtils.toLocalDateTime(f.getArrivalAt()))
                                .build())
                        .collect(toList()));
                aviaOrder.setAdults(aviaExtraData.getAdults());
                aviaOrder.setChildren(aviaExtraData.getChildren());
                aviaOrder.setInfants(aviaExtraData.getInfants());
                aviaOrder.setPnr(aviaExtraData.getPnr());
                result = aviaOrder;
                break;
            case OT_TRAIN:
                result = toTrainSnapshot(orderSnapshot.getTrainExtraData(), orderSnapshot.getItemState());
                break;
            case OT_GENERIC:
                if (serviceType == EServiceType.PT_SUBURBAN) {
                    result = toSuburbanSnapshot(orderSnapshot.getSuburbanExtraData());
                    break;
                }
                if (serviceType == EServiceType.PT_BUS) {
                    result = toBusSnapshot(orderSnapshot.getBusExtraData());
                    break;
                }
            default:
                throw new IllegalArgumentException(
                        String.format("Unsupported order type %s with service type %s", orderType, serviceType)
                );
        }
        result.setCreatedAt(ProtoUtils.toInstant(orderSnapshot.getCreatedAt()));
        result.setUpdatedAt(ProtoUtils.toInstant(orderSnapshot.getUpdatedAt()));
        result.setAmount(ProtoUtils.fromTPrice(orderSnapshot.getAmount()));
        result.setLabel(orderSnapshot.getLabel());
        result.setPartnerOrderId(orderSnapshot.getPartnerOrderId());
        result.setBoyOrderId(orderSnapshot.getBoyOrderId());
        result.setTravelOrderId(orderSnapshot.getTravelOrderId());
        result.setOrderStatus(CpaOrderStatus.fromProto(orderSnapshot.getCpaOrderStatus()));
        result.setItemState(CpaItemState.BY_PROTO.getByValueOrNull(orderSnapshot.getItemState()));
        result.setPromoActions(orderSnapshot.getPromoActionList());
        result.setPromoCodes(orderSnapshot.getPromoCodeList());
        if (orderSnapshot.hasDiscountAmount()) {
            result.setDiscountAmount(ProtoUtils.fromTPrice(orderSnapshot.getDiscountAmount()));
        }
        if (orderSnapshot.hasPromocodeDiscountAmount()) {
            result.setPromocodeDiscountAmount(ProtoUtils.fromTPrice(orderSnapshot.getPromocodeDiscountAmount()));
        }
        if (orderSnapshot.hasStrikeThroughDiscountAmount()) {
            result.setStrikeThroughDiscountAmount(ProtoUtils.fromTPrice(orderSnapshot.getStrikeThroughDiscountAmount()));
        }
        if (orderSnapshot.hasProfit()) {
            result.setProfit(ProtoUtils.fromTPrice(orderSnapshot.getProfit()));
        }
        result.setUsesDeferredPayment(orderSnapshot.getUsesDeferredPayment());
        if (orderSnapshot.hasAmountReceiveFromUser()) {
            result.setAmountReceivedFromUser(ProtoUtils.fromTPrice(orderSnapshot.getAmountReceiveFromUser()));
        }
        if (!orderSnapshot.hasEligibleForDeferredPayment()) {
            result.setDeferredPaymentEligibility(DeferredPaymentEligibility.UNKNOWN);
        } else {
            if (orderSnapshot.getEligibleForDeferredPayment().getValue()) {
                result.setDeferredPaymentEligibility(DeferredPaymentEligibility.ELIGIBLE);
            } else {
                result.setDeferredPaymentEligibility(DeferredPaymentEligibility.NON_ELIGIBLE);
            }
        }
        if (orderSnapshot.hasInitialPaymentAmount()) {
            result.setInitialPaymentAmount(ProtoUtils.fromTPrice(orderSnapshot.getInitialPaymentAmount()));
        }
        if (orderSnapshot.hasLastPaymentScheduledAt()) {
            result.setLastPaymentScheduledAt(ProtoUtils.toInstant(orderSnapshot.getLastPaymentScheduledAt()));
        }
        if (orderSnapshot.hasPenaltyAmountIfUnpaid()) {
            result.setPenaltyAmountIfUnpaid(ProtoUtils.fromTPrice(orderSnapshot.getPenaltyAmountIfUnpaid()));
        }
        if (orderSnapshot.hasFullyPaidAt()) {
            result.setFullyPaidAt(ProtoUtils.toInstant(orderSnapshot.getFullyPaidAt()));
        }
        if (orderSnapshot.hasPersonalUserData() && addPersonalData) {
            result.setPersonalUserData(new OrderSnapshotDto.PersonalUserData(
                    orderSnapshot.getPersonalUserData().getEmail(),
                    orderSnapshot.getPersonalUserData().getPhone(),
                    orderSnapshot.getPersonalUserData().getOwnerLogin(),
                    orderSnapshot.getPersonalUserData().getOwnerPassportId()
            ));
        }
        result.setPostPayEligible(orderSnapshot.getPostPayEligible());
        result.setPostPayUsed(orderSnapshot.getPostPayUsed());
        return result;
    }

    private static String getFirstNotEmpty(String... variants) {
        return Arrays.stream(variants).filter(s -> !Strings.isNullOrEmpty(s)).findFirst().orElse(null);
    }

    private BusOrderSnapshotDto toBusSnapshot(TBusExtraData extra) {
        var snapshot = new BusOrderSnapshotDto();
        BusReservation payload = ProtoUtils.fromTJson(extra.getPayload(), BusReservation.class);
        BusRide ride = payload.getRide();
        snapshot.setProvider(ride.getBusPartner().getValue());
        snapshot.setArrival(ride.getArrivalTime());
        snapshot.setDeparture(ride.getDepartureTime());
        snapshot.setRouteName(ride.getRouteName());
        snapshot.setBusInternalStatus(extra.getOrderItemState().name());
        snapshot.setBusModel(ride.getBus());
        snapshot.setCarrierId(ride.getCarrierCode());
        snapshot.setFromId(getFirstNotEmpty(
                ride.getPointFrom().getPointKey(),
                ride.getTitlePointFrom().getPointKey()
        ));
        snapshot.setFromPartnerDescription(getFirstNotEmpty(
                ride.getPointFrom().getSupplierDescription(),
                ride.getTitlePointFrom().getSupplierDescription()
        ));
        snapshot.setSearchFromId(ride.getTitlePointFrom().getPointKey());
        snapshot.setToId(getFirstNotEmpty(
                ride.getPointTo().getPointKey(),
                ride.getTitlePointTo().getPointKey()
        ));
        snapshot.setToPartnerDescription(getFirstNotEmpty(
                ride.getPointTo().getSupplierDescription(),
                ride.getTitlePointTo().getSupplierDescription()
        ));
        snapshot.setSearchToId(ride.getTitlePointTo().getPointKey());
        snapshot.setOnlineRefund(ride.isOnlineRefund());
        if (extra.hasConfirmedAt()) {
            snapshot.setFinishedAt(ProtoUtils.toInstant(extra.getConfirmedAt()));
        }
        if (extra.hasPaymentTs()) {
            snapshot.setPaymentTs(ProtoUtils.toInstant(extra.getPaymentTs()));
        }

        int totalTicketCount = 0;
        int adultPassengersCount = 0;
        int childrenWithSeatsCount = 0;
        int activeTicketCount = 0;
        int baggageTicketCount = 0;
        int refundedTicketCount = 0;
        var zero = Money.zero(ProtoCurrencyUnit.RUB);
        Money totalTariffAmount = zero;
        Money totalFeeAmount = zero;
        Money totalPartnerFeeAmount = zero;
        Money totalRefundTicketAmount = zero;
        Money totalPartnerRefundFeeAmount = zero;
        Money totalRefundFeeAmount = zero;
        Money totalAgencyFeeAmount = zero;

        if (payload.getOrder() != null) {
            for (var ticket : payload.getOrder().getTickets()) {
                totalTicketCount++;
                totalTariffAmount = totalTariffAmount.add(ticket.getTicketPrice());
                totalFeeAmount = totalFeeAmount.add(ticket.getYandexFee());
                totalPartnerFeeAmount = totalPartnerFeeAmount.add(ticket.getPartnerFee());
                if (ticket.getRevenue() != null) {
                    totalAgencyFeeAmount = totalAgencyFeeAmount.add(ticket.getRevenue());
                }
                if (ticket.getStatus() == BusTicketStatus.SOLD) {
                    activeTicketCount++;
                } else if (ticket.getStatus() == BusTicketStatus.RETURNED) {
                    refundedTicketCount++;
                }
                if (ticket.getPassenger().getTicketType() == BusTicketType.FULL) {
                    adultPassengersCount++;
                } else if (ticket.getPassenger().getTicketType() == BusTicketType.CHILD) {
                    childrenWithSeatsCount++;
                } else if (ticket.getPassenger().getTicketType() == BusTicketType.BAGGAGE) {
                    baggageTicketCount++;
                }
            }
        }
        for (var r : extra.getRefundList()) {
            if (r.getState() == EOrderRefundState.RS_REFUNDED && r.hasPayload()) {
                BusesTicketRefund refundPayload = ProtoUtils.fromTJson(r.getPayload(), BusesTicketRefund.class);
                totalRefundTicketAmount = totalRefundTicketAmount.add(refundPayload.getRefundAmount());
            }
        }
        snapshot.setTotalTicketCount(totalTicketCount);
        snapshot.setAdultPassengersCount(adultPassengersCount);
        snapshot.setChildrenWithSeatsCount(childrenWithSeatsCount);
        snapshot.setActiveTicketCount(activeTicketCount);
        snapshot.setBaggageTicketsCount(baggageTicketCount);
        snapshot.setRequestedTicketCount(payload.getRequestPassengers().size());
        snapshot.setRefundedTicketCount(refundedTicketCount);
        snapshot.setTotalTariffAmount(totalTariffAmount);
        snapshot.setTotalFeeAmount(totalFeeAmount);
        snapshot.setTotalPartnerFeeAmount(totalPartnerFeeAmount);
        snapshot.setTotalRefundTicketAmount(totalRefundTicketAmount);
        snapshot.setTotalPartnerRefundFeeAmount(totalPartnerRefundFeeAmount);
        snapshot.setTotalRefundFeeAmount(totalRefundFeeAmount);
        snapshot.setTotalAgencyFeeAmount(totalAgencyFeeAmount);
        return snapshot;
    }

    private TrainOrderSnapshotDto toTrainSnapshot(TTrainExtraData trainExtra, ECpaItemState itemState) {
        var trainOrder = new TrainOrderSnapshotDto();
        var payload = ProtoUtils.fromTJson(trainExtra.getPayload(), TrainReservation.class);
        trainOrder.setOrderNumber(payload.getReservationNumber());
        trainOrder.setOfferId(payload.getOfferId() != null ? payload.getOfferId().toString() : null);
        trainOrder.setPermille(payload.getPermille());
        trainOrder.setBanditType(payload.getBanditType());
        trainOrder.setRequestedBanditType(payload.getReservationRequestData().getBanditType());
        trainOrder.setBanditVersion(payload.getBanditVersion());
        trainOrder.setTrainName(payload.getReservationRequestData().getBrandTitle());
        trainOrder.setTrainNumber(payload.getReservationRequestData().getTrainNumber());
        trainOrder.setTrainTicketNumber(payload.getReservationRequestData().getTrainTicketNumber());
        //TODO (@mbobrov, @tlg-13) probablly set it optional on cpa platform level
        // const for now to resume cpa function
        trainOrder.setTrainInternalStatus("UNKNOWN");
        if (payload.getDepartureTime() != null) {
            trainOrder.setDeparture(payload.getDepartureTime());
        } else {
            trainOrder.setDeparture(payload.getReservationRequestData().getDepartureTime());
        }
        trainOrder.setArrival(payload.getArrivalTime());
        trainOrder.setServiceClass(payload.getReservationRequestData().getServiceClass());
        trainOrder.setSchemeId(payload.getReservationRequestData().getSchemeId());
        if (payload.getCarType() != null) {
            trainOrder.setCoachType(TRAIN_CPA_COACH_TYPES.get(payload.getCarType()));
        } else if (payload.getReservationRequestData().getCarType() != null) {
            trainOrder.setCoachType(TRAIN_CPA_COACH_TYPES.get(payload.getReservationRequestData().getCarType()));
        }
        if (payload.getCarNumber() != null) {
            trainOrder.setCoachNumber(payload.getCarNumber());
        } else {
            trainOrder.setCoachNumber(payload.getReservationRequestData().getCarNumber());
        }
        trainOrder.setCoachOwner(payload.getCarrier());
        trainOrder.setGender(payload.getReservationRequestData().getCabinGenderKind().getValue());
        trainOrder.setPartnerDataCompartmentGender(payload.getReservationRequestData().getCabinGenderKind().getValue());
        trainOrder.setTwoStorey(payload.getReservationRequestData().getCarStorey() == CarStorey.SECOND);
        if (trainExtra.hasConfirmedAt()) {
            trainOrder.setFinishedAt(ProtoUtils.toInstant(trainExtra.getConfirmedAt()));
        }
        trainOrder.setInsuranceAutoReturn(payload.getInsuranceStatus() == InsuranceStatus.AUTO_RETURN);
        TUserInfo owner = trainExtra.getOwner();
        trainOrder.setUserIp(owner.getIp());
        trainOrder.setUserYandexUid(owner.getYandexUid());
        trainOrder.setUserPassportUid(owner.getPassportId());
        trainOrder.setUserRegionId(owner.getGeoId());
        trainOrder.setUserIsMobile(payload.isMobile());
        trainOrder.setPaymentAttempts(trainExtra.getInvoiceCount());
        if (trainExtra.hasPaymentTs()) {
            trainOrder.setPaymentTs(ProtoUtils.toInstant(trainExtra.getPaymentTs()));
        }
        if (trainExtra.hasCurrentInvoice()) {
            var invoice = trainExtra.getCurrentInvoice();
            trainOrder.setPaymentUid(invoice.getInvoiceId());
            trainOrder.setPaymentStatus(invoice.getTrustInvoiceState().name());
            trainOrder.setPaymentPurchaseToken(invoice.getPurchaseToken());
            if (invoice.hasCreatedAt()) {
                trainOrder.setPaymentTrustCreatedAt(ProtoUtils.toInstant(invoice.getCreatedAt()));
            }
            trainOrder.setPaymentUseDeferredClearing(true);
        }
        trainOrder.setRebooked(payload.getIsRebookingFor() != null);
        trainOrder.setRebookingAvailable(trainExtra.getRebookingEnabled());
        trainOrder.setStationFromId(payload.getReservationRequestData().getStationFromId());
        trainOrder.setStationToId(payload.getReservationRequestData().getStationToId());
        trainOrder.setRouteInfoFromStationTitle(payload.getUiData().getStationFromTitle());
        trainOrder.setRouteInfoToStationTitle(payload.getUiData().getStationToTitle());
        if (trainExtra.hasLabelParams()) {
            trainOrder.setLabelParams(ProtoUtils.fromTJson(trainExtra.getLabelParams()));
        }
        trainOrder.setPartnerDataImOrderId(payload.getPartnerOrderId());
        trainOrder.setPartnerDataExpireSetEr(TrainModelHelpers.calculateCanChangeElectronicRegistrationTill(payload));
        trainOrder.setPartnerDataIsOnlyFullReturnPossible(payload.isOnlyFullReturnPossible());
        trainOrder.setPartnerDataIsSuburban(payload.isSuburban());
        trainOrder.setPartnerDataOperationId(payload.getPartnerBuyOperationIds().stream()
                .map(x -> Integer.toString(x)).collect(Collectors.joining(",")));
        trainOrder.setPartnerDataProvider(payload.getProvider());

        var zero = Money.zero(ProtoCurrencyUnit.RUB);
        Money profit = zero;
        Money totalFeeAmount = zero;
        Money totalInsuranceAmount = zero;
        Money totalInsuranceProfitAmount = zero;
        Money totalPartnerFeeAmount = zero;
        Money totalPartnerRefundFeeAmount = zero;
        Money totalRefundFeeAmount = zero;
        Money totalRefundInsuranceAmount = zero;
        Money totalRefundTicketAmount = zero;
        Money totalServiceAmount = zero;
        Money totalTariffAmount = zero;
        int totalTicketCount = 0;
        int passengersCount = 0;
        int adultPassengersCount = 0;
        int activeTicketCount = 0;
        int boughtInsuranceCount = 0;
        int childrenWithSeatsCount = 0;
        int childrenWithoutSeatsCount = 0;
        int refundedTicketCount = 0;
        int refundsCount = 0;
        int requestedTicketCount = 0;
        int ticketsWithPlacesCount = 0;
        int ticketsWithoutPlacesCount = 0;
        int banditFeeAppliedCount = 0;
        int nonRefundableTicketsCount = 0;
        for (var p : payload.getPassengers()) {
            requestedTicketCount++;
            passengersCount++;
            var t = p.getTicket();
            if (t != null && t.isBanditFeeApplied()) {
                banditFeeAppliedCount++;
            }
            if (p.getCategory() == PassengerCategory.ADULT) {
                adultPassengersCount++;
            } else if (p.getCategory() == PassengerCategory.CHILD) {
                childrenWithSeatsCount++;
            } else if (p.getCategory() == PassengerCategory.BABY) {
                childrenWithoutSeatsCount++;
            }
            if (p.isNonRefundableTariff()) {
                nonRefundableTicketsCount++;
            }
            // most aggregations needs confirmed orders
            if (!(itemState == ECpaItemState.IS_CONFIRMED ||
                    itemState == ECpaItemState.IS_REFUNDED)) {
                break;
            }
            totalTicketCount++;
            if (t.getPlaces() == null || t.getPlaces().size() == 0) {
                ticketsWithoutPlacesCount++;
            } else {
                ticketsWithPlacesCount++;
            }
            totalFeeAmount = totalFeeAmount.add(t.getFeeAmount());
            totalServiceAmount = totalServiceAmount.add(t.getServiceAmount());
            totalTariffAmount = totalTariffAmount.add(t.getTariffAmount());
            totalPartnerFeeAmount = totalPartnerFeeAmount.add(t.getPartnerFee());
            if (t.getRefundStatus() == TrainTicketRefundStatus.REFUNDED) {
                refundedTicketCount++;
                totalPartnerRefundFeeAmount = totalPartnerRefundFeeAmount.add(t.getPartnerRefundFee());
            } else {
                activeTicketCount++;
            }
            profit = profit.add(t.getFeeAmount());
            if (payload.getInsuranceStatus() == InsuranceStatus.CHECKED_OUT) {
                boughtInsuranceCount++;
                totalInsuranceAmount = totalInsuranceAmount.add(p.getInsurance().getAmount());
                var insuranceProfit = p.getInsurance().getAmount().multiply(TRAIN_INSURANCE_FEE_COEFFICIENT);
                totalInsuranceProfitAmount = totalInsuranceProfitAmount.add(insuranceProfit);
                profit = profit.add(insuranceProfit);
            } else if (payload.getInsuranceStatus() == InsuranceStatus.AUTO_RETURN) {
                totalInsuranceAmount = totalInsuranceAmount.add(p.getInsurance().getAmount());
                totalRefundInsuranceAmount = totalRefundInsuranceAmount.add(p.getInsurance().getAmount());
            }
        }
        for (var r : trainExtra.getRefundList()) {
            if (r.getState() == EOrderRefundState.RS_REFUNDED && r.hasPayload() &&
                    (r.getRefundType() == EOrderRefundType.RT_TRAIN_USER_REFUND ||
                            r.getRefundType() == EOrderRefundType.RT_TRAIN_OFFICE_REFUND)) {
                refundsCount++;
                TicketRefund refundPayload = ProtoUtils.fromTJson(r.getPayload(), TicketRefund.class);
                for (var ri : refundPayload.getItems()) {
                    totalRefundTicketAmount = totalRefundTicketAmount.add(ri.getActualRefundTicketAmount());
                    totalRefundInsuranceAmount =
                            totalRefundInsuranceAmount.add(ri.getCalculatedRefundInsuranceAmount());
                    totalRefundFeeAmount = totalRefundFeeAmount.add(ri.getCalculatedRefundFeeAmount());
                    var refundedInsuranceProfit =
                            ri.getCalculatedRefundInsuranceAmount().multiply(TRAIN_INSURANCE_FEE_COEFFICIENT);
                    totalInsuranceProfitAmount = totalInsuranceProfitAmount.subtract(refundedInsuranceProfit);
                }
            }
        }
        trainOrder.setProfit(profit);
        trainOrder.setTotalFeeAmount(totalFeeAmount);
        trainOrder.setTotalInsuranceAmount(totalInsuranceAmount);
        trainOrder.setTotalInsuranceProfitAmount(totalInsuranceProfitAmount);
        trainOrder.setTotalPartnerFeeAmount(totalPartnerFeeAmount);
        trainOrder.setTotalPartnerRefundFeeAmount(totalPartnerRefundFeeAmount);
        trainOrder.setTotalRefundFeeAmount(totalRefundFeeAmount);
        trainOrder.setTotalRefundInsuranceAmount(totalRefundInsuranceAmount);
        trainOrder.setTotalRefundTicketAmount(totalRefundTicketAmount);
        trainOrder.setTotalServiceAmount(totalServiceAmount);
        trainOrder.setTotalTariffAmount(totalTariffAmount);
        trainOrder.setTotalTicketCount(totalTicketCount);
        trainOrder.setPassengersCount(passengersCount);
        trainOrder.setAdultPassengersCount(adultPassengersCount);
        trainOrder.setActiveTicketCount(activeTicketCount);
        trainOrder.setBoughtInsuranceCount(boughtInsuranceCount);
        trainOrder.setChildrenWithSeatsCount(childrenWithSeatsCount);
        trainOrder.setChildrenWithoutSeatsCount(childrenWithoutSeatsCount);
        trainOrder.setRefundedTicketCount(refundedTicketCount);
        trainOrder.setRefundsCount(refundsCount);
        trainOrder.setRequestedTicketCount(requestedTicketCount);
        trainOrder.setTicketsWithPlacesCount(ticketsWithPlacesCount);
        trainOrder.setTicketsWithoutPlacesCount(ticketsWithoutPlacesCount);
        trainOrder.setBanditFeeAppliedCount(banditFeeAppliedCount);
        trainOrder.setNonRefundableTicketsCount(nonRefundableTicketsCount);
        return trainOrder;
    }

    private SuburbanOrderSnapshotDto toSuburbanSnapshot(TSuburbanExtraData suburbanExtra) {
        var suburbanSnapshot = new SuburbanOrderSnapshotDto();
        suburbanSnapshot.setOrderItemState(suburbanExtra.getOrderItemState());
        suburbanSnapshot.setProvider(suburbanExtra.getProvider());
        suburbanSnapshot.setCarrierPartner(suburbanExtra.getCarrierPartner());
        suburbanSnapshot.setProviderOrderId(suburbanExtra.getProviderOrderId());
        suburbanSnapshot.setStationFromId(suburbanExtra.getStationFromId());
        suburbanSnapshot.setStationToId(suburbanExtra.getStationToId());
        suburbanSnapshot.setStationFromTitle(suburbanExtra.getStationFromTitle());
        suburbanSnapshot.setStationToTitle(suburbanExtra.getStationToTitle());
        suburbanSnapshot.setDepartureDate(ProtoUtils.toInstant(suburbanExtra.getDepartureDate()));
        return suburbanSnapshot;
    }
}
