package ru.yandex.travel.api.endpoints.generic_booking_flow;

import java.util.UUID;
import java.util.concurrent.CompletableFuture;

import javax.servlet.http.HttpServletRequest;

import com.google.common.base.Preconditions;
import io.grpc.StatusRuntimeException;
import io.swagger.annotations.ApiOperation;
import lombok.RequiredArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.bind.annotation.ExceptionHandler;
import org.springframework.web.bind.annotation.GetMapping;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.RequestMethod;
import org.springframework.web.bind.annotation.ResponseStatus;
import org.springframework.web.bind.annotation.RestController;
import org.springframework.web.context.request.async.DeferredResult;

import ru.yandex.travel.api.endpoints.booking_flow.req_rsp.EstimateDiscountRspV1;
import ru.yandex.travel.api.endpoints.generic_booking_flow.req_rsp.CalculateRefundAmountReqV1;
import ru.yandex.travel.api.endpoints.generic_booking_flow.req_rsp.CalculateRefundAmountRspV1;
import ru.yandex.travel.api.endpoints.generic_booking_flow.req_rsp.CancelGenericOrderReqV1;
import ru.yandex.travel.api.endpoints.generic_booking_flow.req_rsp.CreateGenericOrderReqV1;
import ru.yandex.travel.api.endpoints.generic_booking_flow.req_rsp.DownloadBlankReqV1;
import ru.yandex.travel.api.endpoints.generic_booking_flow.req_rsp.GenericOrderAddServiceReqV1;
import ru.yandex.travel.api.endpoints.generic_booking_flow.req_rsp.GenericOrderAddServiceRspV1;
import ru.yandex.travel.api.endpoints.generic_booking_flow.req_rsp.GetGenericOrderReqV1;
import ru.yandex.travel.api.endpoints.generic_booking_flow.req_rsp.GetGenericOrderRspV1;
import ru.yandex.travel.api.endpoints.generic_booking_flow.req_rsp.GetGenericOrderStateBatchReqV1;
import ru.yandex.travel.api.endpoints.generic_booking_flow.req_rsp.GetGenericOrderStateBatchRspV1;
import ru.yandex.travel.api.endpoints.generic_booking_flow.req_rsp.GetGenericOrderStateReqV1;
import ru.yandex.travel.api.endpoints.generic_booking_flow.req_rsp.GetGenericOrderStateRspV1;
import ru.yandex.travel.api.endpoints.generic_booking_flow.req_rsp.StartGenericOrderPaymentReqV1;
import ru.yandex.travel.api.endpoints.generic_booking_flow.req_rsp.StartRefundReqV1;
import ru.yandex.travel.api.exceptions.GrpcError;
import ru.yandex.travel.api.infrastucture.BindFromQuery;
import ru.yandex.travel.api.infrastucture.ResponseProcessor;
import ru.yandex.travel.api.services.common.RetryStrategyExceptionHelpers;
import ru.yandex.travel.api.services.orders.GenericOrdersService;
import ru.yandex.travel.api.services.trips.TripsProvider;
import ru.yandex.travel.commons.experiments.ExperimentDataProvider;
import ru.yandex.travel.commons.experiments.OrderExperiments;
import ru.yandex.travel.commons.http.CommonHttpHeaders;
import ru.yandex.travel.credentials.UserCredentials;
import ru.yandex.travel.komod.trips.common.TripsUtils;

@RestController
@RequestMapping(value = "/api/generic_booking_flow/")
@RequiredArgsConstructor
@Slf4j
public class GenericBookingFlowController {
    private final ResponseProcessor responseProcessor;
    private final GenericOrdersService ordersService;
    private final ExperimentDataProvider experimentDataProvider;
    private final TripsProvider tripsProvider;

    @ExceptionHandler(IllegalArgumentException.class)
    @ResponseStatus(HttpStatus.BAD_REQUEST)
    public ResponseEntity<String> handleIllegalArgumentException(IllegalArgumentException e) {
        return ResponseEntity.badRequest().contentType(MediaType.TEXT_PLAIN).body(e.getMessage());
    }

    @ExceptionHandler(StatusRuntimeException.class)
    public ResponseEntity<GrpcError> handleGrpcErrors(StatusRuntimeException ex) {
        GrpcError error = GrpcError.fromGrpcStatusRuntimeException(ex);
        return ResponseEntity.status(error.getStatus()).contentType(MediaType.APPLICATION_JSON).body(error);
    }

    @PostMapping("/v1/estimate_discount")
    public DeferredResult<EstimateDiscountRspV1> estimateDiscount(@RequestBody CreateGenericOrderReqV1 request) {
        throw new UnsupportedOperationException("not implemented yet");
    }

    @PostMapping("/v1/add_service")
    public DeferredResult<GenericOrderAddServiceRspV1> addService(@RequestBody GenericOrderAddServiceReqV1 request) {
        return responseProcessor.replyWithFutureRetrying("GenericBookingFlowAddService",
                () -> ordersService.addService(request),
                RetryStrategyExceptionHelpers.defaultStatusUnavailableRetryStrategy()
        );
    }

    @PostMapping("/v1/create_order")
    public DeferredResult<GetGenericOrderRspV1> createOrder(@RequestBody CreateGenericOrderReqV1 request) {
        UUID deduplicationId = request.getDeduplicationKey();
        return responseProcessor.replyWithFutureRetrying("GenericBookingFlowCreateOrder",
                () -> ordersService.createOrder(request, currentUser(), CommonHttpHeaders.get().getExperiments()),
                RetryStrategyExceptionHelpers.defaultStatusUnavailableRetryStrategy(),
                null, deduplicationId != null ? deduplicationId.toString() : null
        );
    }

    @GetMapping("/v1/get_order")
    public DeferredResult<GetGenericOrderRspV1> getOrder(GetGenericOrderReqV1 request, HttpServletRequest httpServletRequest) {
        var userCredentials = UserCredentials.get();
        return responseProcessor.replyWithFutureRetrying("GenericBookingFlowGetOrderInfo",
                () -> {
                    var getOrderByIdFuture = ordersService.getOrderInfo(
                            request, httpServletRequest);
                    var getTripIdByOrderIdFuture = tripsProvider.getTripIdByOrderId(
                            request.getOrderId(), userCredentials);
                    return CompletableFuture.allOf(getOrderByIdFuture, getTripIdByOrderIdFuture)
                            .thenApply(ignored -> {
                                var order = getOrderByIdFuture.join();
                                TripsUtils.fillTripId(order, getTripIdByOrderIdFuture.join());
                                return order;
                            });
                },
                RetryStrategyExceptionHelpers.defaultStatusUnavailableRetryStrategy()
        );
    }

    @GetMapping("/v1/get_order_state")
    public DeferredResult<GetGenericOrderStateRspV1> getOrderState(GetGenericOrderStateReqV1 request) {
        return responseProcessor.replyWithFutureRetrying("GenericBookingFlowGetOrderState",
                () -> ordersService.getOrderState(request),
                RetryStrategyExceptionHelpers.defaultStatusUnavailableRetryStrategy()
        );
    }

    @PostMapping("/v1/get_order_state_batch")
    public DeferredResult<GetGenericOrderStateBatchRspV1> getOrderStateBatch(@RequestBody GetGenericOrderStateBatchReqV1 request) {
        return responseProcessor.replyWithFutureRetrying("GenericBookingFlowGetOrderStateBatch",
                () -> ordersService.getOrderStateBatch(request),
                RetryStrategyExceptionHelpers.defaultStatusUnavailableRetryStrategy()
        );
    }

    @PostMapping("/v1/start_payment")
    public DeferredResult<Void> startPayment(@RequestBody StartGenericOrderPaymentReqV1 request) {
        return responseProcessor.replyWithFutureRetrying(
                "GenericBookingFlowStartPayment",
                () -> ordersService.payOrder(
                        request,
                        experimentDataProvider.getInstance(OrderExperiments.class, CommonHttpHeaders.get())
                ),
                RetryStrategyExceptionHelpers.defaultStatusUnavailableRetryStrategy()
        );
    }

    @PostMapping("/v1/cancel_order")
    public DeferredResult<Void> cancelOrder(@RequestBody CancelGenericOrderReqV1 request) {
        return responseProcessor.replyWithFutureRetrying("GenericBookingFlowCancelOrder",
                () -> ordersService.cancelOrder(request),
                RetryStrategyExceptionHelpers.defaultStatusUnavailableRetryStrategy()
        );
    }

    @PostMapping("/v1/calculate_refund_amount")
    public DeferredResult<CalculateRefundAmountRspV1> calculateRefundAmount(@RequestBody CalculateRefundAmountReqV1 request) {
        return responseProcessor.replyWithFutureRetrying("GenericBookingFlowCalculateRefundAmount",
                () -> ordersService.calculateRefundAmount(request),
                RetryStrategyExceptionHelpers.defaultStatusUnavailableRetryStrategy()
        );
    }

    @PostMapping("/v1/start_refund")
    public DeferredResult<Void> startRefund(@RequestBody StartRefundReqV1 request) {
        return responseProcessor.replyWithFutureRetrying("GenericBookingFlowStartRefund",
                () -> ordersService.startRefund(request),
                RetryStrategyExceptionHelpers.defaultStatusUnavailableRetryStrategy()
        );
    }

    @RequestMapping(value = "/v1/download_blank", method = RequestMethod.GET)
    @ApiOperation("Получение pdf билета")
    public DeferredResult<ResponseEntity<byte[]>> downloadBlank(@BindFromQuery DownloadBlankReqV1 request) {
        return responseProcessor.replyWithFutureRetrying("GenericBookingFlowDownloadBlank",
                () -> ordersService.downloadBlank(request),
                RetryStrategyExceptionHelpers.defaultStatusUnavailableRetryStrategy()
        );
    }

    private UserCredentials currentUser() {
        return Preconditions.checkNotNull(UserCredentials.get(), "UserCredentials are not set");
    }
}
