package ru.yandex.travel.api.endpoints.hotels_extranet.auth

import io.grpc.StatusRuntimeException
import io.swagger.annotations.ApiOperation
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.http.MediaType
import org.springframework.http.ResponseEntity
import org.springframework.web.bind.annotation.ExceptionHandler
import org.springframework.web.bind.annotation.GetMapping
import org.springframework.web.bind.annotation.PostMapping
import org.springframework.web.bind.annotation.RequestBody
import org.springframework.web.bind.annotation.RequestMapping
import org.springframework.web.bind.annotation.RestController
import org.springframework.web.context.request.async.DeferredResult
import ru.yandex.travel.api.endpoints.hotels_extranet.auth.req_rsp.ApproveReqV1
import ru.yandex.travel.api.endpoints.hotels_extranet.auth.req_rsp.ApproveRspV1
import ru.yandex.travel.api.endpoints.hotels_extranet.auth.req_rsp.ListApprovedRspV1
import ru.yandex.travel.api.endpoints.hotels_extranet.auth.req_rsp.RequestReqV1
import ru.yandex.travel.api.endpoints.hotels_extranet.auth.req_rsp.RequestRspV1
import ru.yandex.travel.api.exceptions.GrpcError
import ru.yandex.travel.api.infrastucture.ResponseProcessor
import ru.yandex.travel.api.services.common.RetryStrategyExceptionHelpers
import ru.yandex.travel.api.services.extranet.ExtranetClientFactory
import ru.yandex.travel.api.services.hotels.slug.HotelSlugService
import ru.yandex.travel.commons.concurrent.FutureUtils
import ru.yandex.travel.hotels.extranet.TListApprovedReq
import javax.validation.Valid

@RestController
@RequestMapping(value = ["/api/hotels_extranet/auth"])
class HotelExtranetAuthController @Autowired constructor(
    private val responseProcessor: ResponseProcessor,
    private val clientFactory: ExtranetClientFactory,
    private val slugService: HotelSlugService,
) {
    @ExceptionHandler(StatusRuntimeException::class)
    fun handleGrpcErrors(ex: StatusRuntimeException?): ResponseEntity<GrpcError>? {
        val error = GrpcError.fromGrpcStatusRuntimeException(ex)
        return ResponseEntity.status(error.status).contentType(MediaType.APPLICATION_JSON).body(error)
    }

    @PostMapping(value = ["/v1/request"], produces = [MediaType.APPLICATION_JSON_VALUE])
    @ApiOperation(value = "Request authorization to the hotel's cabinet", response = RequestRspV1::class)
    fun request(@RequestBody request: @Valid RequestReqV1): DeferredResult<RequestRspV1> {
        return responseProcessor.replyWithFutureRetrying(
            "ExtranetRequestHotelAuth",
            {
                FutureUtils.buildCompletableFuture(
                    clientFactory.createAuthServiceFutureStub().requestAuth(request.toProto())
                ).thenApply(RequestRspV1::fromProto)
            },
            RetryStrategyExceptionHelpers.defaultStatusUnavailableRetryStrategy()
        )
    }

    @PostMapping(value = ["/v1/approve"], produces = [MediaType.APPLICATION_JSON_VALUE])
    @ApiOperation(value = "Approve authorization request with token", response = RequestRspV1::class)
    fun approve(@RequestBody request: @Valid ApproveReqV1): DeferredResult<ApproveRspV1> {
        return responseProcessor.replyWithFutureRetrying(
            "ExtranetApproveHotelAuth",
            {
                FutureUtils.buildCompletableFuture(
                    clientFactory.createAuthServiceFutureStub().approveAuth(request.toProto())
                ).thenApply(ApproveRspV1::fromProto)
            },
            RetryStrategyExceptionHelpers.defaultStatusUnavailableRetryStrategy()
        )
    }

    @GetMapping(value = ["/v1/list_approved"], produces = [MediaType.APPLICATION_JSON_VALUE])
    @ApiOperation(value = "List authorized hotels", response = ListApprovedRspV1::class)
    fun listApproved(): DeferredResult<ListApprovedRspV1> {
        return responseProcessor.replyWithFutureRetrying(
            "ExtranetListApprovedHotels",
            {
                FutureUtils.buildCompletableFuture(
                    clientFactory.createAuthServiceFutureStub().listApproved(TListApprovedReq.getDefaultInstance())
                ).thenApply { it ->
                    ListApprovedRspV1.fromProto(it, slugService)
                }
            },
            RetryStrategyExceptionHelpers.defaultStatusUnavailableRetryStrategy()
        )
    }
}
